<?php
/**
 * Sistema Médico - Registrar Nuevo Paciente
 */
require_once '../includes/config.php';
require_once '../includes/database.php';
require_once '../includes/auth.php';

$page_title = 'Nuevo Paciente';

$database = new Database();
$db = $database->getConnection();

$mensaje = '';
$tipo_mensaje = '';

// Procesar formulario
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $nombre = trim($_POST['nombre'] ?? '');
    $apellido = trim($_POST['apellido'] ?? '');
    $cedula = trim($_POST['cedula'] ?? '');
    $fecha_nacimiento = $_POST['fecha_nacimiento'] ?? '';
    $telefono = trim($_POST['telefono'] ?? '');
    $domicilio = trim($_POST['domicilio'] ?? '');
    
    // Alertas médicas
    $alergias = trim($_POST['alergias'] ?? '');
    $resistencias = trim($_POST['resistencias'] ?? '');
    $medicamentos_permanentes = trim($_POST['medicamentos_permanentes'] ?? '');
    $otras_dolencias = trim($_POST['otras_dolencias'] ?? '');

    try {
        $db->beginTransaction();

        // Verificar si la cédula ya existe
        $query = "SELECT id FROM pacientes WHERE cedula = :cedula";
        $stmt = $db->prepare($query);
        $stmt->bindParam(':cedula', $cedula);
        $stmt->execute();

        if ($stmt->rowCount() > 0) {
            throw new Exception('Ya existe un paciente registrado con esta cédula');
        }

        // Insertar paciente
        $query = "INSERT INTO pacientes (nombre, apellido, cedula, fecha_nacimiento, telefono, domicilio, tiene_alertas) 
                  VALUES (:nombre, :apellido, :cedula, :fecha_nacimiento, :telefono, :domicilio, :tiene_alertas)";
        
        $tiene_alertas = (!empty($alergias) || !empty($resistencias) || !empty($medicamentos_permanentes) || !empty($otras_dolencias)) ? 1 : 0;
        
        $stmt = $db->prepare($query);
        $stmt->bindParam(':nombre', $nombre);
        $stmt->bindParam(':apellido', $apellido);
        $stmt->bindParam(':cedula', $cedula);
        $stmt->bindParam(':fecha_nacimiento', $fecha_nacimiento);
        $stmt->bindParam(':telefono', $telefono);
        $stmt->bindParam(':domicilio', $domicilio);
        $stmt->bindParam(':tiene_alertas', $tiene_alertas);
        $stmt->execute();

        $paciente_id = $db->lastInsertId();

        // Insertar alertas médicas
        if (!empty($alergias)) {
            $query = "INSERT INTO alertas_medicas (paciente_id, tipo, descripcion, severidad) 
                      VALUES (:paciente_id, 'alergia', :descripcion, 'alta')";
            $stmt = $db->prepare($query);
            $stmt->bindParam(':paciente_id', $paciente_id);
            $stmt->bindParam(':descripcion', $alergias);
            $stmt->execute();
        }

        if (!empty($resistencias)) {
            $query = "INSERT INTO alertas_medicas (paciente_id, tipo, descripcion, severidad) 
                      VALUES (:paciente_id, 'resistencia_medicamento', :descripcion, 'alta')";
            $stmt = $db->prepare($query);
            $stmt->bindParam(':paciente_id', $paciente_id);
            $stmt->bindParam(':descripcion', $resistencias);
            $stmt->execute();
        }

        if (!empty($medicamentos_permanentes)) {
            $query = "INSERT INTO alertas_medicas (paciente_id, tipo, descripcion, severidad) 
                      VALUES (:paciente_id, 'medicamento_permanente', :descripcion, 'media')";
            $stmt = $db->prepare($query);
            $stmt->bindParam(':paciente_id', $paciente_id);
            $stmt->bindParam(':descripcion', $medicamentos_permanentes);
            $stmt->execute();
        }

        if (!empty($otras_dolencias)) {
            $query = "INSERT INTO alertas_medicas (paciente_id, tipo, descripcion, severidad) 
                      VALUES (:paciente_id, 'otra_dolencia', :descripcion, 'media')";
            $stmt = $db->prepare($query);
            $stmt->bindParam(':paciente_id', $paciente_id);
            $stmt->bindParam(':descripcion', $otras_dolencias);
            $stmt->execute();
        }

        $db->commit();

        $mensaje = 'Paciente registrado exitosamente';
        $tipo_mensaje = 'success';

        // Redirigir a la ficha del paciente después de 2 segundos
        header("refresh:2;url=ficha_paciente.php?id=$paciente_id");

    } catch (Exception $e) {
        $db->rollBack();
        $mensaje = 'Error al registrar paciente: ' . $e->getMessage();
        $tipo_mensaje = 'error';
    }
}

include '../includes/header.php';
?>

<div class="card">
    <div class="card-header">
        <h2>Registrar Nuevo Paciente</h2>
    </div>
    <div class="card-body">
        <?php if ($mensaje): ?>
            <div class="alert alert-<?php echo $tipo_mensaje; ?>">
                <?php echo htmlspecialchars($mensaje); ?>
            </div>
        <?php endif; ?>

        <form method="POST" action="" id="formNuevoPaciente">
            <h3 style="color: var(--medical-blue); margin-bottom: 20px; border-bottom: 2px solid var(--medical-light-blue); padding-bottom: 10px;">
                Datos Personales
            </h3>

            <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(300px, 1fr)); gap: 20px;">
                <div class="form-group">
                    <label for="nombre">Nombre *</label>
                    <input type="text" id="nombre" name="nombre" required>
                </div>

                <div class="form-group">
                    <label for="apellido">Apellido *</label>
                    <input type="text" id="apellido" name="apellido" required>
                </div>

                <div class="form-group">
                    <label for="cedula">Cédula *</label>
                    <input type="text" id="cedula" name="cedula" required>
                </div>

                <div class="form-group">
                    <label for="fecha_nacimiento">Fecha de Nacimiento *</label>
                    <input type="date" id="fecha_nacimiento" name="fecha_nacimiento" required>
                </div>

                <div class="form-group">
                    <label for="telefono">Teléfono</label>
                    <input type="tel" id="telefono" name="telefono">
                </div>
            </div>

            <div class="form-group">
                <label for="domicilio">Domicilio</label>
                <textarea id="domicilio" name="domicilio" rows="3"></textarea>
            </div>

            <h3 style="color: var(--medical-blue); margin: 30px 0 20px; border-bottom: 2px solid var(--medical-light-blue); padding-bottom: 10px;">
                Alertas Médicas
            </h3>

            <div class="alert alert-warning">
                <strong>Importante:</strong> Si el paciente tiene alergias, resistencias a medicamentos u otras condiciones médicas importantes, regístrelas aquí. Esta información aparecerá resaltada en la ficha del paciente.
            </div>

            <div class="form-group">
                <label for="alergias">Alergias</label>
                <textarea id="alergias" name="alergias" rows="3" placeholder="Ej: Penicilina, Polen, Mariscos..."></textarea>
            </div>

            <div class="form-group">
                <label for="resistencias">Resistencias a Medicamentos</label>
                <textarea id="resistencias" name="resistencias" rows="3" placeholder="Ej: Resistencia a antibióticos específicos..."></textarea>
            </div>

            <div class="form-group">
                <label for="medicamentos_permanentes">Medicamentos de por Vida</label>
                <textarea id="medicamentos_permanentes" name="medicamentos_permanentes" rows="3" placeholder="Ej: Insulina, Anticoagulantes, Antihipertensivos..."></textarea>
            </div>

            <div class="form-group">
                <label for="otras_dolencias">Otras Dolencias o Condiciones Crónicas</label>
                <textarea id="otras_dolencias" name="otras_dolencias" rows="3" placeholder="Ej: Diabetes, Hipertensión, Asma..."></textarea>
            </div>

            <div style="display: flex; gap: 15px; margin-top: 30px;">
                <button type="submit" class="btn btn-success">💾 Registrar Paciente</button>
                <a href="buscar_paciente.php" class="btn btn-secondary">Cancelar</a>
            </div>
        </form>
    </div>
</div>

<?php include '../includes/footer.php'; ?>

