<?php
/**
 * Sistema Médico - Exportar Reportes a Excel/PDF
 */
require_once '../../includes/config.php';
require_once '../../includes/database.php';
require_once '../../includes/auth.php';
require_once '../../includes/permissions.php';

Permissions::requirePermission('ExportData');

$database = new Database();
$db = $database->getConnection();

$fecha_desde = $_GET['fecha_desde'] ?? date('Y-m-01');
$fecha_hasta = $_GET['fecha_hasta'] ?? date('Y-m-d');
$formato = $_GET['formato'] ?? 'excel';

// Obtener datos para el reporte
$query = "SELECT 
          COUNT(*) as total_facturas,
          SUM(CASE WHEN estado = 'pagada' THEN 1 ELSE 0 END) as facturas_pagadas,
          SUM(CASE WHEN estado = 'pendiente' THEN 1 ELSE 0 END) as facturas_pendientes,
          SUM(CASE WHEN estado = 'anulada' THEN 1 ELSE 0 END) as facturas_anuladas,
          SUM(CASE WHEN estado = 'pagada' THEN total ELSE 0 END) as total_recaudado,
          SUM(CASE WHEN estado = 'pendiente' THEN total ELSE 0 END) as total_pendiente
          FROM facturas
          WHERE DATE(fecha_emision) BETWEEN :fecha_desde AND :fecha_hasta";

$stmt = $db->prepare($query);
$stmt->bindParam(':fecha_desde', $fecha_desde);
$stmt->bindParam(':fecha_hasta', $fecha_hasta);
$stmt->execute();
$resumen = $stmt->fetch();

// Recaudación diaria
$query = "SELECT 
          DATE(fecha_emision) as fecha,
          COUNT(*) as total_facturas,
          SUM(CASE WHEN estado = 'pagada' THEN total ELSE 0 END) as total_recaudado
          FROM facturas
          WHERE DATE(fecha_emision) BETWEEN :fecha_desde AND :fecha_hasta
          GROUP BY DATE(fecha_emision)
          ORDER BY fecha DESC";

$stmt = $db->prepare($query);
$stmt->bindParam(':fecha_desde', $fecha_desde);
$stmt->bindParam(':fecha_hasta', $fecha_hasta);
$stmt->execute();
$recaudacion_diaria = $stmt->fetchAll();

// Productos más vendidos
$query = "SELECT 
          p.codigo,
          p.nombre,
          SUM(fp.cantidad) as cantidad_vendida,
          SUM(fp.subtotal) as total_ingresos
          FROM factura_productos fp
          INNER JOIN productos p ON fp.producto_id = p.id
          INNER JOIN facturas f ON fp.factura_id = f.id
          WHERE f.estado = 'pagada' AND DATE(f.fecha_emision) BETWEEN :fecha_desde AND :fecha_hasta
          GROUP BY fp.producto_id, p.codigo, p.nombre
          ORDER BY cantidad_vendida DESC
          LIMIT 20";

$stmt = $db->prepare($query);
$stmt->bindParam(':fecha_desde', $fecha_desde);
$stmt->bindParam(':fecha_hasta', $fecha_hasta);
$stmt->execute();
$productos_mas_vendidos = $stmt->fetchAll();

// Servicios más solicitados
$query = "SELECT 
          s.codigo,
          s.nombre,
          SUM(fs.cantidad) as cantidad_solicitada,
          SUM(fs.subtotal) as total_ingresos
          FROM factura_servicios fs
          INNER JOIN servicios s ON fs.servicio_id = s.id
          INNER JOIN facturas f ON fs.factura_id = f.id
          WHERE f.estado = 'pagada' AND DATE(f.fecha_emision) BETWEEN :fecha_desde AND :fecha_hasta
          GROUP BY fs.servicio_id, s.codigo, s.nombre
          ORDER BY cantidad_solicitada DESC
          LIMIT 20";

$stmt = $db->prepare($query);
$stmt->bindParam(':fecha_desde', $fecha_desde);
$stmt->bindParam(':fecha_hasta', $fecha_hasta);
$stmt->execute();
$servicios_mas_solicitados = $stmt->fetchAll();

if ($formato == 'excel') {
    // Exportar a CSV (compatible con Excel)
    header('Content-Type: text/csv; charset=utf-8');
    header('Content-Disposition: attachment; filename="reporte_recaudacion_' . $fecha_desde . '_' . $fecha_hasta . '.csv"');
    
    $output = fopen('php://output', 'w');
    
    // BOM para UTF-8
    fprintf($output, chr(0xEF).chr(0xBB).chr(0xBF));
    
    // Encabezado del reporte
    fputcsv($output, ['REPORTE DE RECAUDACIÓN']);
    fputcsv($output, ['Centro Médico: ' . SITE_NAME]);
    fputcsv($output, ['Período: ' . date('d/m/Y', strtotime($fecha_desde)) . ' - ' . date('d/m/Y', strtotime($fecha_hasta))]);
    fputcsv($output, ['Fecha de generación: ' . date('d/m/Y H:i')]);
    fputcsv($output, []);
    
    // Resumen general
    fputcsv($output, ['RESUMEN GENERAL']);
    fputcsv($output, ['Concepto', 'Valor']);
    fputcsv($output, ['Total Facturas', $resumen['total_facturas']]);
    fputcsv($output, ['Facturas Pagadas', $resumen['facturas_pagadas']]);
    fputcsv($output, ['Facturas Pendientes', $resumen['facturas_pendientes']]);
    fputcsv($output, ['Facturas Anuladas', $resumen['facturas_anuladas']]);
    fputcsv($output, ['Total Recaudado (₲)', number_format($resumen['total_recaudado'], 0, ',', '.')]);
    fputcsv($output, ['Total Pendiente (₲)', number_format($resumen['total_pendiente'], 0, ',', '.')]);
    fputcsv($output, []);
    
    // Recaudación diaria
    fputcsv($output, ['RECAUDACIÓN DIARIA']);
    fputcsv($output, ['Fecha', 'Facturas', 'Total Recaudado (₲)']);
    foreach ($recaudacion_diaria as $dia) {
        fputcsv($output, [
            date('d/m/Y', strtotime($dia['fecha'])),
            $dia['total_facturas'],
            number_format($dia['total_recaudado'], 0, ',', '.')
        ]);
    }
    fputcsv($output, []);
    
    // Productos más vendidos
    fputcsv($output, ['PRODUCTOS MÁS VENDIDOS']);
    fputcsv($output, ['Código', 'Producto', 'Cantidad Vendida', 'Total Ingresos (₲)']);
    foreach ($productos_mas_vendidos as $producto) {
        fputcsv($output, [
            $producto['codigo'],
            $producto['nombre'],
            $producto['cantidad_vendida'],
            number_format($producto['total_ingresos'], 0, ',', '.')
        ]);
    }
    fputcsv($output, []);
    
    // Servicios más solicitados
    fputcsv($output, ['SERVICIOS MÁS SOLICITADOS']);
    fputcsv($output, ['Código', 'Servicio', 'Cantidad Solicitada', 'Total Ingresos (₲)']);
    foreach ($servicios_mas_solicitados as $servicio) {
        fputcsv($output, [
            $servicio['codigo'],
            $servicio['nombre'],
            $servicio['cantidad_solicitada'],
            number_format($servicio['total_ingresos'], 0, ',', '.')
        ]);
    }
    
    fclose($output);
    exit();
    
} elseif ($formato == 'pdf') {
    // Exportar a PDF usando HTML
    ?>
    <!DOCTYPE html>
    <html>
    <head>
        <meta charset="UTF-8">
        <title>Reporte de Recaudación</title>
        <style>
            body {
                font-family: Arial, sans-serif;
                font-size: 12px;
                margin: 20px;
            }
            h1 {
                color: #2c5aa0;
                font-size: 20px;
                border-bottom: 3px solid #2c5aa0;
                padding-bottom: 10px;
            }
            h2 {
                color: #2c5aa0;
                font-size: 16px;
                margin-top: 30px;
                border-bottom: 2px solid #4caf50;
                padding-bottom: 5px;
            }
            .info {
                background: #f0f0f0;
                padding: 15px;
                margin: 20px 0;
                border-radius: 5px;
            }
            table {
                width: 100%;
                border-collapse: collapse;
                margin: 15px 0;
            }
            th {
                background: #2c5aa0;
                color: white;
                padding: 10px;
                text-align: left;
            }
            td {
                padding: 8px;
                border-bottom: 1px solid #ddd;
            }
            tr:nth-child(even) {
                background: #f9f9f9;
            }
            .resumen {
                display: grid;
                grid-template-columns: repeat(3, 1fr);
                gap: 15px;
                margin: 20px 0;
            }
            .resumen-item {
                background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
                color: white;
                padding: 20px;
                border-radius: 8px;
                text-align: center;
            }
            .resumen-item.green {
                background: linear-gradient(135deg, #43e97b 0%, #38f9d7 100%);
            }
            .resumen-item.orange {
                background: linear-gradient(135deg, #fa709a 0%, #fee140 100%);
            }
            .resumen-label {
                font-size: 11px;
                opacity: 0.9;
                margin-bottom: 8px;
            }
            .resumen-valor {
                font-size: 24px;
                font-weight: bold;
            }
            .footer {
                margin-top: 50px;
                text-align: center;
                font-size: 10px;
                color: #666;
            }
            @media print {
                body { margin: 0; }
                .no-print { display: none; }
            }
        </style>
    </head>
    <body>
        <button onclick="window.print()" class="no-print" style="padding: 10px 20px; background: #2c5aa0; color: white; border: none; border-radius: 5px; cursor: pointer; margin-bottom: 20px;">
            🖨️ Imprimir / Guardar como PDF
        </button>
        
        <h1>REPORTE DE RECAUDACIÓN</h1>
        
        <div class="info">
            <strong>Centro Médico:</strong> <?php echo SITE_NAME; ?><br>
            <strong>RUC:</strong> <?php echo SITE_RUC; ?><br>
            <strong>Período:</strong> <?php echo date('d/m/Y', strtotime($fecha_desde)); ?> - <?php echo date('d/m/Y', strtotime($fecha_hasta)); ?><br>
            <strong>Fecha de generación:</strong> <?php echo date('d/m/Y H:i'); ?>
        </div>
        
        <h2>Resumen General</h2>
        <div class="resumen">
            <div class="resumen-item">
                <div class="resumen-label">Total Facturas</div>
                <div class="resumen-valor"><?php echo number_format($resumen['total_facturas'], 0, ',', '.'); ?></div>
            </div>
            <div class="resumen-item green">
                <div class="resumen-label">Total Recaudado</div>
                <div class="resumen-valor">₲ <?php echo number_format($resumen['total_recaudado'], 0, ',', '.'); ?></div>
            </div>
            <div class="resumen-item orange">
                <div class="resumen-label">Total Pendiente</div>
                <div class="resumen-valor">₲ <?php echo number_format($resumen['total_pendiente'], 0, ',', '.'); ?></div>
            </div>
        </div>
        
        <table>
            <tr>
                <td><strong>Facturas Pagadas:</strong></td>
                <td><?php echo $resumen['facturas_pagadas']; ?></td>
            </tr>
            <tr>
                <td><strong>Facturas Pendientes:</strong></td>
                <td><?php echo $resumen['facturas_pendientes']; ?></td>
            </tr>
            <tr>
                <td><strong>Facturas Anuladas:</strong></td>
                <td><?php echo $resumen['facturas_anuladas']; ?></td>
            </tr>
        </table>
        
        <h2>Recaudación Diaria</h2>
        <table>
            <thead>
                <tr>
                    <th>Fecha</th>
                    <th>Facturas</th>
                    <th>Total Recaudado</th>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($recaudacion_diaria as $dia): ?>
                    <tr>
                        <td><?php echo date('d/m/Y', strtotime($dia['fecha'])); ?></td>
                        <td><?php echo $dia['total_facturas']; ?></td>
                        <td><strong>₲ <?php echo number_format($dia['total_recaudado'], 0, ',', '.'); ?></strong></td>
                    </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
        
        <h2>Productos Más Vendidos</h2>
        <table>
            <thead>
                <tr>
                    <th>Código</th>
                    <th>Producto</th>
                    <th>Cantidad</th>
                    <th>Total Ingresos</th>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($productos_mas_vendidos as $producto): ?>
                    <tr>
                        <td><?php echo htmlspecialchars($producto['codigo']); ?></td>
                        <td><?php echo htmlspecialchars($producto['nombre']); ?></td>
                        <td><?php echo $producto['cantidad_vendida']; ?></td>
                        <td><strong>₲ <?php echo number_format($producto['total_ingresos'], 0, ',', '.'); ?></strong></td>
                    </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
        
        <h2>Servicios Más Solicitados</h2>
        <table>
            <thead>
                <tr>
                    <th>Código</th>
                    <th>Servicio</th>
                    <th>Cantidad</th>
                    <th>Total Ingresos</th>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($servicios_mas_solicitados as $servicio): ?>
                    <tr>
                        <td><?php echo htmlspecialchars($servicio['codigo']); ?></td>
                        <td><?php echo htmlspecialchars($servicio['nombre']); ?></td>
                        <td><?php echo $servicio['cantidad_solicitada']; ?></td>
                        <td><strong>₲ <?php echo number_format($servicio['total_ingresos'], 0, ',', '.'); ?></strong></td>
                    </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
        
        <div class="footer">
            <p>Este reporte fue generado automáticamente por el Sistema Médico - <?php echo SITE_NAME; ?></p>
            <p>Fecha y hora de generación: <?php echo date('d/m/Y H:i:s'); ?></p>
        </div>
    </body>
    </html>
    <?php
    exit();
}
?>
