<?php
/**
 * Sistema Médico - Gestión de Permisos y Roles
 */

class Permissions {
    
    /**
     * Verificar si el usuario tiene un rol específico
     */
    public static function hasRole($role) {
        if (!Auth::isLoggedIn()) {
            return false;
        }
        
        $user = Auth::getCurrentUser();
        return $user['rol'] === $role;
    }
    
    /**
     * Verificar si el usuario es administrador
     */
    public static function isAdmin() {
        return self::hasRole('admin');
    }
    
    /**
     * Verificar si el usuario es médico
     */
    public static function isMedico() {
        return self::hasRole('medico');
    }
    
    /**
     * Verificar si el usuario es colaborador
     */
    public static function isColaborador() {
        return self::hasRole('colaborador');
    }
    
    /**
     * Verificar si el usuario puede gestionar productos y servicios
     * Solo administradores
     */
    public static function canManageProductsServices() {
        return self::isAdmin();
    }
    
    /**
     * Verificar si el usuario puede crear facturas
     * Colaboradores y administradores
     */
    public static function canCreateInvoices() {
        return self::isAdmin() || self::isColaborador();
    }
    
    /**
     * Verificar si el usuario puede ver reportes
     * Solo administradores
     */
    public static function canViewReports() {
        return self::isAdmin();
    }
    
    /**
     * Verificar si el usuario puede exportar datos
     * Solo administradores
     */
    public static function canExportData() {
        return self::isAdmin();
    }
    
    /**
     * Verificar si el usuario puede gestionar consultas médicas
     * Médicos y administradores
     */
    public static function canManageConsultations() {
        return self::isAdmin() || self::isMedico();
    }
    
    /**
     * Verificar si el usuario puede registrar pacientes
     * Todos los roles
     */
    public static function canRegisterPatients() {
        return Auth::isLoggedIn();
    }
    
    /**
     * Verificar si el usuario puede ver pacientes
     * Todos los roles
     */
    public static function canViewPatients() {
        return Auth::isLoggedIn();
    }
    
    /**
     * Verificar si el usuario puede gestionar citas
     * Médicos y administradores
     */
    public static function canManageAppointments() {
        return self::isAdmin() || self::isMedico();
    }
    
    /**
     * Verificar si el usuario puede anular facturas
     * Solo administradores
     */
    public static function canCancelInvoices() {
        return self::isAdmin();
    }
    
    /**
     * Redirigir si no tiene permiso
     */
    public static function requirePermission($permission) {
        $method = 'can' . ucfirst($permission);
        
        if (!method_exists('Permissions', $method) || !self::$method()) {
            header('Location: ../index.php?error=sin_permisos');
            exit();
        }
    }
    
    /**
     * Obtener nombre del rol en español
     */
    public static function getRoleName($role) {
        $roles = [
            'admin' => 'Administrador',
            'medico' => 'Médico',
            'colaborador' => 'Colaborador'
        ];
        
        return $roles[$role] ?? 'Desconocido';
    }
    
    /**
     * Obtener permisos del rol actual
     */
    public static function getCurrentPermissions() {
        if (!Auth::isLoggedIn()) {
            return [];
        }
        
        $user = Auth::getCurrentUser();
        $permissions = [];
        
        switch ($user['rol']) {
            case 'admin':
                $permissions = [
                    'manage_products_services' => true,
                    'create_invoices' => true,
                    'view_reports' => true,
                    'export_data' => true,
                    'manage_consultations' => true,
                    'register_patients' => true,
                    'view_patients' => true,
                    'manage_appointments' => true,
                    'cancel_invoices' => true
                ];
                break;
                
            case 'medico':
                $permissions = [
                    'manage_consultations' => true,
                    'register_patients' => true,
                    'view_patients' => true,
                    'manage_appointments' => true
                ];
                break;
                
            case 'colaborador':
                $permissions = [
                    'create_invoices' => true,
                    'register_patients' => true,
                    'view_patients' => true
                ];
                break;
        }
        
        return $permissions;
    }
}
?>
