<?php
/**
 * Portal del Paciente - Dashboard
 * Sistema de Administración de Veterinaria
 */

require_once '../config/config.php';
requerir_rol('paciente');

$titulo_pagina = 'Mis Mascotas';

$dueno_id = $_SESSION['usuario_id'];

try {
    $db = Database::getInstance()->getConnection();
    
    // Obtener información del dueño
    $stmt = $db->prepare("SELECT * FROM duenos WHERE id = :id");
    $stmt->execute(['id' => $dueno_id]);
    $dueno = $stmt->fetch();
    
    // Obtener mascotas del dueño
    $stmt = $db->prepare("SELECT * FROM mascotas WHERE dueno_id = :dueno_id AND activo = 1 ORDER BY nombre");
    $stmt->execute(['dueno_id' => $dueno_id]);
    $mascotas = $stmt->fetchAll();
    
    // Obtener próximas visitas
    $stmt = $db->prepare("SELECT v.*, m.nombre as mascota_nombre, u.nombre_completo as profesional
                          FROM visitas v
                          INNER JOIN mascotas m ON v.mascota_id = m.id
                          LEFT JOIN usuarios u ON v.profesional_id = u.id
                          WHERE m.dueno_id = :dueno_id 
                            AND v.proxima_cita IS NOT NULL 
                            AND v.proxima_cita >= CURDATE()
                          ORDER BY v.proxima_cita ASC
                          LIMIT 5");
    $stmt->execute(['dueno_id' => $dueno_id]);
    $proximas_visitas = $stmt->fetchAll();
    
    // Obtener promociones activas
    $stmt = $db->query("SELECT * FROM promociones WHERE activo = 1 AND fecha_fin >= CURDATE() ORDER BY fecha_inicio DESC");
    $promociones = $stmt->fetchAll();
    
    // Obtener vacunas próximas a vencer (próximos 30 días)
    $stmt = $db->prepare("SELECT v.*, m.nombre as mascota_nombre
                          FROM vacunas v
                          INNER JOIN mascotas m ON v.mascota_id = m.id
                          WHERE m.dueno_id = :dueno_id 
                            AND v.proxima_dosis IS NOT NULL
                            AND v.proxima_dosis BETWEEN CURDATE() AND DATE_ADD(CURDATE(), INTERVAL 30 DAY)
                          ORDER BY v.proxima_dosis ASC");
    $stmt->execute(['dueno_id' => $dueno_id]);
    $vacunas_proximas = $stmt->fetchAll();
    
} catch (PDOException $e) {
    die("Error en la base de datos: " . $e->getMessage());
}

include '../includes/header.php';
?>

<div class="row mb-4">
    <div class="col-12">
        <h1><i class="bi bi-house-heart"></i> Bienvenido, <?php echo htmlspecialchars($dueno['nombre_completo']); ?></h1>
        <p class="text-muted">Aquí puede ver el estado de sus mascotas y próximas citas</p>
    </div>
</div>

<!-- Alertas de vacunas próximas -->
<?php if (count($vacunas_proximas) > 0): ?>
    <div class="alert alert-warning alert-dismissible fade show">
        <h5 class="alert-heading"><i class="bi bi-exclamation-triangle"></i> Vacunas Próximas</h5>
        <p>Tiene <?php echo count($vacunas_proximas); ?> vacuna(s) programada(s) en los próximos 30 días:</p>
        <ul class="mb-0">
            <?php foreach ($vacunas_proximas as $vacuna): ?>
                <li>
                    <strong><?php echo htmlspecialchars($vacuna['mascota_nombre']); ?></strong> - 
                    <?php echo htmlspecialchars($vacuna['nombre_vacuna']); ?> - 
                    <?php echo formatear_fecha($vacuna['proxima_dosis']); ?>
                </li>
            <?php endforeach; ?>
        </ul>
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    </div>
<?php endif; ?>

<!-- Próximas Visitas -->
<?php if (count($proximas_visitas) > 0): ?>
    <div class="card mb-4">
        <div class="card-header bg-primary text-white">
            <h5 class="mb-0"><i class="bi bi-calendar-check"></i> Próximas Citas</h5>
        </div>
        <div class="card-body">
            <div class="list-group list-group-flush">
                <?php foreach ($proximas_visitas as $visita): ?>
                    <div class="list-group-item">
                        <div class="d-flex justify-content-between align-items-center">
                            <div>
                                <h6 class="mb-1">
                                    <i class="bi bi-heart text-danger"></i> 
                                    <?php echo htmlspecialchars($visita['mascota_nombre']); ?>
                                </h6>
                                <p class="mb-1">
                                    <i class="bi bi-calendar"></i> 
                                    <strong><?php echo formatear_fecha($visita['proxima_cita']); ?></strong>
                                </p>
                                <?php if ($visita['profesional']): ?>
                                    <p class="mb-0 text-muted">
                                        <i class="bi bi-person"></i> 
                                        <?php echo htmlspecialchars($visita['profesional']); ?>
                                    </p>
                                <?php endif; ?>
                            </div>
                            <span class="badge bg-info">Programada</span>
                        </div>
                    </div>
                <?php endforeach; ?>
            </div>
        </div>
    </div>
<?php endif; ?>

<!-- Mis Mascotas -->
<div class="card mb-4">
    <div class="card-header bg-success text-white">
        <h5 class="mb-0"><i class="bi bi-heart-fill"></i> Mis Mascotas</h5>
    </div>
    <div class="card-body">
        <?php if (count($mascotas) > 0): ?>
            <div class="row">
                <?php foreach ($mascotas as $mascota): ?>
                    <?php
                    $edad = '';
                    if ($mascota['fecha_nacimiento']) {
                        $fecha_nac = new DateTime($mascota['fecha_nacimiento']);
                        $hoy = new DateTime();
                        $diff = $hoy->diff($fecha_nac);
                        if ($diff->y > 0) {
                            $edad = $diff->y . ' año' . ($diff->y > 1 ? 's' : '');
                        } else {
                            $edad = $diff->m . ' mes' . ($diff->m > 1 ? 'es' : '');
                        }
                    }
                    ?>
                    <div class="col-md-6 col-lg-4 mb-3">
                        <div class="card h-100">
                            <div class="card-body text-center">
                                <?php if ($mascota['foto']): ?>
                                    <img src="<?php echo BASE_URL; ?>/uploads/mascotas/<?php echo htmlspecialchars($mascota['foto']); ?>" 
                                         class="img-fluid rounded-circle mb-3" 
                                         alt="<?php echo htmlspecialchars($mascota['nombre']); ?>"
                                         style="width: 120px; height: 120px; object-fit: cover;">
                                <?php else: ?>
                                    <i class="bi bi-heart-fill text-muted mb-3" style="font-size: 5rem;"></i>
                                <?php endif; ?>
                                
                                <h5 class="card-title"><?php echo htmlspecialchars($mascota['nombre']); ?></h5>
                                <p class="text-muted mb-2"><?php echo ucfirst($mascota['tipo_mascota']); ?></p>
                                
                                <?php if ($mascota['raza']): ?>
                                    <p class="mb-1"><small><strong>Raza:</strong> <?php echo htmlspecialchars($mascota['raza']); ?></small></p>
                                <?php endif; ?>
                                
                                <?php if ($edad): ?>
                                    <p class="mb-1"><small><strong>Edad:</strong> <?php echo $edad; ?></small></p>
                                <?php endif; ?>
                                
                                <?php if ($mascota['peso']): ?>
                                    <p class="mb-3"><small><strong>Peso:</strong> <?php echo number_format($mascota['peso'], 1); ?> kg</small></p>
                                <?php endif; ?>
                                
                                <a href="ver_mascota.php?id=<?php echo $mascota['id']; ?>" class="btn btn-primary btn-sm">
                                    <i class="bi bi-eye"></i> Ver Detalles
                                </a>
                            </div>
                        </div>
                    </div>
                <?php endforeach; ?>
            </div>
        <?php else: ?>
            <div class="text-center py-4">
                <i class="bi bi-heart" style="font-size: 4rem; color: #ccc;"></i>
                <p class="text-muted mt-3">No tiene mascotas registradas</p>
            </div>
        <?php endif; ?>
    </div>
</div>

<!-- Promociones Activas -->
<?php if (count($promociones) > 0): ?>
    <div class="card mb-4">
        <div class="card-header bg-warning">
            <h5 class="mb-0"><i class="bi bi-gift"></i> Promociones Disponibles</h5>
        </div>
        <div class="card-body">
            <div class="row">
                <?php foreach ($promociones as $promocion): ?>
                    <div class="col-md-6 mb-3">
                        <div class="card border-warning">
                            <div class="card-body">
                                <h6 class="card-title">
                                    <i class="bi bi-tag-fill text-warning"></i> 
                                    <?php echo htmlspecialchars($promocion['nombre']); ?>
                                </h6>
                                <p class="card-text"><?php echo htmlspecialchars($promocion['descripcion']); ?></p>
                                <div class="d-flex justify-content-between align-items-center">
                                    <span class="badge bg-success">
                                        <?php 
                                        if ($promocion['tipo_descuento'] == 'porcentaje') {
                                            echo number_format($promocion['valor_descuento'], 0) . '% OFF';
                                        } else {
                                            echo formatear_moneda($promocion['valor_descuento']) . ' OFF';
                                        }
                                        ?>
                                    </span>
                                    <small class="text-muted">
                                        Válido hasta: <?php echo formatear_fecha($promocion['fecha_fin']); ?>
                                    </small>
                                </div>
                            </div>
                        </div>
                    </div>
                <?php endforeach; ?>
            </div>
            <div class="alert alert-info mt-3 mb-0">
                <i class="bi bi-info-circle"></i> 
                <strong>Nota:</strong> Mencione estas promociones en su próxima visita para aplicar el descuento.
            </div>
        </div>
    </div>
<?php endif; ?>

<!-- Información de Contacto -->
<div class="card">
    <div class="card-header bg-info text-white">
        <h5 class="mb-0"><i class="bi bi-info-circle"></i> Información de Contacto</h5>
    </div>
    <div class="card-body">
        <p class="mb-2"><i class="bi bi-telephone"></i> <strong>Teléfono:</strong> <?php echo htmlspecialchars($dueno['telefono']); ?></p>
        <?php if ($dueno['email']): ?>
            <p class="mb-2"><i class="bi bi-envelope"></i> <strong>Email:</strong> <?php echo htmlspecialchars($dueno['email']); ?></p>
        <?php endif; ?>
        <?php if ($dueno['direccion']): ?>
            <p class="mb-0"><i class="bi bi-geo-alt"></i> <strong>Dirección:</strong> <?php echo htmlspecialchars($dueno['direccion']); ?></p>
        <?php endif; ?>
    </div>
</div>

<?php include '../includes/footer.php'; ?>
