<?php
/**
 * Página de Inicio de Sesión
 * Sistema de Administración de Veterinaria
 */

require_once 'config/config.php';

// Si ya está autenticado, redirigir
if (esta_autenticado()) {
    $rol = $_SESSION['rol'];
    if ($rol === 'administrador') {
        header('Location: admin/index.php');
    } elseif ($rol === 'colaborador') {
        header('Location: colaborador/index.php');
    } elseif ($rol === 'paciente') {
        header('Location: paciente/index.php');
    }
    exit();
}

$error = '';

// Procesar formulario de login
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $usuario = limpiar_entrada($_POST['usuario'] ?? '');
    $contrasena = $_POST['contrasena'] ?? '';
    $tipo_usuario = $_POST['tipo_usuario'] ?? 'sistema';
    
    // Para pacientes, solo se requiere cédula
    if ($tipo_usuario === 'paciente') {
        if (empty($usuario)) {
            $error = 'Por favor, ingrese su cédula';
        }
    } else {
        if (empty($usuario) || empty($contrasena)) {
            $error = 'Por favor, complete todos los campos';
        }
    }
    
    if (empty($error)) {
        try {
            $db = Database::getInstance()->getConnection();
            
            if ($tipo_usuario === 'paciente') {
                // Login de paciente (dueño de mascota) - solo con cédula
                $stmt = $db->prepare("SELECT id, nombre_completo, usuario_paciente, cedula 
                                      FROM duenos 
                                      WHERE cedula = :cedula");
                $stmt->execute(['cedula' => $usuario]);
                $user = $stmt->fetch();
                
                if ($user) {
                    $_SESSION['usuario_id'] = $user['id'];
                    $_SESSION['nombre'] = $user['nombre_completo'];
                    $_SESSION['usuario'] = $user['usuario_paciente'];
                    $_SESSION['rol'] = 'paciente';
                    
                    header('Location: paciente/index.php');
                    exit();
                } else {
                    $error = 'Usuario o contraseña incorrectos';
                }
            } else {
                // Login de administrador o colaborador
                $stmt = $db->prepare("SELECT id, usuario, contrasena, nombre_completo, rol, activo 
                                      FROM usuarios 
                                      WHERE usuario = :usuario");
                $stmt->execute(['usuario' => $usuario]);
                $user = $stmt->fetch();
                
                if ($user) {
                    if ($user['activo'] != 1) {
                        $error = 'Usuario inactivo. Contacte al administrador';
                    } elseif (password_verify($contrasena, $user['contrasena'])) {
                        $_SESSION['usuario_id'] = $user['id'];
                        $_SESSION['nombre'] = $user['nombre_completo'];
                        $_SESSION['usuario'] = $user['usuario'];
                        $_SESSION['rol'] = $user['rol'];
                        
                        if ($user['rol'] === 'administrador') {
                            header('Location: admin/index.php');
                        } else {
                            header('Location: colaborador/index.php');
                        }
                        exit();
                    } else {
                        $error = 'Usuario o contraseña incorrectos';
                    }
                } else {
                    $error = 'Usuario o contraseña incorrectos';
                }
            }
        } catch (PDOException $e) {
            $error = 'Error en el sistema. Intente nuevamente';
        }
    }
}
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Iniciar Sesión - <?php echo APP_NAME; ?></title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.0/font/bootstrap-icons.css">
    <link rel="stylesheet" href="assets/css/style.css">
</head>
<body class="login-page">
    <div class="login-container">
        <div class="container">
            <div class="row justify-content-center">
                <div class="col-md-5">
                    <div class="card shadow-lg">
                        <div class="card-body p-5">
                            <div class="text-center mb-4">
                                <img src="assets/images/logo.png" alt="Logo Veterinaria" class="logo-login mb-3">
                                <h2 class="mb-2">Bienvenido</h2>
                                <p class="text-muted">Sistema de Administración de Veterinaria</p>
                            </div>
                            
                            <?php if (!empty($error)): ?>
                                <div class="alert alert-danger alert-dismissible fade show" role="alert">
                                    <i class="bi bi-exclamation-triangle-fill me-2"></i>
                                    <?php echo htmlspecialchars($error); ?>
                                    <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                                </div>
                            <?php endif; ?>
                            
                            <form method="POST" action="">
                                <div class="mb-3">
                                    <label class="form-label">Tipo de Usuario</label>
                                    <select name="tipo_usuario" id="tipo_usuario" class="form-select" required onchange="togglePassword()">
                                        <option value="sistema">Administrador / Colaborador</option>
                                        <option value="paciente">Dueño de Mascota</option>
                                    </select>
                                </div>
                                
                                <div class="mb-3">
                                    <label for="usuario" class="form-label" id="label_usuario">Usuario</label>
                                    <div class="input-group">
                                        <span class="input-group-text"><i class="bi bi-person"></i></span>
                                        <input type="text" class="form-control" id="usuario" name="usuario" 
                                               placeholder="Ingrese su usuario" required autofocus>
                                    </div>
                                </div>
                                
                                <div class="mb-4" id="password_field">
                                    <label for="contrasena" class="form-label">Contraseña</label>
                                    <div class="input-group">
                                        <span class="input-group-text"><i class="bi bi-lock"></i></span>
                                        <input type="password" class="form-control" id="contrasena" name="contrasena" 
                                               placeholder="Ingrese su contraseña">
                                    </div>
                                </div>
                                
                                <button type="submit" class="btn btn-primary w-100 mb-3">
                                    <i class="bi bi-box-arrow-in-right me-2"></i>Iniciar Sesión
                                </button>
                            </form>
                            
                            <div class="text-center text-muted small">
                                <p class="mb-0">Sistema Veterinaria v<?php echo APP_VERSION; ?></p>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        function togglePassword() {
            const tipoUsuario = document.getElementById('tipo_usuario').value;
            const passwordField = document.getElementById('password_field');
            const passwordInput = document.getElementById('contrasena');
            const labelUsuario = document.getElementById('label_usuario');
            const inputUsuario = document.getElementById('usuario');
            
            if (tipoUsuario === 'paciente') {
                // Ocultar campo de contraseña para pacientes
                passwordField.style.display = 'none';
                passwordInput.removeAttribute('required');
                labelUsuario.textContent = 'Cédula';
                inputUsuario.placeholder = 'Ingrese su número de cédula';
            } else {
                // Mostrar campo de contraseña para admin/colaborador
                passwordField.style.display = 'block';
                passwordInput.setAttribute('required', 'required');
                labelUsuario.textContent = 'Usuario';
                inputUsuario.placeholder = 'Ingrese su usuario';
            }
        }
        
        // Ejecutar al cargar la página
        document.addEventListener('DOMContentLoaded', togglePassword);
    </script>
</body>
</html>
