<?php
/**
 * Registrar Nueva Visita con Facturación
 * Sistema de Administración de Veterinaria
 */

require_once '../config/config.php';
requerir_autenticacion();

if (!tiene_rol('colaborador') && !tiene_rol('administrador')) {
    header('Location: ' . BASE_URL . '/index.php');
    exit();
}

$titulo_pagina = 'Nueva Visita';

$mascota_id = (int)($_GET['mascota_id'] ?? 0);
$mascota = null;

// Si se especificó una mascota, cargar sus datos
if ($mascota_id > 0) {
    try {
        $db = Database::getInstance()->getConnection();
        $stmt = $db->prepare("SELECT m.*, d.nombre_completo as dueno_nombre 
                              FROM mascotas m 
                              INNER JOIN duenos d ON m.dueno_id = d.id 
                              WHERE m.id = :id");
        $stmt->execute(['id' => $mascota_id]);
        $mascota = $stmt->fetch();
    } catch (PDOException $e) {
        $error = "Error al cargar la mascota: " . $e->getMessage();
    }
}

// Obtener lista de tratamientos y medicamentos
try {
    $db = Database::getInstance()->getConnection();
    
    $stmt = $db->query("SELECT * FROM tratamientos WHERE activo = 1 ORDER BY nombre");
    $tratamientos = $stmt->fetchAll();
    
    $stmt = $db->query("SELECT * FROM medicamentos WHERE activo = 1 ORDER BY nombre");
    $medicamentos = $stmt->fetchAll();
    
} catch (PDOException $e) {
    die("Error en la base de datos: " . $e->getMessage());
}

// Procesar formulario
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    try {
        $mascota_id = (int)($_POST['mascota_id'] ?? 0);
        
        // Validar que se seleccionó una mascota
        if ($mascota_id <= 0) {
            $error = 'Debe seleccionar una mascota';
        } else {
            // Verificar que la mascota existe y está activa
            $stmt = $db->prepare("SELECT id FROM mascotas WHERE id = :id AND activo = 1");
            $stmt->execute(['id' => $mascota_id]);
            if (!$stmt->fetch()) {
                $error = 'La mascota seleccionada no existe o está inactiva';
            }
        }
        
        // Si no hay errores, proceder con la transacción
        if (empty($error)) {
            $db->beginTransaction();
        
        $profesional_id = $_SESSION['usuario_id'];
        $fecha_visita = $_POST['fecha_visita'];
        $motivo = limpiar_entrada($_POST['motivo']);
        $diagnostico = limpiar_entrada($_POST['diagnostico']);
        $observaciones = limpiar_entrada($_POST['observaciones']);
        $peso_actual = !empty($_POST['peso_actual']) ? (float)$_POST['peso_actual'] : null;
        $proxima_cita = !empty($_POST['proxima_cita']) ? $_POST['proxima_cita'] : null;
        
        // Insertar visita
        $stmt = $db->prepare("INSERT INTO visitas (mascota_id, profesional_id, fecha_visita, motivo, diagnostico, observaciones, peso_actual, proxima_cita) 
                              VALUES (:mascota_id, :profesional_id, :fecha_visita, :motivo, :diagnostico, :observaciones, :peso_actual, :proxima_cita)");
        $stmt->execute([
            'mascota_id' => $mascota_id,
            'profesional_id' => $profesional_id,
            'fecha_visita' => $fecha_visita,
            'motivo' => $motivo,
            'diagnostico' => $diagnostico,
            'observaciones' => $observaciones,
            'peso_actual' => $peso_actual,
            'proxima_cita' => $proxima_cita
        ]);
        $visita_id = $db->lastInsertId();
        
        // Actualizar peso de la mascota si se proporcionó
        if ($peso_actual) {
            $stmt = $db->prepare("UPDATE mascotas SET peso = :peso WHERE id = :id");
            $stmt->execute(['peso' => $peso_actual, 'id' => $mascota_id]);
        }
        
        // Procesar tratamientos
        $subtotal_tratamientos = 0;
        if (!empty($_POST['tratamientos'])) {
            foreach ($_POST['tratamientos'] as $tratamiento_id) {
                $cantidad = (int)$_POST['tratamiento_cantidad'][$tratamiento_id];
                $precio = (float)$_POST['tratamiento_precio'][$tratamiento_id];
                $subtotal = $cantidad * $precio;
                $subtotal_tratamientos += $subtotal;
                
                $stmt = $db->prepare("INSERT INTO visita_tratamientos (visita_id, tratamiento_id, cantidad, precio_unitario, subtotal) 
                                      VALUES (:visita_id, :tratamiento_id, :cantidad, :precio, :subtotal)");
                $stmt->execute([
                    'visita_id' => $visita_id,
                    'tratamiento_id' => $tratamiento_id,
                    'cantidad' => $cantidad,
                    'precio' => $precio,
                    'subtotal' => $subtotal
                ]);
            }
        }
        
        // Procesar medicamentos
        $subtotal_medicamentos = 0;
        if (!empty($_POST['medicamentos'])) {
            foreach ($_POST['medicamentos'] as $medicamento_id) {
                $cantidad = (int)$_POST['medicamento_cantidad'][$medicamento_id];
                $precio = (float)$_POST['medicamento_precio'][$medicamento_id];
                $subtotal = $cantidad * $precio;
                $subtotal_medicamentos += $subtotal;
                
                $stmt = $db->prepare("INSERT INTO visita_medicamentos (visita_id, medicamento_id, cantidad, precio_unitario, subtotal) 
                                      VALUES (:visita_id, :medicamento_id, :cantidad, :precio, :subtotal)");
                $stmt->execute([
                    'visita_id' => $visita_id,
                    'medicamento_id' => $medicamento_id,
                    'cantidad' => $cantidad,
                    'precio' => $precio,
                    'subtotal' => $subtotal
                ]);
            }
        }
        
        // Crear factura
        $subtotal_factura = $subtotal_tratamientos + $subtotal_medicamentos;
        $descuento = !empty($_POST['descuento']) ? (float)$_POST['descuento'] : 0;
        $total_factura = $subtotal_factura - $descuento;
        $estado_factura = $_POST['estado_factura'] ?? 'pendiente';
        
        // Generar número de factura único
        $numero_factura = 'FAC-' . date('Ymd') . '-' . str_pad($visita_id, 6, '0', STR_PAD_LEFT);
        
        $stmt = $db->prepare("INSERT INTO facturas (visita_id, numero_factura, fecha_factura, subtotal, descuento, total, estado) 
                              VALUES (:visita_id, :numero, :fecha, :subtotal, :descuento, :total, :estado)");
        $stmt->execute([
            'visita_id' => $visita_id,
            'numero' => $numero_factura,
            'fecha' => date('Y-m-d', strtotime($fecha_visita)),
            'subtotal' => $subtotal_factura,
            'descuento' => $descuento,
            'total' => $total_factura,
            'estado' => $estado_factura
        ]);
        
        // Registrar vacunas si se proporcionaron
        if (!empty($_POST['vacuna_nombre'])) {
            $vacuna_nombre = limpiar_entrada($_POST['vacuna_nombre']);
            $vacuna_proxima = !empty($_POST['vacuna_proxima']) ? $_POST['vacuna_proxima'] : null;
            
            $stmt = $db->prepare("INSERT INTO vacunas (mascota_id, nombre_vacuna, fecha_aplicacion, proxima_dosis, profesional_id) 
                                  VALUES (:mascota_id, :nombre, :fecha, :proxima, :profesional_id)");
            $stmt->execute([
                'mascota_id' => $mascota_id,
                'nombre' => $vacuna_nombre,
                'fecha' => $fecha_visita,
                'proxima' => $vacuna_proxima,
                'profesional_id' => $profesional_id
            ]);
        }
        
            $db->commit();
            
            redirigir_con_mensaje('ver_mascota.php?id=' . $mascota_id, 'Visita registrada exitosamente', 'success');
        } // Fin del if (empty($error))
        
    } catch (PDOException $e) {
        if ($db->inTransaction()) {
            $db->rollBack();
        }
        $error = "Error al registrar la visita: " . $e->getMessage();
    }
}

include '../includes/header.php';
?>

<div class="row mb-4">
    <div class="col-12">
        <h1><i class="bi bi-plus-circle"></i> Registrar Nueva Visita</h1>
    </div>
</div>

<?php if (isset($error)): ?>
    <div class="alert alert-danger alert-dismissible fade show">
        <i class="bi bi-exclamation-triangle"></i> <?php echo htmlspecialchars($error); ?>
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    </div>
<?php endif; ?>

<form method="POST" action="" id="formVisita">
    <div class="row">
        <!-- Información de la Mascota -->
        <div class="col-md-4">
            <div class="card mb-4">
                <div class="card-header bg-primary text-white">
                    <h5 class="mb-0"><i class="bi bi-heart"></i> Mascota</h5>
                </div>
                <div class="card-body">
                    <?php if ($mascota): ?>
                        <input type="hidden" name="mascota_id" value="<?php echo $mascota['id']; ?>">
                        <div class="text-center mb-3">
                            <?php if ($mascota['foto']): ?>
                                <img src="<?php echo BASE_URL; ?>/uploads/mascotas/<?php echo htmlspecialchars($mascota['foto']); ?>" 
                                     class="img-fluid rounded" style="max-height: 150px;">
                            <?php else: ?>
                                <i class="bi bi-heart-fill text-muted" style="font-size: 4rem;"></i>
                            <?php endif; ?>
                        </div>
                        <h5 class="text-center"><?php echo htmlspecialchars($mascota['nombre']); ?></h5>
                        <p class="text-center text-muted mb-2"><?php echo ucfirst($mascota['tipo_mascota']); ?></p>
                        <p class="text-center"><strong>Dueño:</strong> <?php echo htmlspecialchars($mascota['dueno_nombre']); ?></p>
                    <?php else: ?>
                        <div class="mb-3">
                            <label class="form-label">Buscar Mascota *</label>
                            <input type="text" class="form-control" id="buscarMascota" placeholder="Buscar por nombre, código...">
                            <input type="hidden" name="mascota_id" id="mascotaId" required>
                        </div>
                        <div id="resultadosBusqueda"></div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
        
        <!-- Datos de la Visita -->
        <div class="col-md-8">
            <div class="card mb-4">
                <div class="card-header bg-success text-white">
                    <h5 class="mb-0"><i class="bi bi-calendar-check"></i> Datos de la Visita</h5>
                </div>
                <div class="card-body">
                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label class="form-label">Fecha de Visita *</label>
                            <input type="datetime-local" class="form-control" name="fecha_visita" 
                                   value="<?php echo date('Y-m-d\TH:i'); ?>" required>
                        </div>
                        <div class="col-md-6 mb-3">
                            <label class="form-label">Peso Actual (kg)</label>
                            <input type="number" class="form-control" name="peso_actual" step="0.01" placeholder="5.5">
                        </div>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Motivo de la Visita *</label>
                        <input type="text" class="form-control" name="motivo" required 
                               placeholder="Ej: Consulta general, vacunación, control">
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Diagnóstico</label>
                        <textarea class="form-control" name="diagnostico" rows="2"></textarea>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Observaciones</label>
                        <textarea class="form-control" name="observaciones" rows="2"></textarea>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Próxima Cita</label>
                        <input type="date" class="form-control" name="proxima_cita">
                    </div>
                </div>
            </div>
            
            <!-- Vacunación -->
            <div class="card mb-4">
                <div class="card-header bg-warning text-white">
                    <h5 class="mb-0"><i class="bi bi-shield-check"></i> Vacunación (Opcional)</h5>
                </div>
                <div class="card-body">
                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label class="form-label">Nombre de la Vacuna</label>
                            <input type="text" class="form-control" name="vacuna_nombre" 
                                   placeholder="Ej: Antirrábica, Séxtuple">
                        </div>
                        <div class="col-md-6 mb-3">
                            <label class="form-label">Próxima Dosis</label>
                            <input type="date" class="form-control" name="vacuna_proxima">
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Facturación -->
    <div class="card mb-4">
        <div class="card-header bg-info text-white">
            <h5 class="mb-0"><i class="bi bi-receipt"></i> Facturación</h5>
        </div>
        <div class="card-body">
            <!-- Tratamientos -->
            <h6 class="mb-3">Tratamientos</h6>
            <div id="tratamientosContainer">
                <?php foreach ($tratamientos as $tratamiento): ?>
                    <div class="form-check mb-2">
                        <input class="form-check-input tratamiento-check" type="checkbox" 
                               name="tratamientos[]" value="<?php echo $tratamiento['id']; ?>" 
                               id="trat_<?php echo $tratamiento['id']; ?>"
                               data-precio="<?php echo $tratamiento['precio_base']; ?>">
                        <label class="form-check-label" for="trat_<?php echo $tratamiento['id']; ?>">
                            <?php echo htmlspecialchars($tratamiento['nombre']); ?> - 
                            <strong><?php echo formatear_moneda($tratamiento['precio_base']); ?></strong>
                        </label>
                        <div class="ms-4 mt-2 tratamiento-detalles" id="detalles_trat_<?php echo $tratamiento['id']; ?>" style="display:none;">
                            <div class="row">
                                <div class="col-md-3">
                                    <label class="form-label">Cantidad</label>
                                    <input type="number" class="form-control form-control-sm cantidad-input" 
                                           name="tratamiento_cantidad[<?php echo $tratamiento['id']; ?>]" 
                                           value="1" min="1">
                                </div>
                                <div class="col-md-4">
                                    <label class="form-label">Precio Unitario</label>
                                    <input type="number" class="form-control form-control-sm precio-input" 
                                           name="tratamiento_precio[<?php echo $tratamiento['id']; ?>]" 
                                           value="<?php echo $tratamiento['precio_base']; ?>" step="1000">
                                </div>
                            </div>
                        </div>
                    </div>
                <?php endforeach; ?>
            </div>
            
            <hr>
            
            <!-- Medicamentos -->
            <h6 class="mb-3">Medicamentos</h6>
            <div id="medicamentosContainer">
                <?php foreach ($medicamentos as $medicamento): ?>
                    <div class="form-check mb-2">
                        <input class="form-check-input medicamento-check" type="checkbox" 
                               name="medicamentos[]" value="<?php echo $medicamento['id']; ?>" 
                               id="med_<?php echo $medicamento['id']; ?>"
                               data-precio="<?php echo $medicamento['precio_unitario']; ?>">
                        <label class="form-check-label" for="med_<?php echo $medicamento['id']; ?>">
                            <?php echo htmlspecialchars($medicamento['nombre']); ?> - 
                            <strong><?php echo formatear_moneda($medicamento['precio_unitario']); ?></strong>
                        </label>
                        <div class="ms-4 mt-2 medicamento-detalles" id="detalles_med_<?php echo $medicamento['id']; ?>" style="display:none;">
                            <div class="row">
                                <div class="col-md-3">
                                    <label class="form-label">Cantidad</label>
                                    <input type="number" class="form-control form-control-sm cantidad-input" 
                                           name="medicamento_cantidad[<?php echo $medicamento['id']; ?>]" 
                                           value="1" min="1">
                                </div>
                                <div class="col-md-4">
                                    <label class="form-label">Precio Unitario</label>
                                    <input type="number" class="form-control form-control-sm precio-input" 
                                           name="medicamento_precio[<?php echo $medicamento['id']; ?>]" 
                                           value="<?php echo $medicamento['precio_unitario']; ?>" step="1000">
                                </div>
                            </div>
                        </div>
                    </div>
                <?php endforeach; ?>
            </div>
            
            <hr>
            
            <!-- Total -->
            <div class="row">
                <div class="col-md-6">
                    <label class="form-label">Estado de Pago</label>
                    <select class="form-select" name="estado_factura">
                        <option value="pendiente">Pendiente</option>
                        <option value="pagada">Pagada</option>
                    </select>
                </div>
                <div class="col-md-6">
                    <div class="alert alert-success mb-0">
                        <h5 class="mb-0">Total: <span id="totalFactura">Gs. 0</span></h5>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <div class="card">
        <div class="card-body text-center">
            <button type="submit" class="btn btn-primary btn-lg">
                <i class="bi bi-save"></i> Registrar Visita y Facturar
            </button>
            <a href="index.php" class="btn btn-secondary btn-lg">
                <i class="bi bi-x-circle"></i> Cancelar
            </a>
        </div>
    </div>
</form>

<script>
// Mostrar/ocultar detalles de tratamientos
document.querySelectorAll('.tratamiento-check').forEach(function(checkbox) {
    checkbox.addEventListener('change', function() {
        const detalles = document.getElementById('detalles_trat_' + this.value);
        detalles.style.display = this.checked ? 'block' : 'none';
        calcularTotal();
    });
});

// Mostrar/ocultar detalles de medicamentos
document.querySelectorAll('.medicamento-check').forEach(function(checkbox) {
    checkbox.addEventListener('change', function() {
        const detalles = document.getElementById('detalles_med_' + this.value);
        detalles.style.display = this.checked ? 'block' : 'none';
        calcularTotal();
    });
});

// Calcular total cuando cambian cantidades o precios
document.querySelectorAll('.cantidad-input, .precio-input').forEach(function(input) {
    input.addEventListener('input', calcularTotal);
});

function calcularTotal() {
    let total = 0;
    
    // Sumar tratamientos
    document.querySelectorAll('.tratamiento-check:checked').forEach(function(checkbox) {
        const id = checkbox.value;
        const cantidad = parseInt(document.querySelector(`input[name="tratamiento_cantidad[${id}]"]`).value) || 0;
        const precio = parseFloat(document.querySelector(`input[name="tratamiento_precio[${id}]"]`).value) || 0;
        total += cantidad * precio;
    });
    
    // Sumar medicamentos
    document.querySelectorAll('.medicamento-check:checked').forEach(function(checkbox) {
        const id = checkbox.value;
        const cantidad = parseInt(document.querySelector(`input[name="medicamento_cantidad[${id}]"]`).value) || 0;
        const precio = parseFloat(document.querySelector(`input[name="medicamento_precio[${id}]"]`).value) || 0;
        total += cantidad * precio;
    });
    
    document.getElementById('totalFactura').textContent = 'Gs. ' + total.toLocaleString('es-PY');
}
</script>

<?php include '../includes/footer.php'; ?>
