<?php
/**
 * Generar Reporte PDF de Visita
 * Sistema de Administración de Veterinaria
 */

require_once '../config/config.php';
require_once '../vendor/autoload.php'; // Para FPDF o similar

requerir_autenticacion();

if (!tiene_rol('colaborador') && !tiene_rol('administrador')) {
    header('Location: ' . BASE_URL . '/index.php');
    exit();
}

$visita_id = (int)($_GET['id'] ?? 0);

if ($visita_id <= 0) {
    die('Visita no encontrada');
}

try {
    $db = Database::getInstance()->getConnection();
    
    // Obtener datos de la visita
    $stmt = $db->prepare("SELECT v.*, m.nombre as mascota_nombre, m.tipo_mascota, m.raza, m.codigo_barras,
                                 d.nombre_completo as dueno_nombre, d.telefono, d.direccion, d.cedula,
                                 u.nombre_completo as profesional_nombre,
                                 f.total as factura_total, f.estado as factura_estado
                          FROM visitas v
                          INNER JOIN mascotas m ON v.mascota_id = m.id
                          INNER JOIN duenos d ON m.dueno_id = d.id
                          INNER JOIN usuarios u ON v.profesional_id = u.id
                          LEFT JOIN facturas f ON v.id = f.visita_id
                          WHERE v.id = :id");
    $stmt->execute(['id' => $visita_id]);
    $visita = $stmt->fetch();
    
    if (!$visita) {
        die('Visita no encontrada');
    }
    
    // Obtener tratamientos de la visita
    $stmt = $db->prepare("SELECT vt.*, t.nombre as tratamiento_nombre
                          FROM visita_tratamientos vt
                          INNER JOIN tratamientos t ON vt.tratamiento_id = t.id
                          WHERE vt.visita_id = :visita_id");
    $stmt->execute(['visita_id' => $visita_id]);
    $tratamientos = $stmt->fetchAll();
    
    // Obtener medicamentos de la visita
    $stmt = $db->prepare("SELECT vm.*, m.nombre as medicamento_nombre
                          FROM visita_medicamentos vm
                          INNER JOIN medicamentos m ON vm.medicamento_id = m.id
                          WHERE vm.visita_id = :visita_id");
    $stmt->execute(['visita_id' => $visita_id]);
    $medicamentos = $stmt->fetchAll();
    
} catch (PDOException $e) {
    die("Error en la base de datos: " . $e->getMessage());
}

// Generar HTML para el reporte (que luego se puede convertir a PDF)
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Reporte de Visita - <?php echo htmlspecialchars($visita['mascota_nombre']); ?></title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <style>
        @media print {
            .no-print {
                display: none;
            }
            body {
                background: white;
            }
        }
        
        .reporte {
            max-width: 800px;
            margin: 20px auto;
            background: white;
            padding: 30px;
            box-shadow: 0 0 10px rgba(0,0,0,0.1);
        }
        
        .reporte-header {
            border-bottom: 3px solid #5fa89c;
            padding-bottom: 15px;
            margin-bottom: 20px;
        }
        
        .reporte-header img {
            height: 60px;
        }
        
        .reporte-section {
            margin-bottom: 20px;
        }
        
        .reporte-section h5 {
            background: #5fa89c;
            color: white;
            padding: 8px 15px;
            margin-bottom: 10px;
            border-radius: 5px;
        }
        
        .info-row {
            display: flex;
            padding: 5px 0;
            border-bottom: 1px solid #eee;
        }
        
        .info-label {
            font-weight: bold;
            width: 150px;
            flex-shrink: 0;
        }
        
        .info-value {
            flex-grow: 1;
        }
        
        table {
            width: 100%;
            margin-top: 10px;
        }
        
        table th {
            background: #f8f9fa;
            padding: 8px;
            text-align: left;
        }
        
        table td {
            padding: 8px;
            border-bottom: 1px solid #eee;
        }
        
        .total-box {
            background: #5fa89c;
            color: white;
            padding: 15px;
            text-align: right;
            border-radius: 5px;
            margin-top: 20px;
        }
        
        .firma {
            margin-top: 50px;
            text-align: center;
        }
        
        .firma-linea {
            border-top: 2px solid #333;
            width: 300px;
            margin: 50px auto 10px;
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="text-center mb-3 no-print">
            <button onclick="window.print()" class="btn btn-primary">
                <i class="bi bi-printer"></i> Imprimir Reporte
            </button>
            <a href="ver_mascota.php?id=<?php echo $visita['mascota_id']; ?>" class="btn btn-secondary">
                <i class="bi bi-arrow-left"></i> Volver
            </a>
        </div>
        
        <div class="reporte">
            <!-- Header -->
            <div class="reporte-header">
                <div class="row align-items-center">
                    <div class="col-md-3">
                        <img src="<?php echo BASE_URL; ?>/assets/images/logo.png" alt="Logo" class="img-fluid">
                    </div>
                    <div class="col-md-9 text-end">
                        <h2>REPORTE DE VISITA</h2>
                        <p class="mb-0">Fecha: <?php echo formatear_fecha($visita['fecha_visita'], true); ?></p>
                        <p class="mb-0">Nº de Visita: <?php echo str_pad($visita_id, 6, '0', STR_PAD_LEFT); ?></p>
                    </div>
                </div>
            </div>
            
            <!-- Información de la Mascota -->
            <div class="reporte-section">
                <h5>Información de la Mascota</h5>
                <div class="info-row">
                    <div class="info-label">Nombre:</div>
                    <div class="info-value"><?php echo htmlspecialchars($visita['mascota_nombre']); ?></div>
                </div>
                <div class="info-row">
                    <div class="info-label">Tipo:</div>
                    <div class="info-value"><?php echo ucfirst($visita['tipo_mascota']); ?></div>
                </div>
                <?php if ($visita['raza']): ?>
                    <div class="info-row">
                        <div class="info-label">Raza:</div>
                        <div class="info-value"><?php echo htmlspecialchars($visita['raza']); ?></div>
                    </div>
                <?php endif; ?>
                <div class="info-row">
                    <div class="info-label">Código:</div>
                    <div class="info-value"><?php echo htmlspecialchars($visita['codigo_barras']); ?></div>
                </div>
            </div>
            
            <!-- Información del Dueño -->
            <div class="reporte-section">
                <h5>Información del Dueño</h5>
                <div class="info-row">
                    <div class="info-label">Nombre:</div>
                    <div class="info-value"><?php echo htmlspecialchars($visita['dueno_nombre']); ?></div>
                </div>
                <?php if ($visita['cedula']): ?>
                    <div class="info-row">
                        <div class="info-label">CI:</div>
                        <div class="info-value"><?php echo htmlspecialchars($visita['cedula']); ?></div>
                    </div>
                <?php endif; ?>
                <div class="info-row">
                    <div class="info-label">Teléfono:</div>
                    <div class="info-value"><?php echo htmlspecialchars($visita['telefono']); ?></div>
                </div>
                <?php if ($visita['direccion']): ?>
                    <div class="info-row">
                        <div class="info-label">Dirección:</div>
                        <div class="info-value"><?php echo htmlspecialchars($visita['direccion']); ?></div>
                    </div>
                <?php endif; ?>
            </div>
            
            <!-- Detalles de la Visita -->
            <div class="reporte-section">
                <h5>Detalles de la Visita</h5>
                <div class="info-row">
                    <div class="info-label">Profesional:</div>
                    <div class="info-value">Dr. <?php echo htmlspecialchars($visita['profesional_nombre']); ?></div>
                </div>
                <div class="info-row">
                    <div class="info-label">Motivo:</div>
                    <div class="info-value"><?php echo htmlspecialchars($visita['motivo']); ?></div>
                </div>
                <?php if ($visita['diagnostico']): ?>
                    <div class="info-row">
                        <div class="info-label">Diagnóstico:</div>
                        <div class="info-value"><?php echo htmlspecialchars($visita['diagnostico']); ?></div>
                    </div>
                <?php endif; ?>
                <?php if ($visita['observaciones']): ?>
                    <div class="info-row">
                        <div class="info-label">Observaciones:</div>
                        <div class="info-value"><?php echo htmlspecialchars($visita['observaciones']); ?></div>
                    </div>
                <?php endif; ?>
                <?php if ($visita['peso_actual']): ?>
                    <div class="info-row">
                        <div class="info-label">Peso:</div>
                        <div class="info-value"><?php echo $visita['peso_actual']; ?> kg</div>
                    </div>
                <?php endif; ?>
                <?php if ($visita['proxima_cita']): ?>
                    <div class="info-row">
                        <div class="info-label">Próxima Cita:</div>
                        <div class="info-value"><?php echo formatear_fecha($visita['proxima_cita']); ?></div>
                    </div>
                <?php endif; ?>
            </div>
            
            <!-- Tratamientos -->
            <?php if (count($tratamientos) > 0): ?>
                <div class="reporte-section">
                    <h5>Tratamientos Realizados</h5>
                    <table>
                        <thead>
                            <tr>
                                <th>Tratamiento</th>
                                <th>Cantidad</th>
                                <th>Precio Unit.</th>
                                <th>Subtotal</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($tratamientos as $trat): ?>
                                <tr>
                                    <td><?php echo htmlspecialchars($trat['tratamiento_nombre']); ?></td>
                                    <td><?php echo $trat['cantidad']; ?></td>
                                    <td><?php echo formatear_moneda($trat['precio_unitario']); ?></td>
                                    <td><?php echo formatear_moneda($trat['subtotal']); ?></td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            <?php endif; ?>
            
            <!-- Medicamentos -->
            <?php if (count($medicamentos) > 0): ?>
                <div class="reporte-section">
                    <h5>Medicamentos Administrados</h5>
                    <table>
                        <thead>
                            <tr>
                                <th>Medicamento</th>
                                <th>Cantidad</th>
                                <th>Precio Unit.</th>
                                <th>Subtotal</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($medicamentos as $med): ?>
                                <tr>
                                    <td><?php echo htmlspecialchars($med['medicamento_nombre']); ?></td>
                                    <td><?php echo $med['cantidad']; ?></td>
                                    <td><?php echo formatear_moneda($med['precio_unitario']); ?></td>
                                    <td><?php echo formatear_moneda($med['subtotal']); ?></td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            <?php endif; ?>
            
            <!-- Total -->
            <?php if ($visita['factura_total']): ?>
                <div class="total-box">
                    <h4 class="mb-0">TOTAL: <?php echo formatear_moneda($visita['factura_total']); ?></h4>
                    <p class="mb-0">Estado: <?php echo $visita['factura_estado'] === 'pagada' ? 'PAGADO' : 'PENDIENTE'; ?></p>
                </div>
            <?php endif; ?>
            
            <!-- Firma -->
            <div class="firma">
                <div class="firma-linea"></div>
                <p class="mb-0"><strong>Dr. <?php echo htmlspecialchars($visita['profesional_nombre']); ?></strong></p>
                <p class="text-muted">Profesional Veterinario</p>
            </div>
        </div>
    </div>
</body>
</html>
