<?php
/**
 * Agregar Mascota a Dueño Existente
 * Sistema de Administración de Veterinaria
 */

require_once '../config/config.php';
requerir_autenticacion();

if (!tiene_rol('colaborador') && !tiene_rol('administrador')) {
    header('Location: ' . BASE_URL . '/index.php');
    exit();
}

// Verificar que se recibió un ID de dueño
if (!isset($_GET['dueno_id']) || empty($_GET['dueno_id'])) {
    header('Location: ' . BASE_URL . '/colaborador/duenos.php');
    exit();
}

$dueno_id = (int)$_GET['dueno_id'];

// Obtener datos del dueño
try {
    $db = Database::getInstance()->getConnection();
    $stmt = $db->prepare("SELECT * FROM duenos WHERE id = :id");
    $stmt->execute(['id' => $dueno_id]);
    $dueno = $stmt->fetch();
    
    if (!$dueno) {
        $_SESSION['error'] = 'El dueño no existe';
        header('Location: ' . BASE_URL . '/colaborador/duenos.php');
        exit();
    }
    
    // Obtener mascotas del dueño
    $stmt = $db->prepare("SELECT * FROM mascotas WHERE dueno_id = :id AND activo = 1 ORDER BY nombre ASC");
    $stmt->execute(['id' => $dueno_id]);
    $mascotas = $stmt->fetchAll();
    
} catch (PDOException $e) {
    die("Error en la base de datos: " . $e->getMessage());
}

$titulo_pagina = 'Agregar Mascota - ' . $dueno['nombre_completo'];

// Procesar formulario
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    try {
        $db->beginTransaction();
        
        // Datos de la mascota
        $nombre_mascota = limpiar_entrada($_POST['nombre_mascota']);
        $tipo_mascota = limpiar_entrada($_POST['tipo_mascota']);
        $raza = limpiar_entrada($_POST['raza']);
        $fecha_nacimiento = $_POST['fecha_nacimiento'];
        $peso = !empty($_POST['peso']) ? (float)$_POST['peso'] : null;
        
        // Generar código de barras único
        $codigo_barras = generar_codigo_barras();
        while (true) {
            $stmt = $db->prepare("SELECT id FROM mascotas WHERE codigo_barras = :codigo");
            $stmt->execute(['codigo' => $codigo_barras]);
            if (!$stmt->fetch()) break;
            $codigo_barras = generar_codigo_barras();
        }
        
        // Procesar foto
        $foto_path = null;
        if (isset($_FILES['foto']) && $_FILES['foto']['error'] === UPLOAD_ERR_OK) {
            $resultado = subir_archivo($_FILES['foto'], 'mascotas');
            if ($resultado['exito']) {
                $foto_path = $resultado['path'];
            }
        }
        
        // Insertar mascota
        $stmt = $db->prepare("INSERT INTO mascotas (dueno_id, nombre, tipo_mascota, raza, fecha_nacimiento, 
                              peso, codigo_barras, foto) 
                              VALUES (:dueno_id, :nombre, :tipo, :raza, :fecha_nac, :peso, :codigo, :foto)");
        $stmt->execute([
            'dueno_id' => $dueno_id,
            'nombre' => $nombre_mascota,
            'tipo' => $tipo_mascota,
            'raza' => $raza,
            'fecha_nac' => $fecha_nacimiento,
            'peso' => $peso,
            'codigo' => $codigo_barras,
            'foto' => $foto_path
        ]);
        
        $db->commit();
        
        $_SESSION['exito'] = 'Mascota "' . $nombre_mascota . '" agregada exitosamente';
        
        // Recargar la página para agregar otra mascota o ir a ver dueño
        header('Location: agregar_mascota.php?dueno_id=' . $dueno_id);
        exit();
        
    } catch (Exception $e) {
        if ($db->inTransaction()) {
            $db->rollBack();
        }
        $error = 'Error al agregar la mascota: ' . $e->getMessage();
    }
}

include '../includes/header.php';
?>

<div class="row mb-4">
    <div class="col-md-12">
        <h1><i class="bi bi-heart-fill"></i> Agregar Mascota</h1>
        <p class="text-muted">Dueño: <strong><?php echo htmlspecialchars($dueno['nombre_completo']); ?></strong></p>
    </div>
</div>

<?php if (isset($_SESSION['exito'])): ?>
    <div class="alert alert-success alert-dismissible fade show">
        <i class="bi bi-check-circle"></i> <?php echo $_SESSION['exito']; unset($_SESSION['exito']); ?>
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    </div>
<?php endif; ?>

<?php if (isset($error)): ?>
    <div class="alert alert-danger alert-dismissible fade show">
        <i class="bi bi-exclamation-triangle"></i> <?php echo $error; ?>
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    </div>
<?php endif; ?>

<div class="row">
    <!-- Información del Dueño -->
    <div class="col-md-4">
        <div class="card mb-4">
            <div class="card-header bg-info text-white">
                <h5 class="mb-0"><i class="bi bi-person"></i> Información del Dueño</h5>
            </div>
            <div class="card-body">
                <p><strong>Nombre:</strong><br><?php echo htmlspecialchars($dueno['nombre_completo']); ?></p>
                <p><strong>Cédula:</strong><br><?php echo htmlspecialchars($dueno['cedula'] ?? '-'); ?></p>
                <p><strong>Teléfono:</strong><br>
                    <?php if ($dueno['tiene_whatsapp']): ?>
                        <a href="https://wa.me/<?php echo preg_replace('/[^0-9]/', '', $dueno['telefono']); ?>" 
                           target="_blank" class="btn btn-sm btn-success">
                            <i class="bi bi-whatsapp"></i> <?php echo htmlspecialchars($dueno['telefono']); ?>
                        </a>
                    <?php elseif ($dueno['tiene_telegram']): ?>
                        <a href="https://t.me/<?php echo preg_replace('/[^0-9]/', '', $dueno['telefono']); ?>" 
                           target="_blank" class="btn btn-sm btn-info">
                            <i class="bi bi-telegram"></i> <?php echo htmlspecialchars($dueno['telefono']); ?>
                        </a>
                    <?php else: ?>
                        <?php echo htmlspecialchars($dueno['telefono']); ?>
                    <?php endif; ?>
                </p>
                <p><strong>Email:</strong><br><?php echo htmlspecialchars($dueno['email'] ?? '-'); ?></p>
                <p><strong>Dirección:</strong><br><?php echo htmlspecialchars($dueno['direccion'] ?? '-'); ?></p>
                
                <hr>
                
                <h6><i class="bi bi-heart"></i> Mascotas Registradas (<?php echo count($mascotas); ?>)</h6>
                <?php if (count($mascotas) > 0): ?>
                    <ul class="list-group list-group-flush">
                        <?php foreach ($mascotas as $m): ?>
                            <li class="list-group-item px-0">
                                <i class="bi bi-check-circle text-success"></i> 
                                <strong><?php echo htmlspecialchars($m['nombre']); ?></strong>
                                <br><small class="text-muted"><?php echo ucfirst($m['tipo_mascota']); ?> - <?php echo htmlspecialchars($m['raza'] ?? 'Sin raza'); ?></small>
                            </li>
                        <?php endforeach; ?>
                    </ul>
                <?php else: ?>
                    <p class="text-muted"><small>Aún no tiene mascotas registradas</small></p>
                <?php endif; ?>
                
                <div class="mt-3">
                    <a href="ver_dueno.php?id=<?php echo $dueno_id; ?>" class="btn btn-sm btn-outline-primary w-100">
                        <i class="bi bi-eye"></i> Ver Perfil Completo
                    </a>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Formulario de Nueva Mascota -->
    <div class="col-md-8">
        <form method="POST" action="" enctype="multipart/form-data">
            <div class="card">
                <div class="card-header bg-success text-white">
                    <h5 class="mb-0"><i class="bi bi-heart"></i> Nueva Mascota</h5>
                </div>
                <div class="card-body">
                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label class="form-label">Nombre de la Mascota *</label>
                                <input type="text" class="form-control" name="nombre_mascota" required>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label class="form-label">Tipo de Mascota *</label>
                                <select class="form-select" name="tipo_mascota" required>
                                    <option value="">Seleccione...</option>
                                    <option value="perro">Perro</option>
                                    <option value="gato">Gato</option>
                                    <option value="ave">Ave</option>
                                    <option value="conejo">Conejo</option>
                                    <option value="hamster">Hámster</option>
                                    <option value="reptil">Reptil</option>
                                    <option value="otro">Otro</option>
                                </select>
                            </div>
                        </div>
                    </div>
                    
                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label class="form-label">Raza</label>
                                <input type="text" class="form-control" name="raza">
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label class="form-label">Fecha de Nacimiento</label>
                                <input type="date" class="form-control" name="fecha_nacimiento">
                            </div>
                        </div>
                    </div>
                    
                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label class="form-label">Peso (kg) - Opcional</label>
                                <input type="number" class="form-control" name="peso" step="0.01" placeholder="5.5">
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label class="form-label">Fotografía</label>
                                <input type="file" class="form-control" name="foto" accept="image/*">
                                <small class="text-muted">Formatos: JPG, PNG, GIF. Máx 5MB</small>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            
            <div class="card mt-3">
                <div class="card-body text-center">
                    <button type="submit" class="btn btn-success btn-lg">
                        <i class="bi bi-plus-circle"></i> Agregar Mascota
                    </button>
                    <a href="duenos.php" class="btn btn-primary btn-lg">
                        <i class="bi bi-check-circle"></i> Finalizar
                    </a>
                    <a href="duenos.php" class="btn btn-secondary btn-lg">
                        <i class="bi bi-x-circle"></i> Cancelar
                    </a>
                </div>
            </div>
        </form>
    </div>
</div>

<?php include '../includes/footer.php'; ?>
