<?php
/**
 * Gestión de Medicamentos
 * Sistema de Administración de Veterinaria
 */

require_once '../config/config.php';
requerir_rol('administrador');

$titulo_pagina = 'Gestión de Medicamentos';

// Procesar acciones
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $accion = $_POST['accion'] ?? '';
    
    try {
        $db = Database::getInstance()->getConnection();
        
        if ($accion === 'crear') {
            $nombre = limpiar_entrada($_POST['nombre']);
            $descripcion = limpiar_entrada($_POST['descripcion']);
            $precio_unitario = (float)$_POST['precio_unitario'];
            $stock = (int)$_POST['stock'];
            $unidad_medida = limpiar_entrada($_POST['unidad_medida']);
            
            $stmt = $db->prepare("INSERT INTO medicamentos (nombre, descripcion, precio_unitario, stock, unidad_medida) 
                                  VALUES (:nombre, :descripcion, :precio_unitario, :stock, :unidad_medida)");
            $stmt->execute([
                'nombre' => $nombre,
                'descripcion' => $descripcion,
                'precio_unitario' => $precio_unitario,
                'stock' => $stock,
                'unidad_medida' => $unidad_medida
            ]);
            
            redirigir_con_mensaje('medicamentos.php', 'Medicamento creado exitosamente', 'success');
            
        } elseif ($accion === 'editar') {
            $id = (int)$_POST['id'];
            $nombre = limpiar_entrada($_POST['nombre']);
            $descripcion = limpiar_entrada($_POST['descripcion']);
            $precio_unitario = (float)$_POST['precio_unitario'];
            $stock = (int)$_POST['stock'];
            $unidad_medida = limpiar_entrada($_POST['unidad_medida']);
            $activo = isset($_POST['activo']) ? 1 : 0;
            
            $stmt = $db->prepare("UPDATE medicamentos 
                                  SET nombre = :nombre, descripcion = :descripcion, 
                                      precio_unitario = :precio_unitario, stock = :stock, 
                                      unidad_medida = :unidad_medida, activo = :activo
                                  WHERE id = :id");
            $stmt->execute([
                'nombre' => $nombre,
                'descripcion' => $descripcion,
                'precio_unitario' => $precio_unitario,
                'stock' => $stock,
                'unidad_medida' => $unidad_medida,
                'activo' => $activo,
                'id' => $id
            ]);
            
            redirigir_con_mensaje('medicamentos.php', 'Medicamento actualizado exitosamente', 'success');
            
        } elseif ($accion === 'eliminar') {
            $id = (int)$_POST['id'];
            $stmt = $db->prepare("DELETE FROM medicamentos WHERE id = :id");
            $stmt->execute(['id' => $id]);
            
            redirigir_con_mensaje('medicamentos.php', 'Medicamento eliminado exitosamente', 'success');
        }
        
    } catch (PDOException $e) {
        redirigir_con_mensaje('medicamentos.php', 'Error: ' . $e->getMessage(), 'danger');
    }
}

// Obtener medicamentos
try {
    $db = Database::getInstance()->getConnection();
    $stmt = $db->query("SELECT * FROM medicamentos ORDER BY nombre");
    $medicamentos = $stmt->fetchAll();
} catch (PDOException $e) {
    die("Error en la base de datos: " . $e->getMessage());
}

include '../includes/header.php';
?>

<div class="row mb-4">
    <div class="col-md-6">
        <h1><i class="bi bi-capsule"></i> Gestión de Medicamentos</h1>
    </div>
    <div class="col-md-6 text-end">
        <button type="button" class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#modalNuevoMedicamento">
            <i class="bi bi-plus-circle"></i> Nuevo Medicamento
        </button>
    </div>
</div>

<div class="card">
    <div class="card-body">
        <div class="table-responsive">
            <table class="table table-hover">
                <thead>
                    <tr>
                        <th>Nombre</th>
                        <th>Descripción</th>
                        <th>Precio Unitario</th>
                        <th>Stock</th>
                        <th>Unidad</th>
                        <th>Estado</th>
                        <th>Acciones</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($medicamentos as $med): ?>
                        <tr class="<?php echo $med['stock'] < 10 ? 'table-warning' : ''; ?>">
                            <td><strong><?php echo htmlspecialchars($med['nombre']); ?></strong></td>
                            <td><?php echo htmlspecialchars($med['descripcion'] ?? '-'); ?></td>
                            <td><?php echo formatear_moneda($med['precio_unitario']); ?></td>
                            <td>
                                <?php echo formatear_numero($med['stock']); ?>
                                <?php if ($med['stock'] < 10): ?>
                                    <i class="bi bi-exclamation-triangle text-warning" title="Stock bajo"></i>
                                <?php endif; ?>
                            </td>
                            <td><?php echo htmlspecialchars($med['unidad_medida']); ?></td>
                            <td>
                                <?php if ($med['activo']): ?>
                                    <span class="badge bg-success">Activo</span>
                                <?php else: ?>
                                    <span class="badge bg-secondary">Inactivo</span>
                                <?php endif; ?>
                            </td>
                            <td>
                                <button type="button" class="btn btn-sm btn-warning" 
                                        onclick="editarMedicamento(<?php echo htmlspecialchars(json_encode($med)); ?>)">
                                    <i class="bi bi-pencil"></i>
                                </button>
                                <button type="button" class="btn btn-sm btn-danger" 
                                        onclick="eliminarMedicamento(<?php echo $med['id']; ?>, '<?php echo htmlspecialchars($med['nombre']); ?>')">
                                    <i class="bi bi-trash"></i>
                                </button>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
    </div>
</div>

<!-- Modal Nuevo Medicamento -->
<div class="modal fade" id="modalNuevoMedicamento" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <form method="POST" action="">
                <input type="hidden" name="accion" value="crear">
                <div class="modal-header">
                    <h5 class="modal-title">Nuevo Medicamento</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <div class="mb-3">
                        <label class="form-label">Nombre *</label>
                        <input type="text" class="form-control" name="nombre" required>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Descripción</label>
                        <textarea class="form-control" name="descripcion" rows="2"></textarea>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Precio Unitario (<?php echo MONEDA; ?>) *</label>
                        <input type="number" class="form-control" name="precio_unitario" step="1" required>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Stock Inicial *</label>
                        <input type="number" class="form-control" name="stock" required>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Unidad de Medida *</label>
                        <input type="text" class="form-control" name="unidad_medida" placeholder="ej: ml, comprimidos, dosis" required>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancelar</button>
                    <button type="submit" class="btn btn-primary">Guardar</button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Modal Editar Medicamento -->
<div class="modal fade" id="modalEditarMedicamento" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <form method="POST" action="">
                <input type="hidden" name="accion" value="editar">
                <input type="hidden" name="id" id="edit_id">
                <div class="modal-header">
                    <h5 class="modal-title">Editar Medicamento</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <div class="mb-3">
                        <label class="form-label">Nombre *</label>
                        <input type="text" class="form-control" name="nombre" id="edit_nombre" required>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Descripción</label>
                        <textarea class="form-control" name="descripcion" id="edit_descripcion" rows="2"></textarea>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Precio Unitario (<?php echo MONEDA; ?>) *</label>
                        <input type="number" class="form-control" name="precio_unitario" id="edit_precio_unitario" step="1" required>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Stock *</label>
                        <input type="number" class="form-control" name="stock" id="edit_stock" required>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Unidad de Medida *</label>
                        <input type="text" class="form-control" name="unidad_medida" id="edit_unidad_medida" required>
                    </div>
                    <div class="mb-3">
                        <div class="form-check">
                            <input class="form-check-input" type="checkbox" name="activo" id="edit_activo">
                            <label class="form-check-label" for="edit_activo">Activo</label>
                        </div>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancelar</button>
                    <button type="submit" class="btn btn-primary">Actualizar</button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Formulario oculto para eliminar -->
<form method="POST" action="" id="formEliminar" style="display: none;">
    <input type="hidden" name="accion" value="eliminar">
    <input type="hidden" name="id" id="delete_id">
</form>

<script>
function editarMedicamento(med) {
    document.getElementById('edit_id').value = med.id;
    document.getElementById('edit_nombre').value = med.nombre;
    document.getElementById('edit_descripcion').value = med.descripcion || '';
    document.getElementById('edit_precio_unitario').value = med.precio_unitario;
    document.getElementById('edit_stock').value = med.stock;
    document.getElementById('edit_unidad_medida').value = med.unidad_medida;
    document.getElementById('edit_activo').checked = med.activo == 1;
    
    new bootstrap.Modal(document.getElementById('modalEditarMedicamento')).show();
}

function eliminarMedicamento(id, nombre) {
    if (confirm('¿Está seguro de eliminar el medicamento "' + nombre + '"?')) {
        document.getElementById('delete_id').value = id;
        document.getElementById('formEliminar').submit();
    }
}
</script>

<?php include '../includes/footer.php'; ?>
