<?php
/**
 * Gestión de Colaboradores
 * Sistema de Administración de Veterinaria
 */

require_once '../config/config.php';
requerir_rol('administrador');

$titulo_pagina = 'Gestión de Colaboradores';

// Procesar acciones
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $accion = $_POST['accion'] ?? '';
    
    try {
        $db = Database::getInstance()->getConnection();
        
        if ($accion === 'crear') {
            $usuario = limpiar_entrada($_POST['usuario']);
            $contrasena = $_POST['contrasena'];
            $nombre_completo = limpiar_entrada($_POST['nombre_completo']);
            $email = limpiar_entrada($_POST['email']);
            $telefono = limpiar_entrada($_POST['telefono']);
            $rol = limpiar_entrada($_POST['rol']);
            
            // Verificar si el usuario ya existe
            $stmt = $db->prepare("SELECT id FROM usuarios WHERE usuario = :usuario");
            $stmt->execute(['usuario' => $usuario]);
            
            if ($stmt->fetch()) {
                redirigir_con_mensaje('colaboradores.php', 'El usuario ya existe', 'danger');
            }
            
            $contrasena_hash = password_hash($contrasena, PASSWORD_DEFAULT);
            
            $stmt = $db->prepare("INSERT INTO usuarios (usuario, contrasena, nombre_completo, email, telefono, rol) 
                                  VALUES (:usuario, :contrasena, :nombre_completo, :email, :telefono, :rol)");
            $stmt->execute([
                'usuario' => $usuario,
                'contrasena' => $contrasena_hash,
                'nombre_completo' => $nombre_completo,
                'email' => $email,
                'telefono' => $telefono,
                'rol' => $rol
            ]);
            
            redirigir_con_mensaje('colaboradores.php', 'Colaborador creado exitosamente', 'success');
            
        } elseif ($accion === 'editar') {
            $id = (int)$_POST['id'];
            $usuario = limpiar_entrada($_POST['usuario']);
            $nombre_completo = limpiar_entrada($_POST['nombre_completo']);
            $email = limpiar_entrada($_POST['email']);
            $telefono = limpiar_entrada($_POST['telefono']);
            $rol = limpiar_entrada($_POST['rol']);
            $activo = isset($_POST['activo']) ? 1 : 0;
            
            $stmt = $db->prepare("UPDATE usuarios 
                                  SET usuario = :usuario, nombre_completo = :nombre_completo, 
                                      email = :email, telefono = :telefono, rol = :rol, activo = :activo
                                  WHERE id = :id");
            $stmt->execute([
                'usuario' => $usuario,
                'nombre_completo' => $nombre_completo,
                'email' => $email,
                'telefono' => $telefono,
                'rol' => $rol,
                'activo' => $activo,
                'id' => $id
            ]);
            
            // Si se proporcionó nueva contraseña
            if (!empty($_POST['nueva_contrasena'])) {
                $nueva_contrasena_hash = password_hash($_POST['nueva_contrasena'], PASSWORD_DEFAULT);
                $stmt = $db->prepare("UPDATE usuarios SET contrasena = :contrasena WHERE id = :id");
                $stmt->execute(['contrasena' => $nueva_contrasena_hash, 'id' => $id]);
            }
            
            redirigir_con_mensaje('colaboradores.php', 'Colaborador actualizado exitosamente', 'success');
            
        } elseif ($accion === 'eliminar') {
            $id = (int)$_POST['id'];
            
            // No permitir eliminar al administrador principal
            if ($id == 1) {
                redirigir_con_mensaje('colaboradores.php', 'No se puede eliminar al administrador principal', 'danger');
            }
            
            $stmt = $db->prepare("DELETE FROM usuarios WHERE id = :id");
            $stmt->execute(['id' => $id]);
            
            redirigir_con_mensaje('colaboradores.php', 'Colaborador eliminado exitosamente', 'success');
        }
        
    } catch (PDOException $e) {
        redirigir_con_mensaje('colaboradores.php', 'Error: ' . $e->getMessage(), 'danger');
    }
}

// Obtener lista de colaboradores
try {
    $db = Database::getInstance()->getConnection();
    $stmt = $db->query("SELECT * FROM usuarios ORDER BY fecha_creacion DESC");
    $colaboradores = $stmt->fetchAll();
} catch (PDOException $e) {
    die("Error en la base de datos: " . $e->getMessage());
}

include '../includes/header.php';
?>

<div class="row mb-4">
    <div class="col-md-6">
        <h1><i class="bi bi-people"></i> Gestión de Colaboradores</h1>
    </div>
    <div class="col-md-6 text-end">
        <button type="button" class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#modalNuevoColaborador">
            <i class="bi bi-plus-circle"></i> Nuevo Colaborador
        </button>
    </div>
</div>

<div class="card">
    <div class="card-body">
        <div class="table-responsive">
            <table class="table table-hover">
                <thead>
                    <tr>
                        <th>Usuario</th>
                        <th>Nombre Completo</th>
                        <th>Email</th>
                        <th>Teléfono</th>
                        <th>Rol</th>
                        <th>Estado</th>
                        <th>Fecha Creación</th>
                        <th>Acciones</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($colaboradores as $colab): ?>
                        <tr>
                            <td><?php echo htmlspecialchars($colab['usuario']); ?></td>
                            <td><?php echo htmlspecialchars($colab['nombre_completo']); ?></td>
                            <td><?php echo htmlspecialchars($colab['email'] ?? '-'); ?></td>
                            <td><?php echo htmlspecialchars($colab['telefono'] ?? '-'); ?></td>
                            <td>
                                <?php if ($colab['rol'] === 'administrador'): ?>
                                    <span class="badge bg-danger">Administrador</span>
                                <?php else: ?>
                                    <span class="badge bg-primary">Colaborador</span>
                                <?php endif; ?>
                            </td>
                            <td>
                                <?php if ($colab['activo']): ?>
                                    <span class="badge bg-success">Activo</span>
                                <?php else: ?>
                                    <span class="badge bg-secondary">Inactivo</span>
                                <?php endif; ?>
                            </td>
                            <td><?php echo formatear_fecha($colab['fecha_creacion'], true); ?></td>
                            <td>
                                <button type="button" class="btn btn-sm btn-warning" 
                                        onclick="editarColaborador(<?php echo htmlspecialchars(json_encode($colab)); ?>)">
                                    <i class="bi bi-pencil"></i>
                                </button>
                                <?php if ($colab['id'] != 1): ?>
                                    <button type="button" class="btn btn-sm btn-danger" 
                                            onclick="eliminarColaborador(<?php echo $colab['id']; ?>, '<?php echo htmlspecialchars($colab['nombre_completo']); ?>')">
                                        <i class="bi bi-trash"></i>
                                    </button>
                                <?php endif; ?>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
    </div>
</div>

<!-- Modal Nuevo Colaborador -->
<div class="modal fade" id="modalNuevoColaborador" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <form method="POST" action="">
                <input type="hidden" name="accion" value="crear">
                <div class="modal-header">
                    <h5 class="modal-title">Nuevo Colaborador</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <div class="mb-3">
                        <label class="form-label">Usuario *</label>
                        <input type="text" class="form-control" name="usuario" required>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Contraseña *</label>
                        <input type="password" class="form-control" name="contrasena" required>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Nombre Completo *</label>
                        <input type="text" class="form-control" name="nombre_completo" required>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Email</label>
                        <input type="email" class="form-control" name="email">
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Teléfono</label>
                        <input type="text" class="form-control" name="telefono" placeholder="+595...">
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Rol *</label>
                        <select class="form-select" name="rol" required>
                            <option value="colaborador">Colaborador</option>
                            <option value="administrador">Administrador</option>
                        </select>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancelar</button>
                    <button type="submit" class="btn btn-primary">Guardar</button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Modal Editar Colaborador -->
<div class="modal fade" id="modalEditarColaborador" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <form method="POST" action="">
                <input type="hidden" name="accion" value="editar">
                <input type="hidden" name="id" id="edit_id">
                <div class="modal-header">
                    <h5 class="modal-title">Editar Colaborador</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <div class="mb-3">
                        <label class="form-label">Usuario *</label>
                        <input type="text" class="form-control" name="usuario" id="edit_usuario" required>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Nueva Contraseña (dejar en blanco para no cambiar)</label>
                        <input type="password" class="form-control" name="nueva_contrasena">
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Nombre Completo *</label>
                        <input type="text" class="form-control" name="nombre_completo" id="edit_nombre_completo" required>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Email</label>
                        <input type="email" class="form-control" name="email" id="edit_email">
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Teléfono</label>
                        <input type="text" class="form-control" name="telefono" id="edit_telefono">
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Rol *</label>
                        <select class="form-select" name="rol" id="edit_rol" required>
                            <option value="colaborador">Colaborador</option>
                            <option value="administrador">Administrador</option>
                        </select>
                    </div>
                    <div class="mb-3">
                        <div class="form-check">
                            <input class="form-check-input" type="checkbox" name="activo" id="edit_activo">
                            <label class="form-check-label" for="edit_activo">Activo</label>
                        </div>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancelar</button>
                    <button type="submit" class="btn btn-primary">Actualizar</button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Formulario oculto para eliminar -->
<form method="POST" action="" id="formEliminar" style="display: none;">
    <input type="hidden" name="accion" value="eliminar">
    <input type="hidden" name="id" id="delete_id">
</form>

<script>
function editarColaborador(colab) {
    document.getElementById('edit_id').value = colab.id;
    document.getElementById('edit_usuario').value = colab.usuario;
    document.getElementById('edit_nombre_completo').value = colab.nombre_completo;
    document.getElementById('edit_email').value = colab.email || '';
    document.getElementById('edit_telefono').value = colab.telefono || '';
    document.getElementById('edit_rol').value = colab.rol;
    document.getElementById('edit_activo').checked = colab.activo == 1;
    
    new bootstrap.Modal(document.getElementById('modalEditarColaborador')).show();
}

function eliminarColaborador(id, nombre) {
    if (confirm('¿Está seguro de eliminar al colaborador "' + nombre + '"?')) {
        document.getElementById('delete_id').value = id;
        document.getElementById('formEliminar').submit();
    }
}
</script>

<?php include '../includes/footer.php'; ?>
