<?php
/**
 * API: Agregar producto al carrito
 */

require_once '../../config/database.php';
require_once '../../config/functions.php';

header('Content-Type: application/json; charset=utf-8');

iniciarSesion();

if (!estaAutenticadoCliente()) {
    responderJSON(['success' => false, 'message' => 'No autenticado'], 401);
}

$input = json_decode(file_get_contents('php://input'), true);
$compraId = $input['compra_id'] ?? 0;
$productoId = $input['producto_id'] ?? 0;
$cantidad = $input['cantidad'] ?? 1;

if (empty($compraId) || empty($productoId) || $cantidad < 1) {
    responderJSON(['success' => false, 'message' => 'Datos inválidos'], 400);
}

try {
    $db = getDB();
    
    // Verificar que la compra pertenece al cliente
    $cliente = obtenerClienteActual();
    $stmt = $db->prepare("SELECT id FROM compras WHERE id = ? AND cliente_id = ? AND estado = 'activa'");
    $stmt->execute([$compraId, $cliente['id']]);
    if (!$stmt->fetch()) {
        responderJSON(['success' => false, 'message' => 'Compra no válida'], 403);
    }
    
    // Obtener información del producto
    $stmt = $db->prepare("SELECT precio FROM productos WHERE id = ? AND activo = 1");
    $stmt->execute([$productoId]);
    $producto = $stmt->fetch();
    
    if (!$producto) {
        responderJSON(['success' => false, 'message' => 'Producto no encontrado'], 404);
    }
    
    // Verificar si el producto ya está en el carrito
    $stmt = $db->prepare("SELECT id, cantidad FROM detalle_compras WHERE compra_id = ? AND producto_id = ?");
    $stmt->execute([$compraId, $productoId]);
    $itemExistente = $stmt->fetch();
    
    if ($itemExistente) {
        // Actualizar cantidad
        $nuevaCantidad = $itemExistente['cantidad'] + $cantidad;
        $subtotal = $nuevaCantidad * $producto['precio'];
        
        $stmt = $db->prepare("
            UPDATE detalle_compras 
            SET cantidad = ?, subtotal = ?
            WHERE id = ?
        ");
        $stmt->execute([$nuevaCantidad, $subtotal, $itemExistente['id']]);
    } else {
        // Insertar nuevo item
        $subtotal = $cantidad * $producto['precio'];
        
        $stmt = $db->prepare("
            INSERT INTO detalle_compras (compra_id, producto_id, cantidad, precio_unitario, subtotal)
            VALUES (?, ?, ?, ?, ?)
        ");
        $stmt->execute([$compraId, $productoId, $cantidad, $producto['precio'], $subtotal]);
    }
    
    // Actualizar total de la compra
    $stmt = $db->prepare("
        UPDATE compras 
        SET total = (
            SELECT COALESCE(SUM(subtotal), 0) 
            FROM detalle_compras 
            WHERE compra_id = ?
        )
        WHERE id = ?
    ");
    $stmt->execute([$compraId, $compraId]);
    
    responderJSON([
        'success' => true,
        'message' => 'Producto agregado al carrito'
    ]);
    
} catch (PDOException $e) {
    responderJSON([
        'success' => false,
        'message' => 'Error al agregar producto'
    ], 500);
}
?>
