<?php
/**
 * Gestión de usuarios administrativos
 * Sistema de Supermercados y Tiendas V3.0
 */

require_once '../config/database.php';
require_once '../config/functions.php';

iniciarSesion();

if (!estaAutenticadoAdmin()) {
    redirigir('login.php');
}

$admin = obtenerAdminActual();

// Solo administradores pueden gestionar usuarios
if ($admin['rol'] !== 'administrador') {
    redirigir('dashboard.php');
}

$db = getDB();
$stmt = $db->query("SELECT * FROM usuarios_admin ORDER BY fecha_creacion DESC");
$usuarios = $stmt->fetchAll();
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Usuarios Admin - Panel de Administración</title>
    <link rel="stylesheet" href="../assets/css/style.css">
    <link rel="stylesheet" href="../assets/css/admin.css">
</head>
<body class="admin-page">
    <?php include 'includes/header.php'; ?>
    <?php include 'includes/sidebar.php'; ?>
    
    <main class="admin-content">
        <div class="content-header">
            <h1>Gestión de Usuarios Administrativos</h1>
            <button class="btn btn-primary" onclick="abrirModalUsuario()">
                ➕ Nuevo Usuario
            </button>
        </div>
        
        <div class="card">
            <div class="card-body">
                <div class="table-responsive">
                    <table class="data-table">
                        <thead>
                            <tr>
                                <th>Usuario</th>
                                <th>Nombre</th>
                                <th>Rol</th>
                                <th>Estado</th>
                                <th>Fecha Creación</th>
                                <th>Acciones</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($usuarios as $usuario): ?>
                                <tr>
                                    <td><strong><?php echo htmlspecialchars($usuario['username']); ?></strong></td>
                                    <td><?php echo htmlspecialchars($usuario['nombre']); ?></td>
                                    <td>
                                        <span class="badge badge-info">
                                            <?php echo ucfirst($usuario['rol']); ?>
                                        </span>
                                    </td>
                                    <td>
                                        <span class="badge badge-<?php echo $usuario['activo'] ? 'success' : 'secondary'; ?>">
                                            <?php echo $usuario['activo'] ? 'Activo' : 'Inactivo'; ?>
                                        </span>
                                    </td>
                                    <td><?php echo date('d/m/Y', strtotime($usuario['fecha_creacion'])); ?></td>
                                    <td class="actions-cell">
                                        <button class="btn btn-sm btn-warning" onclick='editarUsuario(<?php echo json_encode($usuario); ?>)'>✏️</button>
                                        <?php if ($usuario['id'] != $admin['id']): ?>
                                            <button class="btn btn-sm btn-danger" onclick="eliminarUsuario(<?php echo $usuario['id']; ?>)">🗑️</button>
                                        <?php endif; ?>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </main>
    
    <div id="modalUsuario" class="modal">
        <div class="modal-content">
            <span class="modal-close" onclick="cerrarModalUsuario()">&times;</span>
            <h2 id="modalTitulo">Nuevo Usuario Administrativo</h2>
            
            <form id="formUsuario" onsubmit="guardarUsuario(event)">
                <input type="hidden" id="usuario_id" name="id">
                
                <div class="form-group">
                    <label for="username">Usuario *</label>
                    <input type="text" id="username" name="username" class="form-control" required>
                </div>
                
                <div class="form-group">
                    <label for="nombre">Nombre Completo *</label>
                    <input type="text" id="nombre" name="nombre" class="form-control" required>
                </div>
                
                <div class="form-group">
                    <label for="password">Contraseña *</label>
                    <input type="password" id="password" name="password" class="form-control">
                    <small class="form-text">Dejar en blanco para no cambiar (solo al editar)</small>
                </div>
                
                <div class="form-group">
                    <label for="rol">Rol *</label>
                    <select id="rol" name="rol" class="form-control" required>
                        <option value="administrador">Administrador</option>
                        <option value="comercial">Comercial</option>
                        <option value="colaborador">Colaborador</option>
                    </select>
                </div>
                
                <div class="form-group">
                    <label class="checkbox-label">
                        <input type="checkbox" id="activo" name="activo" checked>
                        Usuario activo
                    </label>
                </div>
                
                <div class="form-actions">
                    <button type="button" class="btn btn-secondary" onclick="cerrarModalUsuario()">Cancelar</button>
                    <button type="submit" class="btn btn-primary">Guardar</button>
                </div>
            </form>
        </div>
    </div>
    
    <script>
        function abrirModalUsuario() {
            document.getElementById('modalTitulo').textContent = 'Nuevo Usuario Administrativo';
            document.getElementById('formUsuario').reset();
            document.getElementById('usuario_id').value = '';
            document.getElementById('password').required = true;
            document.getElementById('modalUsuario').style.display = 'block';
        }
        
        function cerrarModalUsuario() {
            document.getElementById('modalUsuario').style.display = 'none';
        }
        
        function editarUsuario(usuario) {
            document.getElementById('modalTitulo').textContent = 'Editar Usuario';
            document.getElementById('usuario_id').value = usuario.id;
            document.getElementById('username').value = usuario.username;
            document.getElementById('nombre').value = usuario.nombre;
            document.getElementById('rol').value = usuario.rol;
            document.getElementById('activo').checked = usuario.activo == 1;
            document.getElementById('password').required = false;
            document.getElementById('password').value = '';
            document.getElementById('modalUsuario').style.display = 'block';
        }
        
        function guardarUsuario(event) {
            event.preventDefault();
            
            const formData = new FormData(document.getElementById('formUsuario'));
            formData.append('activo', document.getElementById('activo').checked ? 1 : 0);
            
            fetch('api/guardar_usuario.php', {
                method: 'POST',
                body: formData
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    alert('Usuario guardado exitosamente');
                    location.reload();
                } else {
                    alert('Error: ' + (data.message || 'No se pudo guardar'));
                }
            });
        }
        
        function eliminarUsuario(id) {
            if (confirm('¿Está seguro de eliminar este usuario?')) {
                fetch('api/eliminar_usuario.php', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify({ id: id })
                })
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        alert('Usuario eliminado');
                        location.reload();
                    }
                });
            }
        }
    </script>
    <script src="../assets/js/admin.js"></script>
</body>
</html>
