<?php
/**
 * Reportes y estadísticas
 * Sistema de Supermercados y Tiendas V3.0
 */

require_once '../config/database.php';
require_once '../config/functions.php';

iniciarSesion();

if (!estaAutenticadoAdmin()) {
    redirigir('login.php');
}

$admin = obtenerAdminActual();
$db = getDB();

// Productos más vendidos
$stmt = $db->query("
    SELECT p.nombre, p.codigo_barras, SUM(dc.cantidad) as total_vendido, SUM(dc.subtotal) as total_ingresos
    FROM detalle_compras dc
    JOIN productos p ON dc.producto_id = p.id
    JOIN compras c ON dc.compra_id = c.id
    WHERE c.estado = 'finalizada'
    GROUP BY p.id
    ORDER BY total_vendido DESC
    LIMIT 10
");
$productosMasVendidos = $stmt->fetchAll();

// Ventas por día (últimos 30 días)
$stmt = $db->query("
    SELECT DATE(fecha_compra) as fecha, COUNT(*) as num_compras, SUM(total) as total_ventas
    FROM compras
    WHERE estado = 'finalizada' AND fecha_compra >= DATE_SUB(NOW(), INTERVAL 30 DAY)
    GROUP BY DATE(fecha_compra)
    ORDER BY fecha DESC
");
$ventasPorDia = $stmt->fetchAll();

// Clientes más frecuentes
$stmt = $db->query("
    SELECT cl.nombre, cl.apellido, cl.cedula_ruc, COUNT(c.id) as num_compras, SUM(c.total) as total_gastado
    FROM clientes cl
    JOIN compras c ON cl.id = c.cliente_id
    WHERE c.estado = 'finalizada'
    GROUP BY cl.id
    ORDER BY num_compras DESC
    LIMIT 10
");
$clientesFrecuentes = $stmt->fetchAll();
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Reportes - Panel de Administración</title>
    <link rel="stylesheet" href="../assets/css/style.css">
    <link rel="stylesheet" href="../assets/css/admin.css">
</head>
<body class="admin-page">
    <?php include 'includes/header.php'; ?>
    <?php include 'includes/sidebar.php'; ?>
    
    <main class="admin-content">
        <div class="content-header">
            <h1>Reportes y Estadísticas</h1>
        </div>
        
        <!-- Productos más vendidos -->
        <div class="card">
            <div class="card-header">
                <h2>Top 10 Productos Más Vendidos</h2>
            </div>
            <div class="card-body">
                <div class="table-responsive">
                    <table class="data-table">
                        <thead>
                            <tr>
                                <th>#</th>
                                <th>Código</th>
                                <th>Producto</th>
                                <th>Unidades Vendidas</th>
                                <th>Ingresos Totales</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($productosMasVendidos as $index => $prod): ?>
                                <tr>
                                    <td><?php echo $index + 1; ?></td>
                                    <td><?php echo htmlspecialchars($prod['codigo_barras']); ?></td>
                                    <td><strong><?php echo htmlspecialchars($prod['nombre']); ?></strong></td>
                                    <td><?php echo $prod['total_vendido']; ?></td>
                                    <td><?php echo formatearGuaranies($prod['total_ingresos']); ?></td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
        
        <!-- Ventas por día -->
        <div class="card">
            <div class="card-header">
                <h2>Ventas de los Últimos 30 Días</h2>
            </div>
            <div class="card-body">
                <div class="table-responsive">
                    <table class="data-table">
                        <thead>
                            <tr>
                                <th>Fecha</th>
                                <th>Número de Compras</th>
                                <th>Total Ventas</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($ventasPorDia as $venta): ?>
                                <tr>
                                    <td><?php echo date('d/m/Y', strtotime($venta['fecha'])); ?></td>
                                    <td><?php echo $venta['num_compras']; ?></td>
                                    <td><?php echo formatearGuaranies($venta['total_ventas']); ?></td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
        
        <!-- Clientes más frecuentes -->
        <div class="card">
            <div class="card-header">
                <h2>Top 10 Clientes Más Frecuentes</h2>
            </div>
            <div class="card-body">
                <div class="table-responsive">
                    <table class="data-table">
                        <thead>
                            <tr>
                                <th>#</th>
                                <th>Cédula/RUC</th>
                                <th>Cliente</th>
                                <th>Compras Realizadas</th>
                                <th>Total Gastado</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($clientesFrecuentes as $index => $cliente): ?>
                                <tr>
                                    <td><?php echo $index + 1; ?></td>
                                    <td><?php echo htmlspecialchars($cliente['cedula_ruc']); ?></td>
                                    <td><strong><?php echo htmlspecialchars($cliente['nombre'] . ' ' . $cliente['apellido']); ?></strong></td>
                                    <td><?php echo $cliente['num_compras']; ?></td>
                                    <td><?php echo formatearGuaranies($cliente['total_gastado']); ?></td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </main>
    
    <script src="../assets/js/admin.js"></script>
</body>
</html>
