<?php
/**
 * Gestión de mensajes emergentes
 * Sistema de Supermercados y Tiendas V3.0
 */

require_once '../config/database.php';
require_once '../config/functions.php';

iniciarSesion();

if (!estaAutenticadoAdmin()) {
    redirigir('login.php');
}

if (!tienePermiso('enviar_mensajes')) {
    redirigir('dashboard.php');
}

$admin = obtenerAdminActual();
$db = getDB();

$stmt = $db->query("
    SELECT m.*, u.nombre as creado_por_nombre
    FROM mensajes_emergentes m
    LEFT JOIN usuarios_admin u ON m.creado_por = u.id
    ORDER BY m.fecha_creacion DESC
    LIMIT 50
");
$mensajes = $stmt->fetchAll();
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Mensajes Emergentes - Panel de Administración</title>
    <link rel="stylesheet" href="../assets/css/style.css">
    <link rel="stylesheet" href="../assets/css/admin.css">
</head>
<body class="admin-page">
    <?php include 'includes/header.php'; ?>
    <?php include 'includes/sidebar.php'; ?>
    
    <main class="admin-content">
        <div class="content-header">
            <h1>Mensajes Emergentes</h1>
            <button class="btn btn-primary" onclick="abrirModalMensaje()">
                ➕ Nuevo Mensaje
            </button>
        </div>
        
        <div class="alert alert-info">
            Los mensajes emergentes se mostrarán a los clientes cuando accedan al sistema.
        </div>
        
        <div class="card">
            <div class="card-body">
                <div class="table-responsive">
                    <table class="data-table">
                        <thead>
                            <tr>
                                <th>Mensaje</th>
                                <th>Tipo</th>
                                <th>Fecha Expiración</th>
                                <th>Estado</th>
                                <th>Creado por</th>
                                <th>Acciones</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($mensajes as $msg): ?>
                                <tr>
                                    <td><?php echo htmlspecialchars($msg['mensaje']); ?></td>
                                    <td>
                                        <span class="badge badge-info">
                                            <?php echo htmlspecialchars($msg['tipo']); ?>
                                        </span>
                                    </td>
                                    <td>
                                        <?php 
                                        if ($msg['fecha_expiracion']) {
                                            echo date('d/m/Y H:i', strtotime($msg['fecha_expiracion']));
                                        } else {
                                            echo 'Sin expiración';
                                        }
                                        ?>
                                    </td>
                                    <td>
                                        <span class="badge badge-<?php echo $msg['activo'] ? 'success' : 'secondary'; ?>">
                                            <?php echo $msg['activo'] ? 'Activo' : 'Inactivo'; ?>
                                        </span>
                                    </td>
                                    <td><?php echo htmlspecialchars($msg['creado_por_nombre'] ?? 'N/A'); ?></td>
                                    <td class="actions-cell">
                                        <button class="btn btn-sm btn-danger" onclick="eliminarMensaje(<?php echo $msg['id']; ?>)">🗑️</button>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </main>
    
    <div id="modalMensaje" class="modal">
        <div class="modal-content">
            <span class="modal-close" onclick="cerrarModalMensaje()">&times;</span>
            <h2>Nuevo Mensaje Emergente</h2>
            
            <form id="formMensaje" onsubmit="guardarMensaje(event)">
                <div class="form-group">
                    <label for="mensaje">Mensaje *</label>
                    <textarea id="mensaje" name="mensaje" class="form-control" rows="4" required></textarea>
                </div>
                
                <div class="form-group">
                    <label for="tipo">Tipo</label>
                    <select id="tipo" name="tipo" class="form-control">
                        <option value="oferta">Oferta</option>
                        <option value="promocion">Promoción</option>
                        <option value="anuncio">Anuncio</option>
                        <option value="alerta">Alerta</option>
                    </select>
                </div>
                
                <div class="form-group">
                    <label for="fecha_expiracion">Fecha de Expiración (opcional)</label>
                    <input type="datetime-local" id="fecha_expiracion" name="fecha_expiracion" class="form-control">
                </div>
                
                <div class="form-group">
                    <label class="checkbox-label">
                        <input type="checkbox" id="activo" name="activo" checked>
                        Mensaje activo
                    </label>
                </div>
                
                <div class="form-actions">
                    <button type="button" class="btn btn-secondary" onclick="cerrarModalMensaje()">Cancelar</button>
                    <button type="submit" class="btn btn-primary">Enviar Mensaje</button>
                </div>
            </form>
        </div>
    </div>
    
    <script>
        function abrirModalMensaje() {
            document.getElementById('formMensaje').reset();
            document.getElementById('modalMensaje').style.display = 'block';
        }
        
        function cerrarModalMensaje() {
            document.getElementById('modalMensaje').style.display = 'none';
        }
        
        function guardarMensaje(event) {
            event.preventDefault();
            
            const formData = new FormData(document.getElementById('formMensaje'));
            formData.append('activo', document.getElementById('activo').checked ? 1 : 0);
            
            fetch('api/guardar_mensaje.php', {
                method: 'POST',
                body: formData
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    alert('Mensaje enviado exitosamente');
                    location.reload();
                } else {
                    alert('Error: ' + (data.message || 'No se pudo enviar'));
                }
            });
        }
        
        function eliminarMensaje(id) {
            if (confirm('¿Está seguro de eliminar este mensaje?')) {
                fetch('api/eliminar_mensaje.php', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify({ id: id })
                })
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        alert('Mensaje eliminado');
                        location.reload();
                    }
                });
            }
        }
    </script>
    <script src="../assets/js/admin.js"></script>
</body>
</html>
