<?php
/**
 * Dashboard principal del panel de administración
 * Sistema de Supermercados y Tiendas V3.0
 */

require_once '../config/database.php';
require_once '../config/functions.php';

iniciarSesion();

if (!estaAutenticadoAdmin()) {
    redirigir('login.php');
}

$admin = obtenerAdminActual();
$db = getDB();

// Obtener estadísticas generales
$stmt = $db->query("SELECT COUNT(*) as total FROM clientes");
$totalClientes = $stmt->fetch()['total'];

$stmt = $db->query("SELECT COUNT(*) as total FROM productos WHERE activo = 1");
$totalProductos = $stmt->fetch()['total'];

$stmt = $db->query("SELECT COUNT(*) as total FROM compras WHERE estado = 'finalizada'");
$totalCompras = $stmt->fetch()['total'];

$stmt = $db->query("SELECT COALESCE(SUM(total), 0) as total FROM compras WHERE estado = 'finalizada'");
$ventasTotal = $stmt->fetch()['total'];

// Ventas del día
$stmt = $db->query("
    SELECT COALESCE(SUM(total), 0) as total 
    FROM compras 
    WHERE estado = 'finalizada' 
    AND DATE(fecha_compra) = CURDATE()
");
$ventasHoy = $stmt->fetch()['total'];

// Ventas del mes
$stmt = $db->query("
    SELECT COALESCE(SUM(total), 0) as total 
    FROM compras 
    WHERE estado = 'finalizada' 
    AND MONTH(fecha_compra) = MONTH(CURDATE())
    AND YEAR(fecha_compra) = YEAR(CURDATE())
");
$ventasMes = $stmt->fetch()['total'];

// Últimas compras
$stmt = $db->query("
    SELECT 
        c.id,
        c.fecha_compra,
        c.total,
        cl.nombre,
        cl.apellido,
        COUNT(dc.id) as total_items
    FROM compras c
    INNER JOIN clientes cl ON c.cliente_id = cl.id
    LEFT JOIN detalle_compras dc ON c.id = dc.compra_id
    WHERE c.estado = 'finalizada'
    GROUP BY c.id
    ORDER BY c.fecha_compra DESC
    LIMIT 10
");
$ultimasCompras = $stmt->fetchAll();

// Productos más vendidos
$stmt = $db->query("
    SELECT 
        p.nombre,
        SUM(dc.cantidad) as total_vendido,
        SUM(dc.subtotal) as ingresos
    FROM detalle_compras dc
    INNER JOIN productos p ON dc.producto_id = p.id
    INNER JOIN compras c ON dc.compra_id = c.id
    WHERE c.estado = 'finalizada'
    GROUP BY p.id
    ORDER BY total_vendido DESC
    LIMIT 5
");
$productosMasVendidos = $stmt->fetchAll();

// Ventas por día (últimos 7 días)
$stmt = $db->query("
    SELECT 
        DATE(fecha_compra) as fecha,
        COUNT(*) as cantidad,
        SUM(total) as monto
    FROM compras
    WHERE estado = 'finalizada'
    AND fecha_compra >= DATE_SUB(CURDATE(), INTERVAL 7 DAY)
    GROUP BY DATE(fecha_compra)
    ORDER BY fecha ASC
");
$ventasPorDia = $stmt->fetchAll();
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Dashboard - Panel de Administración</title>
    <link rel="stylesheet" href="../assets/css/style.css">
    <link rel="stylesheet" href="../assets/css/admin.css">
    <script src="https://cdn.jsdelivr.net/npm/chart.js@4.4.0/dist/chart.umd.min.js"></script>
</head>
<body class="admin-page">
    <?php include 'includes/header.php'; ?>
    <?php include 'includes/sidebar.php'; ?>
    
    <main class="admin-content">
        <div class="content-header">
            <h1>Dashboard</h1>
            <p>Bienvenido, <strong><?php echo htmlspecialchars($admin['nombre']); ?></strong> (<?php echo ucfirst($admin['rol']); ?>)</p>
        </div>
        
        <!-- Tarjetas de estadísticas -->
        <div class="stats-grid">
            <div class="stat-card stat-primary">
                <div class="stat-icon">👥</div>
                <div class="stat-info">
                    <h3><?php echo number_format($totalClientes); ?></h3>
                    <p>Clientes Registrados</p>
                </div>
            </div>
            
            <div class="stat-card stat-success">
                <div class="stat-icon">📦</div>
                <div class="stat-info">
                    <h3><?php echo number_format($totalProductos); ?></h3>
                    <p>Productos Activos</p>
                </div>
            </div>
            
            <div class="stat-card stat-info">
                <div class="stat-icon">🛒</div>
                <div class="stat-info">
                    <h3><?php echo number_format($totalCompras); ?></h3>
                    <p>Compras Realizadas</p>
                </div>
            </div>
            
            <div class="stat-card stat-warning">
                <div class="stat-icon">💰</div>
                <div class="stat-info">
                    <h3><?php echo formatearGuaranies($ventasTotal); ?></h3>
                    <p>Ventas Totales</p>
                </div>
            </div>
        </div>
        
        <!-- Ventas del período -->
        <div class="cards-row">
            <div class="card">
                <div class="card-header">
                    <h2>Ventas de Hoy</h2>
                </div>
                <div class="card-body text-center">
                    <div class="big-number"><?php echo formatearGuaranies($ventasHoy); ?></div>
                </div>
            </div>
            
            <div class="card">
                <div class="card-header">
                    <h2>Ventas del Mes</h2>
                </div>
                <div class="card-body text-center">
                    <div class="big-number"><?php echo formatearGuaranies($ventasMes); ?></div>
                </div>
            </div>
        </div>
        
        <!-- Gráficos -->
        <div class="charts-row">
            <div class="card chart-card">
                <div class="card-header">
                    <h2>Ventas de los Últimos 7 Días</h2>
                </div>
                <div class="card-body">
                    <canvas id="chartVentasDias"></canvas>
                </div>
            </div>
            
            <div class="card chart-card">
                <div class="card-header">
                    <h2>Productos Más Vendidos</h2>
                </div>
                <div class="card-body">
                    <canvas id="chartProductos"></canvas>
                </div>
            </div>
        </div>
        
        <!-- Últimas compras -->
        <div class="card">
            <div class="card-header">
                <h2>Últimas Compras</h2>
            </div>
            <div class="card-body">
                <div class="table-responsive">
                    <table class="data-table">
                        <thead>
                            <tr>
                                <th>ID</th>
                                <th>Cliente</th>
                                <th>Fecha</th>
                                <th>Items</th>
                                <th>Total</th>
                                <th>Acciones</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php if (empty($ultimasCompras)): ?>
                                <tr>
                                    <td colspan="6" class="text-center">No hay compras registradas</td>
                                </tr>
                            <?php else: ?>
                                <?php foreach ($ultimasCompras as $compra): ?>
                                    <tr>
                                        <td>#<?php echo str_pad($compra['id'], 6, '0', STR_PAD_LEFT); ?></td>
                                        <td><?php echo htmlspecialchars($compra['nombre'] . ' ' . $compra['apellido']); ?></td>
                                        <td><?php echo date('d/m/Y H:i', strtotime($compra['fecha_compra'])); ?></td>
                                        <td><?php echo $compra['total_items']; ?></td>
                                        <td><?php echo formatearGuaranies($compra['total']); ?></td>
                                        <td>
                                            <a href="ver_compra.php?id=<?php echo $compra['id']; ?>" class="btn btn-sm btn-info">Ver</a>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </main>
    
    <script>
        // Gráfico de ventas por día
        const ctxVentas = document.getElementById('chartVentasDias').getContext('2d');
        const ventasPorDia = <?php echo json_encode($ventasPorDia); ?>;
        
        new Chart(ctxVentas, {
            type: 'line',
            data: {
                labels: ventasPorDia.map(v => {
                    const fecha = new Date(v.fecha);
                    return fecha.toLocaleDateString('es-PY', { day: '2-digit', month: '2-digit' });
                }),
                datasets: [{
                    label: 'Ventas (₲)',
                    data: ventasPorDia.map(v => v.monto),
                    borderColor: 'rgb(75, 192, 192)',
                    backgroundColor: 'rgba(75, 192, 192, 0.2)',
                    tension: 0.1
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: {
                        display: true
                    }
                },
                scales: {
                    y: {
                        beginAtZero: true,
                        ticks: {
                            callback: function(value) {
                                return '₲ ' + value.toLocaleString('es-PY');
                            }
                        }
                    }
                }
            }
        });
        
        // Gráfico de productos más vendidos
        const ctxProductos = document.getElementById('chartProductos').getContext('2d');
        const productosMasVendidos = <?php echo json_encode($productosMasVendidos); ?>;
        
        new Chart(ctxProductos, {
            type: 'bar',
            data: {
                labels: productosMasVendidos.map(p => p.nombre),
                datasets: [{
                    label: 'Cantidad Vendida',
                    data: productosMasVendidos.map(p => p.total_vendido),
                    backgroundColor: [
                        'rgba(255, 99, 132, 0.7)',
                        'rgba(54, 162, 235, 0.7)',
                        'rgba(255, 206, 86, 0.7)',
                        'rgba(75, 192, 192, 0.7)',
                        'rgba(153, 102, 255, 0.7)'
                    ],
                    borderColor: [
                        'rgba(255, 99, 132, 1)',
                        'rgba(54, 162, 235, 1)',
                        'rgba(255, 206, 86, 1)',
                        'rgba(75, 192, 192, 1)',
                        'rgba(153, 102, 255, 1)'
                    ],
                    borderWidth: 1
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: {
                        display: false
                    }
                },
                scales: {
                    y: {
                        beginAtZero: true
                    }
                }
            }
        });
    </script>
    <script src="../assets/js/admin.js"></script>
</body>
</html>
