<?php
/**
 * API: Guardar producto (crear o actualizar)
 */

require_once '../../config/database.php';
require_once '../../config/functions.php';

header('Content-Type: application/json; charset=utf-8');

iniciarSesion();

if (!estaAutenticadoAdmin()) {
    responderJSON(['success' => false, 'message' => 'No autenticado'], 401);
}

$admin = obtenerAdminActual();

// Validar permisos
if (!tienePermiso('crear') && !tienePermiso('modificar')) {
    responderJSON(['success' => false, 'message' => 'No tiene permisos'], 403);
}

$id = $_POST['id'] ?? null;
$codigoBarras = sanitizar($_POST['codigo_barras'] ?? '');
$nombre = sanitizar($_POST['nombre'] ?? '');
$descripcion = sanitizar($_POST['descripcion'] ?? '');
$precio = $_POST['precio'] ?? 0;
$stock = $_POST['stock'] ?? 0;
$categoria = sanitizar($_POST['categoria'] ?? '');
$activo = $_POST['activo'] ?? 1;

// Validar campos requeridos
if (empty($codigoBarras) || empty($nombre) || $precio <= 0) {
    responderJSON(['success' => false, 'message' => 'Campos requeridos incompletos'], 400);
}

try {
    $db = getDB();
    
    // Manejar imagen si se subió
    $rutaImagen = null;
    if (isset($_FILES['imagen']) && $_FILES['imagen']['error'] === UPLOAD_ERR_OK) {
        $resultado = subirImagen($_FILES['imagen'], 'uploads/productos/');
        if (isset($resultado['success'])) {
            $rutaImagen = $resultado['ruta'];
        }
    }
    
    if ($id) {
        // Actualizar producto existente
        
        // Verificar permisos para editar
        $stmt = $db->prepare("SELECT creado_por FROM productos WHERE id = ?");
        $stmt->execute([$id]);
        $producto = $stmt->fetch();
        
        if (!$producto) {
            responderJSON(['success' => false, 'message' => 'Producto no encontrado'], 404);
        }
        
        // Colaboradores solo pueden editar sus propios productos
        if ($admin['rol'] === 'colaborador' && $producto['creado_por'] != $admin['id']) {
            responderJSON(['success' => false, 'message' => 'No puede modificar productos de otros usuarios'], 403);
        }
        
        $sql = "UPDATE productos SET 
                codigo_barras = ?, 
                nombre = ?, 
                descripcion = ?, 
                precio = ?, 
                stock = ?, 
                categoria = ?, 
                activo = ?";
        
        $params = [$codigoBarras, $nombre, $descripcion, $precio, $stock, $categoria, $activo];
        
        if ($rutaImagen) {
            $sql .= ", imagen = ?";
            $params[] = $rutaImagen;
        }
        
        $sql .= " WHERE id = ?";
        $params[] = $id;
        
        $stmt = $db->prepare($sql);
        $stmt->execute($params);
        
        responderJSON(['success' => true, 'message' => 'Producto actualizado']);
        
    } else {
        // Crear nuevo producto
        
        if (!tienePermiso('crear')) {
            responderJSON(['success' => false, 'message' => 'No tiene permisos para crear productos'], 403);
        }
        
        // Verificar que el código de barras no exista
        $stmt = $db->prepare("SELECT id FROM productos WHERE codigo_barras = ?");
        $stmt->execute([$codigoBarras]);
        if ($stmt->fetch()) {
            responderJSON(['success' => false, 'message' => 'El código de barras ya existe'], 400);
        }
        
        $stmt = $db->prepare("
            INSERT INTO productos (codigo_barras, nombre, descripcion, precio, stock, imagen, categoria, activo, creado_por)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)
        ");
        
        $stmt->execute([
            $codigoBarras, 
            $nombre, 
            $descripcion, 
            $precio, 
            $stock, 
            $rutaImagen, 
            $categoria, 
            $activo, 
            $admin['id']
        ]);
        
        responderJSON(['success' => true, 'message' => 'Producto creado']);
    }
    
} catch (PDOException $e) {
    responderJSON(['success' => false, 'message' => 'Error al guardar producto: ' . $e->getMessage()], 500);
}
?>
