<?php
require_once '../includes/config.php';

$error = '';
$success = '';
$huesped_data = null;

// Buscar huésped por cédula (AJAX)
if (isset($_GET['buscar_cedula']) && !empty($_GET['buscar_cedula'])) {
    $db = getDBConnection();
    $cedula = sanitize($_GET['buscar_cedula']);
    
    $stmt = $db->prepare("SELECT * FROM usuarios WHERE cedula_pasaporte = ? AND rol = 'huesped'");
    $stmt->execute([$cedula]);
    $huesped = $stmt->fetch();
    
    header('Content-Type: application/json');
    echo json_encode($huesped ?: ['not_found' => true]);
    exit;
}

// Procesar formulario de registro
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['registrar'])) {
    $cedula = sanitize($_POST['cedula_pasaporte']);
    $nombre = sanitize($_POST['nombre_completo']);
    $telefono = sanitize($_POST['telefono']);
    $email = sanitize($_POST['email'] ?? '');
    
    if (empty($cedula) || empty($nombre) || empty($telefono)) {
        $error = 'Por favor complete todos los campos obligatorios';
    } else {
        $db = getDBConnection();
        
        // Verificar si ya existe
        $stmt = $db->prepare("SELECT id FROM usuarios WHERE cedula_pasaporte = ?");
        $stmt->execute([$cedula]);
        $existe = $stmt->fetch();
        
        if ($existe) {
            // Actualizar datos
            $stmt = $db->prepare("UPDATE usuarios SET nombre_completo = ?, telefono = ?, email = ? WHERE cedula_pasaporte = ?");
            $stmt->execute([$nombre, $telefono, $email, $cedula]);
            $huesped_id = $existe['id'];
        } else {
            // Insertar nuevo huésped
            $stmt = $db->prepare("INSERT INTO usuarios (cedula_pasaporte, nombre_completo, telefono, email, rol, habilitado) VALUES (?, ?, ?, ?, 'huesped', 1)");
            $stmt->execute([$cedula, $nombre, $telefono, $email]);
            $huesped_id = $db->lastInsertId();
        }
        
        // Guardar ID en sesión para continuar con la reserva
        $_SESSION['huesped_id'] = $huesped_id;
        $_SESSION['huesped_nombre'] = $nombre;
        $_SESSION['user_role'] = 'huesped';
        
        // Redirigir a la página de reserva
        redirect('reserva.php');
    }
}
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Registro de Huésped - <?php echo SITE_NAME; ?></title>
    <link rel="stylesheet" href="../assets/css/style.css">
</head>
<body>
    <!-- Header -->
    <header class="header">
        <div class="header-content">
            <div class="logo-container">
                <img src="../assets/images/logo.png" alt="Logo Posada" class="logo">
                <h1 class="site-title">Registro de Huésped</h1>
            </div>
            <nav>
                <ul class="nav-menu">
                    <li><a href="../index.php">← Volver al Inicio</a></li>
                </ul>
            </nav>
        </div>
    </header>

    <div class="container mt-3">
        <div class="card" style="max-width: 800px; margin: 0 auto;">
            <div class="card-header">
                <h2 class="card-title">Registro de Huésped</h2>
                <p style="color: var(--color-text-light); margin-top: 10px;">
                    Complete sus datos para realizar una reserva. Si ya se ha registrado antes, 
                    ingrese su cédula o pasaporte y sus datos se completarán automáticamente.
                </p>
            </div>
            <div class="card-body">
                <?php if ($error): ?>
                    <div class="alert alert-danger">
                        <?php echo htmlspecialchars($error); ?>
                    </div>
                <?php endif; ?>
                
                <?php if ($success): ?>
                    <div class="alert alert-success">
                        <?php echo htmlspecialchars($success); ?>
                    </div>
                <?php endif; ?>
                
                <form method="POST" action="" id="formRegistro">
                    <div class="form-group">
                        <label class="form-label">Cédula de Identidad o Pasaporte *</label>
                        <input type="text" name="cedula_pasaporte" id="cedula_pasaporte" 
                               class="form-control" placeholder="Ej: 1234567" required>
                        <small style="color: var(--color-text-light); display: block; margin-top: 5px;">
                            Ingrese su cédula o pasaporte. Si ya está registrado, sus datos se completarán automáticamente.
                        </small>
                    </div>
                    
                    <div id="loadingIndicator" style="display: none; text-align: center; padding: 20px;">
                        <div style="color: var(--color-primary);">Buscando datos...</div>
                    </div>
                    
                    <div class="form-group">
                        <label class="form-label">Nombre Completo *</label>
                        <input type="text" name="nombre_completo" id="nombre_completo" 
                               class="form-control" placeholder="Ej: Juan Pérez" required>
                    </div>
                    
                    <div class="form-group">
                        <label class="form-label">Teléfono *</label>
                        <input type="tel" name="telefono" id="telefono" 
                               class="form-control" placeholder="Ej: 0981123456" required>
                    </div>
                    
                    <div class="form-group">
                        <label class="form-label">Email (Opcional)</label>
                        <input type="email" name="email" id="email" 
                               class="form-control" placeholder="Ej: juan@ejemplo.com">
                    </div>
                    
                    <div class="alert alert-info">
                        <strong>Nota:</strong> Los campos marcados con * son obligatorios.
                    </div>
                    
                    <div class="d-flex gap-2" style="justify-content: space-between;">
                        <a href="../index.php" class="btn btn-secondary">
                            Cancelar
                        </a>
                        <button type="submit" name="registrar" class="btn btn-primary btn-lg">
                            Continuar con la Reserva →
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <footer class="footer">
        <div class="container">
            <p>&copy; <?php echo date('Y'); ?> Sistema de Reservas para Posadas. Todos los derechos reservados.</p>
        </div>
    </footer>

    <script src="../assets/js/main.js"></script>
    <script>
        // Autocompletado de datos al ingresar cédula
        const cedulaInput = document.getElementById('cedula_pasaporte');
        const nombreInput = document.getElementById('nombre_completo');
        const telefonoInput = document.getElementById('telefono');
        const emailInput = document.getElementById('email');
        const loadingIndicator = document.getElementById('loadingIndicator');
        
        let searchTimeout;
        
        cedulaInput.addEventListener('input', function() {
            clearTimeout(searchTimeout);
            const cedula = this.value.trim();
            
            if (cedula.length >= 3) {
                searchTimeout = setTimeout(() => {
                    buscarHuesped(cedula);
                }, 500);
            }
        });
        
        function buscarHuesped(cedula) {
            loadingIndicator.style.display = 'block';
            
            fetch(`?buscar_cedula=${encodeURIComponent(cedula)}`)
                .then(response => response.json())
                .then(data => {
                    loadingIndicator.style.display = 'none';
                    
                    if (data && !data.not_found) {
                        // Completar campos automáticamente
                        nombreInput.value = data.nombre_completo || '';
                        telefonoInput.value = data.telefono || '';
                        emailInput.value = data.email || '';
                        
                        showAlert('Datos encontrados y completados automáticamente', 'success');
                    } else {
                        // Limpiar campos si no se encuentra
                        nombreInput.value = '';
                        telefonoInput.value = '';
                        emailInput.value = '';
                    }
                })
                .catch(error => {
                    loadingIndicator.style.display = 'none';
                    console.error('Error:', error);
                });
        }
        
        // Validación del formulario
        document.getElementById('formRegistro').addEventListener('submit', function(e) {
            const cedula = cedulaInput.value.trim();
            const nombre = nombreInput.value.trim();
            const telefono = telefonoInput.value.trim();
            
            if (!cedula || !nombre || !telefono) {
                e.preventDefault();
                showAlert('Por favor complete todos los campos obligatorios', 'danger');
                return false;
            }
        });
    </script>
</body>
</html>
