<?php
require_once 'config.php';

if (!isLoggedIn() || (!hasRole('Administrador') && !hasRole('Supervisor'))) {
    redirect('dashboard.php');
}

$db = getDB();

// Obtener información del usuario actual
$stmt = $db->prepare("SELECT u.*, r.nombre as rol_nombre FROM usuarios u 
                      INNER JOIN roles r ON u.rol_id = r.id 
                      WHERE u.id = ?");
$stmt->execute([$_SESSION['user_id']]);
$usuario = $stmt->fetch();

// Filtros
$fecha_desde = $_GET['fecha_desde'] ?? date('Y-m-01');
$fecha_hasta = $_GET['fecha_hasta'] ?? date('Y-m-d');
$usuario_filtro = $_GET['usuario_filtro'] ?? '';
$tipo_filtro = $_GET['tipo_filtro'] ?? '';

// Construir query según el rol
$where_conditions = ["m.fecha_movimiento BETWEEN ? AND ?"];
$params = [$fecha_desde, $fecha_hasta];

if ($usuario['rol_nombre'] === 'Supervisor') {
    // Solo ver movimientos de sus operativos y propios
    $where_conditions[] = "(u.supervisor_id = ? OR m.usuario_id = ?)";
    $params[] = $_SESSION['user_id'];
    $params[] = $_SESSION['user_id'];
}

if ($usuario_filtro) {
    $where_conditions[] = "m.usuario_id = ?";
    $params[] = $usuario_filtro;
}

if ($tipo_filtro) {
    $where_conditions[] = "m.tipo = ?";
    $params[] = $tipo_filtro;
}

$where_clause = implode(' AND ', $where_conditions);

// Obtener movimientos
$query = "SELECT m.*, u.nombre_completo, c.nombre as categoria, r.nombre as rol 
          FROM movimientos m 
          INNER JOIN usuarios u ON m.usuario_id = u.id 
          INNER JOIN categorias c ON m.categoria_id = c.id 
          INNER JOIN roles r ON u.rol_id = r.id
          WHERE $where_clause
          ORDER BY m.fecha_movimiento DESC, m.created_at DESC";

$stmt = $db->prepare($query);
$stmt->execute($params);
$movimientos = $stmt->fetchAll();

// Calcular totales
$total_ingresos = 0;
$total_egresos = 0;
foreach ($movimientos as $mov) {
    if ($mov['tipo'] === 'Ingreso') {
        $total_ingresos += $mov['monto'];
    } else {
        $total_egresos += $mov['monto'];
    }
}

// Obtener usuarios para el filtro
if ($usuario['rol_nombre'] === 'Administrador') {
    $usuarios_filtro = $db->query("SELECT id, nombre_completo, rol_id FROM usuarios WHERE activo = 1 ORDER BY nombre_completo")->fetchAll();
} else {
    $usuarios_filtro = $db->prepare("SELECT id, nombre_completo, rol_id FROM usuarios 
                                     WHERE (supervisor_id = ? OR id = ?) AND activo = 1 
                                     ORDER BY nombre_completo");
    $usuarios_filtro->execute([$_SESSION['user_id'], $_SESSION['user_id']]);
    $usuarios_filtro = $usuarios_filtro->fetchAll();
}

$page_title = 'Reportes';
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo $page_title; ?> - <?php echo SITE_NAME; ?></title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.10.0/font/bootstrap-icons.css">
    <link rel="stylesheet" href="assets/css/style.css">
</head>
<body>
    <?php include 'includes/navbar.php'; ?>
    
    <div class="container-fluid">
        <div class="row">
            <?php include 'includes/sidebar.php'; ?>
            
            <main class="col-md-9 ms-sm-auto col-lg-10 px-md-4">
                <div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pt-3 pb-2 mb-3 border-bottom">
                    <h1 class="h2">
                        <i class="bi bi-file-earmark-bar-graph"></i> Reportes
                    </h1>
                    <div>
                        <a href="exportar.php?tipo=excel&<?php echo http_build_query($_GET); ?>" class="btn btn-success">
                            <i class="bi bi-file-earmark-excel"></i> Exportar Excel
                        </a>
                        <a href="exportar.php?tipo=pdf&<?php echo http_build_query($_GET); ?>" class="btn btn-danger">
                            <i class="bi bi-file-earmark-pdf"></i> Exportar PDF
                        </a>
                    </div>
                </div>
                
                <!-- Filtros -->
                <div class="card mb-4">
                    <div class="card-header">
                        <h5 class="mb-0"><i class="bi bi-funnel"></i> Filtros</h5>
                    </div>
                    <div class="card-body">
                        <form method="GET" class="row g-3">
                            <div class="col-md-3">
                                <label class="form-label">Desde</label>
                                <input type="date" class="form-control" name="fecha_desde" 
                                       value="<?php echo htmlspecialchars($fecha_desde); ?>">
                            </div>
                            <div class="col-md-3">
                                <label class="form-label">Hasta</label>
                                <input type="date" class="form-control" name="fecha_hasta" 
                                       value="<?php echo htmlspecialchars($fecha_hasta); ?>">
                            </div>
                            <div class="col-md-3">
                                <label class="form-label">Usuario</label>
                                <select class="form-select" name="usuario_filtro">
                                    <option value="">Todos</option>
                                    <?php foreach ($usuarios_filtro as $u): ?>
                                        <option value="<?php echo $u['id']; ?>" 
                                                <?php echo $usuario_filtro == $u['id'] ? 'selected' : ''; ?>>
                                            <?php echo htmlspecialchars($u['nombre_completo']); ?>
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                            <div class="col-md-2">
                                <label class="form-label">Tipo</label>
                                <select class="form-select" name="tipo_filtro">
                                    <option value="">Todos</option>
                                    <option value="Ingreso" <?php echo $tipo_filtro === 'Ingreso' ? 'selected' : ''; ?>>Ingreso</option>
                                    <option value="Egreso" <?php echo $tipo_filtro === 'Egreso' ? 'selected' : ''; ?>>Egreso</option>
                                </select>
                            </div>
                            <div class="col-md-1">
                                <label class="form-label">&nbsp;</label>
                                <button type="submit" class="btn btn-primary w-100">
                                    <i class="bi bi-search"></i>
                                </button>
                            </div>
                        </form>
                    </div>
                </div>
                
                <!-- Resumen -->
                <div class="row mb-4">
                    <div class="col-md-4">
                        <div class="card stat-card stat-success">
                            <div class="card-body">
                                <h6 class="text-muted mb-2">Total Ingresos</h6>
                                <h3 class="mb-0 text-success"><?php echo formatCurrency($total_ingresos); ?></h3>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-4">
                        <div class="card stat-card stat-danger">
                            <div class="card-body">
                                <h6 class="text-muted mb-2">Total Egresos</h6>
                                <h3 class="mb-0 text-danger"><?php echo formatCurrency($total_egresos); ?></h3>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-4">
                        <div class="card stat-card stat-info">
                            <div class="card-body">
                                <h6 class="text-muted mb-2">Balance</h6>
                                <h3 class="mb-0"><?php echo formatCurrency($total_ingresos - $total_egresos); ?></h3>
                            </div>
                        </div>
                    </div>
                </div>
                
                <!-- Tabla de movimientos -->
                <div class="card">
                    <div class="card-header">
                        <h5 class="mb-0"><i class="bi bi-table"></i> Movimientos (<?php echo count($movimientos); ?>)</h5>
                    </div>
                    <div class="card-body">
                        <div class="table-responsive">
                            <table class="table table-hover table-sm">
                                <thead>
                                    <tr>
                                        <th>Fecha</th>
                                        <th>Usuario</th>
                                        <th>Rol</th>
                                        <th>Categoría</th>
                                        <th>Tipo</th>
                                        <th>Monto</th>
                                        <th>Descripción</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php if (!empty($movimientos)): ?>
                                        <?php foreach ($movimientos as $mov): ?>
                                            <tr>
                                                <td><?php echo date('d/m/Y', strtotime($mov['fecha_movimiento'])); ?></td>
                                                <td><?php echo htmlspecialchars($mov['nombre_completo']); ?></td>
                                                <td>
                                                    <span class="badge bg-secondary"><?php echo htmlspecialchars($mov['rol']); ?></span>
                                                </td>
                                                <td><?php echo htmlspecialchars($mov['categoria']); ?></td>
                                                <td>
                                                    <?php if ($mov['tipo'] === 'Ingreso'): ?>
                                                        <span class="badge bg-success">Ingreso</span>
                                                    <?php else: ?>
                                                        <span class="badge bg-danger">Egreso</span>
                                                    <?php endif; ?>
                                                </td>
                                                <td class="<?php echo $mov['tipo'] === 'Ingreso' ? 'text-success' : 'text-danger'; ?>">
                                                    <?php echo formatCurrency($mov['monto']); ?>
                                                </td>
                                                <td><?php echo htmlspecialchars(substr($mov['descripcion'], 0, 50)); ?></td>
                                            </tr>
                                        <?php endforeach; ?>
                                    <?php else: ?>
                                        <tr>
                                            <td colspan="7" class="text-center text-muted">No hay movimientos en el período seleccionado</td>
                                        </tr>
                                    <?php endif; ?>
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
            </main>
        </div>
    </div>
    
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>

