<?php
require_once 'config.php';

if (!isLoggedIn() || !hasRole('Administrador')) {
    redirect('dashboard.php');
}

$db = getDB();
$mensaje = '';
$error = '';

// Procesar acciones
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $accion = $_POST['accion'] ?? '';
    
    if ($accion === 'crear') {
        $username = cleanInput($_POST['username']);
        $password = $_POST['password'];
        $nombre_completo = cleanInput($_POST['nombre_completo']);
        $cedula = cleanInput($_POST['cedula_ruc_pasaporte']);
        $telefono = cleanInput($_POST['telefono']);
        $ciudad = cleanInput($_POST['ciudad']);
        $direccion = cleanInput($_POST['direccion']);
        $observacion = cleanInput($_POST['observacion']);
        $rol_id = (int)$_POST['rol_id'];
        $supervisor_id = !empty($_POST['supervisor_id']) ? (int)$_POST['supervisor_id'] : null;
        
        // Validar que el username no exista
        $stmt = $db->prepare("SELECT id FROM usuarios WHERE username = ?");
        $stmt->execute([$username]);
        
        if ($stmt->fetch()) {
            $error = 'El nombre de usuario ya existe';
        } else {
            $password_hash = password_hash($password, PASSWORD_DEFAULT);
            
            $stmt = $db->prepare("INSERT INTO usuarios (username, password, nombre_completo, cedula_ruc_pasaporte, 
                                  telefono, ciudad, direccion, observacion, rol_id, supervisor_id) 
                                  VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");
            
            if ($stmt->execute([$username, $password_hash, $nombre_completo, $cedula, $telefono, 
                               $ciudad, $direccion, $observacion, $rol_id, $supervisor_id])) {
                $mensaje = 'Usuario creado exitosamente';
                
                // Log
                $stmt = $db->prepare("INSERT INTO logs (usuario_id, accion, detalles, ip_address) VALUES (?, ?, ?, ?)");
                $stmt->execute([$_SESSION['user_id'], 'Crear usuario', "Usuario: $username", $_SERVER['REMOTE_ADDR']]);
            } else {
                $error = 'Error al crear el usuario';
            }
        }
    } elseif ($accion === 'editar') {
        $usuario_id = (int)$_POST['usuario_id'];
        $nombre_completo = cleanInput($_POST['nombre_completo']);
        $cedula = cleanInput($_POST['cedula_ruc_pasaporte']);
        $telefono = cleanInput($_POST['telefono']);
        $ciudad = cleanInput($_POST['ciudad']);
        $direccion = cleanInput($_POST['direccion']);
        $observacion = cleanInput($_POST['observacion']);
        $rol_id = (int)$_POST['rol_id'];
        $supervisor_id = !empty($_POST['supervisor_id']) ? (int)$_POST['supervisor_id'] : null;
        
        // Si se proporciona nueva contraseña
        if (!empty($_POST['password'])) {
            $password_hash = password_hash($_POST['password'], PASSWORD_DEFAULT);
            $stmt = $db->prepare("UPDATE usuarios SET nombre_completo = ?, cedula_ruc_pasaporte = ?, 
                                  telefono = ?, ciudad = ?, direccion = ?, observacion = ?, 
                                  rol_id = ?, supervisor_id = ?, password = ? WHERE id = ?");
            $result = $stmt->execute([$nombre_completo, $cedula, $telefono, $ciudad, $direccion, 
                                     $observacion, $rol_id, $supervisor_id, $password_hash, $usuario_id]);
        } else {
            $stmt = $db->prepare("UPDATE usuarios SET nombre_completo = ?, cedula_ruc_pasaporte = ?, 
                                  telefono = ?, ciudad = ?, direccion = ?, observacion = ?, 
                                  rol_id = ?, supervisor_id = ? WHERE id = ?");
            $result = $stmt->execute([$nombre_completo, $cedula, $telefono, $ciudad, $direccion, 
                                     $observacion, $rol_id, $supervisor_id, $usuario_id]);
        }
        
        if ($result) {
            $mensaje = 'Usuario actualizado exitosamente';
            
            // Log
            $stmt = $db->prepare("INSERT INTO logs (usuario_id, accion, detalles, ip_address) VALUES (?, ?, ?, ?)");
            $stmt->execute([$_SESSION['user_id'], 'Editar usuario', "ID: $usuario_id", $_SERVER['REMOTE_ADDR']]);
        } else {
            $error = 'Error al actualizar el usuario';
        }
    } elseif ($accion === 'suspender') {
        $usuario_id = (int)$_POST['usuario_id'];
        
        $stmt = $db->prepare("UPDATE usuarios SET activo = 0 WHERE id = ?");
        if ($stmt->execute([$usuario_id])) {
            $mensaje = 'Usuario suspendido exitosamente';
            
            // Log
            $stmt = $db->prepare("INSERT INTO logs (usuario_id, accion, detalles, ip_address) VALUES (?, ?, ?, ?)");
            $stmt->execute([$_SESSION['user_id'], 'Suspender usuario', "ID: $usuario_id", $_SERVER['REMOTE_ADDR']]);
        } else {
            $error = 'Error al suspender el usuario';
        }
    } elseif ($accion === 'activar') {
        $usuario_id = (int)$_POST['usuario_id'];
        
        $stmt = $db->prepare("UPDATE usuarios SET activo = 1 WHERE id = ?");
        if ($stmt->execute([$usuario_id])) {
            $mensaje = 'Usuario activado exitosamente';
            
            // Log
            $stmt = $db->prepare("INSERT INTO logs (usuario_id, accion, detalles, ip_address) VALUES (?, ?, ?, ?)");
            $stmt->execute([$_SESSION['user_id'], 'Activar usuario', "ID: $usuario_id", $_SERVER['REMOTE_ADDR']]);
        } else {
            $error = 'Error al activar el usuario';
        }
    } elseif ($accion === 'eliminar') {
        $usuario_id = (int)$_POST['usuario_id'];
        
        try {
            $db->beginTransaction();
            
            // Eliminar logs del usuario
            $stmt = $db->prepare("DELETE FROM logs WHERE usuario_id = ?");
            $stmt->execute([$usuario_id]);
            
            // Eliminar movimientos del usuario
            $stmt = $db->prepare("DELETE FROM movimientos WHERE usuario_id = ?");
            $stmt->execute([$usuario_id]);
            
            // Eliminar asignaciones
            $stmt = $db->prepare("DELETE FROM asignaciones WHERE asignado_a = ? OR asignado_por = ?");
            $stmt->execute([$usuario_id, $usuario_id]);
            
            // Eliminar usuario
            $stmt = $db->prepare("DELETE FROM usuarios WHERE id = ?");
            $stmt->execute([$usuario_id]);
            
            $db->commit();
            
            $mensaje = 'Usuario eliminado permanentemente';
            
            // Log
            $stmt = $db->prepare("INSERT INTO logs (usuario_id, accion, detalles, ip_address) VALUES (?, ?, ?, ?)");
            $stmt->execute([$_SESSION['user_id'], 'Eliminar usuario', "ID: $usuario_id", $_SERVER['REMOTE_ADDR']]);
        } catch (Exception $e) {
            $db->rollBack();
            $error = 'Error al eliminar el usuario: ' . $e->getMessage();
        }
    }
}

// Obtener usuarios (activos e inactivos)
$stmt = $db->query("SELECT u.*, r.nombre as rol_nombre, 
                    s.nombre_completo as supervisor_nombre
                    FROM usuarios u 
                    INNER JOIN roles r ON u.rol_id = r.id 
                    LEFT JOIN usuarios s ON u.supervisor_id = s.id
                    ORDER BY u.activo DESC, r.id, u.nombre_completo");
$usuarios = $stmt->fetchAll();

// Obtener roles
$roles = $db->query("SELECT * FROM roles ORDER BY id")->fetchAll();

// Obtener supervisores para el select
$supervisores = $db->query("SELECT id, nombre_completo FROM usuarios WHERE rol_id = 2 AND activo = 1")->fetchAll();

// Obtener información del usuario actual
$stmt = $db->prepare("SELECT u.*, r.nombre as rol_nombre FROM usuarios u 
                      INNER JOIN roles r ON u.rol_id = r.id 
                      WHERE u.id = ?");
$stmt->execute([$_SESSION['user_id']]);
$usuario = $stmt->fetch();

$page_title = 'Gestión de Usuarios';
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo $page_title; ?> - <?php echo SITE_NAME; ?></title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.10.0/font/bootstrap-icons.css">
    <link rel="stylesheet" href="assets/css/style.css">
</head>
<body>
    <?php include 'includes/navbar.php'; ?>
    
    <div class="container-fluid">
        <div class="row">
            <?php include 'includes/sidebar.php'; ?>
            
            <main class="col-md-9 ms-sm-auto col-lg-10 px-md-4">
                <div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pt-3 pb-2 mb-3 border-bottom">
                    <h1 class="h2">
                        <i class="bi bi-people"></i> Gestión de Usuarios
                    </h1>
                    <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#nuevoUsuarioModal">
                        <i class="bi bi-plus-circle"></i> Nuevo Usuario
                    </button>
                </div>
                
                <?php if ($mensaje): ?>
                    <div class="alert alert-success alert-dismissible fade show" role="alert">
                        <i class="bi bi-check-circle"></i> <?php echo $mensaje; ?>
                        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                    </div>
                <?php endif; ?>
                
                <?php if ($error): ?>
                    <div class="alert alert-danger alert-dismissible fade show" role="alert">
                        <i class="bi bi-exclamation-triangle"></i> <?php echo $error; ?>
                        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                    </div>
                <?php endif; ?>
                
                <div class="card">
                    <div class="card-header">
                        <h5 class="mb-0"><i class="bi bi-list"></i> Lista de Usuarios</h5>
                    </div>
                    <div class="card-body">
                        <div class="table-responsive">
                            <table class="table table-hover">
                                <thead>
                                    <tr>
                                        <th>Estado</th>
                                        <th>Usuario</th>
                                        <th>Nombre Completo</th>
                                        <th>Cédula/RUC</th>
                                        <th>Rol</th>
                                        <th>Supervisor</th>
                                        <th>Fondo Asignado</th>
                                        <th>Saldo Disponible</th>
                                        <th>Acciones</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($usuarios as $u): ?>
                                        <tr class="<?php echo $u['activo'] ? '' : 'table-secondary'; ?>">
                                            <td>
                                                <?php if ($u['activo']): ?>
                                                    <span class="badge bg-success">Activo</span>
                                                <?php else: ?>
                                                    <span class="badge bg-secondary">Suspendido</span>
                                                <?php endif; ?>
                                            </td>
                                            <td><?php echo htmlspecialchars($u['username']); ?></td>
                                            <td><?php echo htmlspecialchars($u['nombre_completo']); ?></td>
                                            <td><?php echo htmlspecialchars($u['cedula_ruc_pasaporte']); ?></td>
                                            <td>
                                                <span class="badge bg-<?php 
                                                    echo $u['rol_id'] == 1 ? 'danger' : ($u['rol_id'] == 2 ? 'warning' : 'info'); 
                                                ?>">
                                                    <?php echo htmlspecialchars($u['rol_nombre']); ?>
                                                </span>
                                            </td>
                                            <td><?php echo htmlspecialchars($u['supervisor_nombre'] ?? '-'); ?></td>
                                            <td><?php echo formatCurrency($u['fondo_asignado']); ?></td>
                                            <td><?php echo formatCurrency($u['saldo_disponible']); ?></td>
                                            <td>
                                                <?php if ($u['id'] != $_SESSION['user_id']): ?>
                                                    <button class="btn btn-sm btn-primary" 
                                                            onclick="editarUsuario(<?php echo htmlspecialchars(json_encode($u)); ?>)"
                                                            title="Editar">
                                                        <i class="bi bi-pencil"></i>
                                                    </button>
                                                    
                                                    <?php if ($u['activo']): ?>
                                                        <form method="POST" style="display: inline;" 
                                                              onsubmit="return confirm('¿Está seguro de suspender este usuario?');">
                                                            <input type="hidden" name="accion" value="suspender">
                                                            <input type="hidden" name="usuario_id" value="<?php echo $u['id']; ?>">
                                                            <button type="submit" class="btn btn-sm btn-warning" title="Suspender">
                                                                <i class="bi bi-pause-circle"></i>
                                                            </button>
                                                        </form>
                                                    <?php else: ?>
                                                        <form method="POST" style="display: inline;" 
                                                              onsubmit="return confirm('¿Está seguro de reactivar este usuario?');">
                                                            <input type="hidden" name="accion" value="activar">
                                                            <input type="hidden" name="usuario_id" value="<?php echo $u['id']; ?>">
                                                            <button type="submit" class="btn btn-sm btn-success" title="Activar">
                                                                <i class="bi bi-play-circle"></i>
                                                            </button>
                                                        </form>
                                                    <?php endif; ?>
                                                    
                                                    <form method="POST" style="display: inline;" 
                                                          onsubmit="return confirm('¿Está seguro de ELIMINAR PERMANENTEMENTE este usuario? Esta acción no se puede deshacer.');">
                                                        <input type="hidden" name="accion" value="eliminar">
                                                        <input type="hidden" name="usuario_id" value="<?php echo $u['id']; ?>">
                                                        <button type="submit" class="btn btn-sm btn-danger" title="Eliminar">
                                                            <i class="bi bi-trash"></i>
                                                        </button>
                                                    </form>
                                                <?php endif; ?>
                                            </td>
                                        </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
            </main>
        </div>
    </div>
    
    <!-- Modal Nuevo Usuario -->
    <div class="modal fade" id="nuevoUsuarioModal" tabindex="-1">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title"><i class="bi bi-person-plus"></i> Nuevo Usuario</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <form method="POST">
                    <div class="modal-body">
                        <input type="hidden" name="accion" value="crear">
                        
                        <div class="row">
                            <div class="col-md-6 mb-3">
                                <label class="form-label">Usuario *</label>
                                <input type="text" class="form-control" name="username" required>
                            </div>
                            <div class="col-md-6 mb-3">
                                <label class="form-label">Contraseña *</label>
                                <input type="password" class="form-control" name="password" required>
                            </div>
                        </div>
                        
                        <div class="mb-3">
                            <label class="form-label">Nombre Completo *</label>
                            <input type="text" class="form-control" name="nombre_completo" required>
                        </div>
                        
                        <div class="row">
                            <div class="col-md-6 mb-3">
                                <label class="form-label">Cédula/RUC/Pasaporte *</label>
                                <input type="text" class="form-control" name="cedula_ruc_pasaporte" required>
                            </div>
                            <div class="col-md-6 mb-3">
                                <label class="form-label">Teléfono</label>
                                <input type="text" class="form-control" name="telefono">
                            </div>
                        </div>
                        
                        <div class="row">
                            <div class="col-md-6 mb-3">
                                <label class="form-label">Ciudad</label>
                                <input type="text" class="form-control" name="ciudad">
                            </div>
                            <div class="col-md-6 mb-3">
                                <label class="form-label">Dirección</label>
                                <input type="text" class="form-control" name="direccion">
                            </div>
                        </div>
                        
                        <div class="row">
                            <div class="col-md-6 mb-3">
                                <label class="form-label">Rol *</label>
                                <select class="form-select" name="rol_id" id="rol_id" required>
                                    <option value="">Seleccione un rol</option>
                                    <?php foreach ($roles as $rol): ?>
                                        <option value="<?php echo $rol['id']; ?>">
                                            <?php echo htmlspecialchars($rol['nombre']); ?>
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                            <div class="col-md-6 mb-3" id="supervisor_field" style="display: none;">
                                <label class="form-label">Supervisor</label>
                                <select class="form-select" name="supervisor_id">
                                    <option value="">Seleccione un supervisor</option>
                                    <?php foreach ($supervisores as $sup): ?>
                                        <option value="<?php echo $sup['id']; ?>">
                                            <?php echo htmlspecialchars($sup['nombre_completo']); ?>
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                        </div>
                        
                        <div class="mb-3">
                            <label class="form-label">Observación</label>
                            <textarea class="form-control" name="observacion" rows="3"></textarea>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancelar</button>
                        <button type="submit" class="btn btn-primary">
                            <i class="bi bi-save"></i> Guardar Usuario
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>
    
    <!-- Modal Editar Usuario -->
    <div class="modal fade" id="editarUsuarioModal" tabindex="-1">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title"><i class="bi bi-pencil"></i> Editar Usuario</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <form method="POST">
                    <div class="modal-body">
                        <input type="hidden" name="accion" value="editar">
                        <input type="hidden" name="usuario_id" id="edit_usuario_id">
                        
                        <div class="alert alert-info">
                            <i class="bi bi-info-circle"></i> Usuario: <strong id="edit_username_display"></strong>
                        </div>
                        
                        <div class="mb-3">
                            <label class="form-label">Nombre Completo *</label>
                            <input type="text" class="form-control" name="nombre_completo" id="edit_nombre_completo" required>
                        </div>
                        
                        <div class="row">
                            <div class="col-md-6 mb-3">
                                <label class="form-label">Cédula/RUC/Pasaporte *</label>
                                <input type="text" class="form-control" name="cedula_ruc_pasaporte" id="edit_cedula" required>
                            </div>
                            <div class="col-md-6 mb-3">
                                <label class="form-label">Teléfono</label>
                                <input type="text" class="form-control" name="telefono" id="edit_telefono">
                            </div>
                        </div>
                        
                        <div class="row">
                            <div class="col-md-6 mb-3">
                                <label class="form-label">Ciudad</label>
                                <input type="text" class="form-control" name="ciudad" id="edit_ciudad">
                            </div>
                            <div class="col-md-6 mb-3">
                                <label class="form-label">Dirección</label>
                                <input type="text" class="form-control" name="direccion" id="edit_direccion">
                            </div>
                        </div>
                        
                        <div class="row">
                            <div class="col-md-6 mb-3">
                                <label class="form-label">Rol *</label>
                                <select class="form-select" name="rol_id" id="edit_rol_id" required>
                                    <?php foreach ($roles as $rol): ?>
                                        <option value="<?php echo $rol['id']; ?>">
                                            <?php echo htmlspecialchars($rol['nombre']); ?>
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                            <div class="col-md-6 mb-3" id="edit_supervisor_field">
                                <label class="form-label">Supervisor</label>
                                <select class="form-select" name="supervisor_id" id="edit_supervisor_id">
                                    <option value="">Sin supervisor</option>
                                    <?php foreach ($supervisores as $sup): ?>
                                        <option value="<?php echo $sup['id']; ?>">
                                            <?php echo htmlspecialchars($sup['nombre_completo']); ?>
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                        </div>
                        
                        <div class="mb-3">
                            <label class="form-label">Nueva Contraseña</label>
                            <input type="password" class="form-control" name="password" id="edit_password">
                            <small class="text-muted">Dejar en blanco para mantener la contraseña actual</small>
                        </div>
                        
                        <div class="mb-3">
                            <label class="form-label">Observación</label>
                            <textarea class="form-control" name="observacion" id="edit_observacion" rows="3"></textarea>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancelar</button>
                        <button type="submit" class="btn btn-primary">
                            <i class="bi bi-save"></i> Actualizar Usuario
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>
    
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Mostrar campo supervisor solo si el rol es Operativo (3)
        document.getElementById('rol_id').addEventListener('change', function() {
            const supervisorField = document.getElementById('supervisor_field');
            if (this.value == '3') {
                supervisorField.style.display = 'block';
            } else {
                supervisorField.style.display = 'none';
            }
        });
        
        // Función para editar usuario
        function editarUsuario(usuario) {
            document.getElementById('edit_usuario_id').value = usuario.id;
            document.getElementById('edit_username_display').textContent = usuario.username;
            document.getElementById('edit_nombre_completo').value = usuario.nombre_completo;
            document.getElementById('edit_cedula').value = usuario.cedula_ruc_pasaporte;
            document.getElementById('edit_telefono').value = usuario.telefono || '';
            document.getElementById('edit_ciudad').value = usuario.ciudad || '';
            document.getElementById('edit_direccion').value = usuario.direccion || '';
            document.getElementById('edit_rol_id').value = usuario.rol_id;
            document.getElementById('edit_supervisor_id').value = usuario.supervisor_id || '';
            document.getElementById('edit_observacion').value = usuario.observacion || '';
            document.getElementById('edit_password').value = '';
            
            // Mostrar/ocultar campo supervisor
            const editSupervisorField = document.getElementById('edit_supervisor_field');
            if (usuario.rol_id == 3) {
                editSupervisorField.style.display = 'block';
            } else {
                editSupervisorField.style.display = 'none';
            }
            
            // Mostrar modal
            const modal = new bootstrap.Modal(document.getElementById('editarUsuarioModal'));
            modal.show();
        }
        
        // Actualizar visibilidad del campo supervisor al cambiar rol en edición
        document.getElementById('edit_rol_id').addEventListener('change', function() {
            const editSupervisorField = document.getElementById('edit_supervisor_field');
            if (this.value == '3') {
                editSupervisorField.style.display = 'block';
            } else {
                editSupervisorField.style.display = 'none';
            }
        });
    </script>
</body>
</html>

