<?php
require_once 'config.php';

if (!isLoggedIn() || !hasRole('Supervisor')) {
    redirect('dashboard.php');
}

$db = getDB();
$mensaje = '';
$error = '';

// Procesar acciones
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $accion = $_POST['accion'] ?? '';
    
    if ($accion === 'crear') {
        $username = cleanInput($_POST['username']);
        $password = $_POST['password'];
        $nombre_completo = cleanInput($_POST['nombre_completo']);
        $cedula = cleanInput($_POST['cedula_ruc_pasaporte']);
        $telefono = cleanInput($_POST['telefono']);
        $ciudad = cleanInput($_POST['ciudad']);
        $direccion = cleanInput($_POST['direccion']);
        $observacion = cleanInput($_POST['observacion']);
        
        // Validar que el username no exista
        $stmt = $db->prepare("SELECT id FROM usuarios WHERE username = ?");
        $stmt->execute([$username]);
        
        if ($stmt->fetch()) {
            $error = 'El nombre de usuario ya existe';
        } else {
            $password_hash = password_hash($password, PASSWORD_DEFAULT);
            
            // Rol 3 = Operativo, supervisor_id = usuario actual
            $stmt = $db->prepare("INSERT INTO usuarios (username, password, nombre_completo, cedula_ruc_pasaporte, 
                                  telefono, ciudad, direccion, observacion, rol_id, supervisor_id) 
                                  VALUES (?, ?, ?, ?, ?, ?, ?, ?, 3, ?)");
            
            if ($stmt->execute([$username, $password_hash, $nombre_completo, $cedula, $telefono, 
                               $ciudad, $direccion, $observacion, $_SESSION['user_id']])) {
                $mensaje = 'Operativo creado exitosamente';
                
                // Log
                $stmt = $db->prepare("INSERT INTO logs (usuario_id, accion, detalles, ip_address) VALUES (?, ?, ?, ?)");
                $stmt->execute([$_SESSION['user_id'], 'Crear operativo', "Usuario: $username", $_SERVER['REMOTE_ADDR']]);
            } else {
                $error = 'Error al crear el operativo';
            }
        }
    } elseif ($accion === 'eliminar') {
        $usuario_id = (int)$_POST['usuario_id'];
        
        // Verificar que el operativo pertenezca al supervisor
        $stmt = $db->prepare("SELECT id FROM usuarios WHERE id = ? AND supervisor_id = ?");
        $stmt->execute([$usuario_id, $_SESSION['user_id']]);
        
        if ($stmt->fetch()) {
            $stmt = $db->prepare("UPDATE usuarios SET activo = 0 WHERE id = ?");
            if ($stmt->execute([$usuario_id])) {
                $mensaje = 'Operativo desactivado exitosamente';
                
                // Log
                $stmt = $db->prepare("INSERT INTO logs (usuario_id, accion, detalles, ip_address) VALUES (?, ?, ?, ?)");
                $stmt->execute([$_SESSION['user_id'], 'Desactivar operativo', "ID: $usuario_id", $_SERVER['REMOTE_ADDR']]);
            } else {
                $error = 'Error al desactivar el operativo';
            }
        } else {
            $error = 'No tiene permisos para desactivar este operativo';
        }
    }
}

// Obtener operativos del supervisor
$stmt = $db->prepare("SELECT u.* FROM usuarios u 
                      WHERE u.supervisor_id = ? AND u.activo = 1 AND u.rol_id = 3
                      ORDER BY u.nombre_completo");
$stmt->execute([$_SESSION['user_id']]);
$operativos = $stmt->fetchAll();

// Obtener información del usuario actual
$stmt = $db->prepare("SELECT u.*, r.nombre as rol_nombre FROM usuarios u 
                      INNER JOIN roles r ON u.rol_id = r.id 
                      WHERE u.id = ?");
$stmt->execute([$_SESSION['user_id']]);
$usuario = $stmt->fetch();

$page_title = 'Mis Operativos';
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo $page_title; ?> - <?php echo SITE_NAME; ?></title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.10.0/font/bootstrap-icons.css">
    <link rel="stylesheet" href="assets/css/style.css">
</head>
<body>
    <?php include 'includes/navbar.php'; ?>
    
    <div class="container-fluid">
        <div class="row">
            <?php include 'includes/sidebar.php'; ?>
            
            <main class="col-md-9 ms-sm-auto col-lg-10 px-md-4">
                <div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pt-3 pb-2 mb-3 border-bottom">
                    <h1 class="h2">
                        <i class="bi bi-people"></i> Mis Operativos
                    </h1>
                    <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#nuevoOperativoModal">
                        <i class="bi bi-plus-circle"></i> Nuevo Operativo
                    </button>
                </div>
                
                <?php if ($mensaje): ?>
                    <div class="alert alert-success alert-dismissible fade show" role="alert">
                        <i class="bi bi-check-circle"></i> <?php echo $mensaje; ?>
                        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                    </div>
                <?php endif; ?>
                
                <?php if ($error): ?>
                    <div class="alert alert-danger alert-dismissible fade show" role="alert">
                        <i class="bi bi-exclamation-triangle"></i> <?php echo $error; ?>
                        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                    </div>
                <?php endif; ?>
                
                <div class="card">
                    <div class="card-header">
                        <h5 class="mb-0"><i class="bi bi-list"></i> Lista de Operativos (<?php echo count($operativos); ?>)</h5>
                    </div>
                    <div class="card-body">
                        <div class="table-responsive">
                            <table class="table table-hover">
                                <thead>
                                    <tr>
                                        <th>Usuario</th>
                                        <th>Nombre Completo</th>
                                        <th>Cédula/RUC</th>
                                        <th>Teléfono</th>
                                        <th>Ciudad</th>
                                        <th>Fondo Asignado</th>
                                        <th>Saldo Disponible</th>
                                        <th>Acciones</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php if (!empty($operativos)): ?>
                                        <?php foreach ($operativos as $op): ?>
                                            <tr>
                                                <td><?php echo htmlspecialchars($op['username']); ?></td>
                                                <td><?php echo htmlspecialchars($op['nombre_completo']); ?></td>
                                                <td><?php echo htmlspecialchars($op['cedula_ruc_pasaporte']); ?></td>
                                                <td><?php echo htmlspecialchars($op['telefono']); ?></td>
                                                <td><?php echo htmlspecialchars($op['ciudad']); ?></td>
                                                <td><?php echo formatCurrency($op['fondo_asignado']); ?></td>
                                                <td><?php echo formatCurrency($op['saldo_disponible']); ?></td>
                                                <td>
                                                    <a href="asignar_operativos.php?operativo_id=<?php echo $op['id']; ?>" 
                                                       class="btn btn-sm btn-success" title="Asignar fondos">
                                                        <i class="bi bi-cash-coin"></i>
                                                    </a>
                                                    <form method="POST" style="display: inline;" 
                                                          onsubmit="return confirm('¿Está seguro de desactivar este operativo?');">
                                                        <input type="hidden" name="accion" value="eliminar">
                                                        <input type="hidden" name="usuario_id" value="<?php echo $op['id']; ?>">
                                                        <button type="submit" class="btn btn-sm btn-danger" title="Desactivar">
                                                            <i class="bi bi-trash"></i>
                                                        </button>
                                                    </form>
                                                </td>
                                            </tr>
                                        <?php endforeach; ?>
                                    <?php else: ?>
                                        <tr>
                                            <td colspan="8" class="text-center text-muted">No tiene operativos asignados</td>
                                        </tr>
                                    <?php endif; ?>
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
            </main>
        </div>
    </div>
    
    <!-- Modal Nuevo Operativo -->
    <div class="modal fade" id="nuevoOperativoModal" tabindex="-1">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title"><i class="bi bi-person-plus"></i> Nuevo Operativo</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <form method="POST">
                    <div class="modal-body">
                        <input type="hidden" name="accion" value="crear">
                        
                        <div class="row">
                            <div class="col-md-6 mb-3">
                                <label class="form-label">Usuario *</label>
                                <input type="text" class="form-control" name="username" required>
                            </div>
                            <div class="col-md-6 mb-3">
                                <label class="form-label">Contraseña *</label>
                                <input type="password" class="form-control" name="password" required>
                            </div>
                        </div>
                        
                        <div class="mb-3">
                            <label class="form-label">Nombre Completo *</label>
                            <input type="text" class="form-control" name="nombre_completo" required>
                        </div>
                        
                        <div class="row">
                            <div class="col-md-6 mb-3">
                                <label class="form-label">Cédula/RUC/Pasaporte *</label>
                                <input type="text" class="form-control" name="cedula_ruc_pasaporte" required>
                            </div>
                            <div class="col-md-6 mb-3">
                                <label class="form-label">Teléfono</label>
                                <input type="text" class="form-control" name="telefono">
                            </div>
                        </div>
                        
                        <div class="row">
                            <div class="col-md-6 mb-3">
                                <label class="form-label">Ciudad</label>
                                <input type="text" class="form-control" name="ciudad">
                            </div>
                            <div class="col-md-6 mb-3">
                                <label class="form-label">Dirección</label>
                                <input type="text" class="form-control" name="direccion">
                            </div>
                        </div>
                        
                        <div class="mb-3">
                            <label class="form-label">Observación</label>
                            <textarea class="form-control" name="observacion" rows="3"></textarea>
                        </div>
                        
                        <div class="alert alert-info">
                            <i class="bi bi-info-circle"></i> El operativo será creado bajo su supervisión automáticamente.
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancelar</button>
                        <button type="submit" class="btn btn-primary">
                            <i class="bi bi-save"></i> Crear Operativo
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>
    
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>

