<?php
require_once 'config.php';

if (!isLoggedIn() || !hasRole('Administrador')) {
    redirect('dashboard.php');
}

$db = getDB();
$mensaje = '';
$error = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $asignado_a = (int)$_POST['asignado_a'];
    $monto = (float)str_replace(['.', ','], ['', '.'], $_POST['monto']);
    $descripcion = cleanInput($_POST['descripcion']);
    $fecha_asignacion = cleanInput($_POST['fecha_asignacion']);
    
    // Procesar comprobante fotográfico
    $comprobante_nombre = null;
    if (isset($_FILES['comprobante']) && $_FILES['comprobante']['error'] === UPLOAD_ERR_OK) {
        $allowed = ['jpg', 'jpeg', 'png', 'gif'];
        $filename = $_FILES['comprobante']['name'];
        $ext = strtolower(pathinfo($filename, PATHINFO_EXTENSION));
        
        if (in_array($ext, $allowed)) {
            $comprobante_nombre = 'comprobante_' . time() . '_' . uniqid() . '.' . $ext;
            $upload_path = __DIR__ . '/uploads/' . $comprobante_nombre;
            
            if (!move_uploaded_file($_FILES['comprobante']['tmp_name'], $upload_path)) {
                $error = 'Error al subir el comprobante';
                $comprobante_nombre = null;
            }
        } else {
            $error = 'Formato de imagen no permitido';
        }
    }
    
    if (!$error) {
        if ($monto <= 0) {
            $error = 'El monto debe ser mayor a cero';
        } else {
            try {
                $db->beginTransaction();
                
                // Registrar asignación con comprobante
                $stmt = $db->prepare("INSERT INTO asignaciones (asignado_por, asignado_a, monto, descripcion, comprobante, fecha_asignacion) 
                                      VALUES (?, ?, ?, ?, ?, ?)");
                $stmt->execute([$_SESSION['user_id'], $asignado_a, $monto, $descripcion, $comprobante_nombre, $fecha_asignacion]);
                
                // Actualizar fondos del usuario
                $stmt = $db->prepare("UPDATE usuarios SET 
                                      fondo_asignado = fondo_asignado + ?,
                                      saldo_disponible = saldo_disponible + ?
                                      WHERE id = ?");
                $stmt->execute([$monto, $monto, $asignado_a]);
                
                // Actualizar saldo disponible del Administrador (restar el monto asignado)
                $stmt = $db->prepare("UPDATE usuarios SET 
                                      saldo_disponible = saldo_disponible - ?
                                      WHERE id = ?");
                $stmt->execute([$monto, $_SESSION['user_id']]);
                
                // Registrar como movimiento de ingreso
                $stmt = $db->prepare("SELECT saldo_disponible FROM usuarios WHERE id = ?");
                $stmt->execute([$asignado_a]);
                $saldo_nuevo = $stmt->fetchColumn();
                
                $stmt = $db->prepare("INSERT INTO movimientos (usuario_id, categoria_id, tipo, monto, descripcion, 
                                      fecha_movimiento, saldo_anterior, saldo_posterior, created_by) 
                                      VALUES (?, 2, 'Ingreso', ?, ?, ?, ?, ?, ?)");
                $stmt->execute([$asignado_a, $monto, $descripcion, $fecha_asignacion, 
                               $saldo_nuevo - $monto, $saldo_nuevo, $_SESSION['user_id']]);
                
                // Log
                $stmt = $db->prepare("INSERT INTO logs (usuario_id, accion, detalles, ip_address) VALUES (?, ?, ?, ?)");
                $stmt->execute([$_SESSION['user_id'], 'Asignar fondos', "Monto: $monto a usuario ID: $asignado_a", $_SERVER['REMOTE_ADDR']]);
                
                $db->commit();
                
                $mensaje = 'Fondos asignados exitosamente';
            } catch (Exception $e) {
                $db->rollBack();
                // Eliminar comprobante si hubo error
                if ($comprobante_nombre && file_exists(__DIR__ . '/uploads/' . $comprobante_nombre)) {
                    unlink(__DIR__ . '/uploads/' . $comprobante_nombre);
                }
                $error = 'Error al asignar fondos: ' . $e->getMessage();
            }
        }
    }
}

// Obtener supervisores y operativos
$supervisores = $db->query("SELECT id, nombre_completo, fondo_asignado, saldo_disponible 
                            FROM usuarios WHERE rol_id = 2 AND activo = 1 
                            ORDER BY nombre_completo")->fetchAll();

$operativos = $db->query("SELECT id, nombre_completo, fondo_asignado, saldo_disponible 
                          FROM usuarios WHERE rol_id = 3 AND activo = 1 
                          ORDER BY nombre_completo")->fetchAll();

// Obtener historial de asignaciones
$stmt = $db->query("SELECT a.*, u.nombre_completo as asignado_a_nombre, r.nombre as rol_asignado
                    FROM asignaciones a 
                    INNER JOIN usuarios u ON a.asignado_a = u.id 
                    INNER JOIN roles r ON u.rol_id = r.id
                    WHERE a.asignado_por = {$_SESSION['user_id']}
                    ORDER BY a.created_at DESC 
                    LIMIT 20");
$asignaciones = $stmt->fetchAll();

// Obtener información del usuario actual
$stmt = $db->prepare("SELECT u.*, r.nombre as rol_nombre FROM usuarios u 
                      INNER JOIN roles r ON u.rol_id = r.id 
                      WHERE u.id = ?");
$stmt->execute([$_SESSION['user_id']]);
$usuario = $stmt->fetch();

$page_title = 'Asignar Fondos';
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo $page_title; ?> - <?php echo SITE_NAME; ?></title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.10.0/font/bootstrap-icons.css">
    <link rel="stylesheet" href="assets/css/style.css">
</head>
<body>
    <?php include 'includes/navbar.php'; ?>
    
    <div class="container-fluid">
        <div class="row">
            <?php include 'includes/sidebar.php'; ?>
            
            <main class="col-md-9 ms-sm-auto col-lg-10 px-md-4">
                <div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pt-3 pb-2 mb-3 border-bottom">
                    <h1 class="h2">
                        <i class="bi bi-cash-coin"></i> Asignar Fondos
                    </h1>
                </div>
                
                <?php if ($mensaje): ?>
                    <div class="alert alert-success alert-dismissible fade show" role="alert">
                        <i class="bi bi-check-circle"></i> <?php echo $mensaje; ?>
                        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                    </div>
                <?php endif; ?>
                
                <?php if ($error): ?>
                    <div class="alert alert-danger alert-dismissible fade show" role="alert">
                        <i class="bi bi-exclamation-triangle"></i> <?php echo $error; ?>
                        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                    </div>
                <?php endif; ?>
                
                <div class="row">
                    <div class="col-md-6 mb-4">
                        <div class="card">
                            <div class="card-header">
                                <h5 class="mb-0"><i class="bi bi-plus-circle"></i> Nueva Asignación</h5>
                            </div>
                            <div class="card-body">
                                <form method="POST" enctype="multipart/form-data">
                                    <div class="mb-3">
                                        <label class="form-label">Asignar a *</label>
                                        <select class="form-select" name="asignado_a" required>
                                            <option value="">Seleccione un usuario</option>
                                            <optgroup label="Supervisores">
                                                <?php foreach ($supervisores as $sup): ?>
                                                    <option value="<?php echo $sup['id']; ?>">
                                                        <?php echo htmlspecialchars($sup['nombre_completo']); ?> 
                                                        (Saldo: <?php echo formatCurrency($sup['saldo_disponible']); ?>)
                                                    </option>
                                                <?php endforeach; ?>
                                            </optgroup>
                                            <optgroup label="Operativos">
                                                <?php foreach ($operativos as $op): ?>
                                                    <option value="<?php echo $op['id']; ?>">
                                                        <?php echo htmlspecialchars($op['nombre_completo']); ?> 
                                                        (Saldo: <?php echo formatCurrency($op['saldo_disponible']); ?>)
                                                    </option>
                                                <?php endforeach; ?>
                                            </optgroup>
                                        </select>
                                    </div>
                                    
                                    <div class="mb-3">
                                        <label class="form-label">Monto (Gs.) *</label>
                                        <input type="text" class="form-control" name="monto" id="monto" 
                                               placeholder="0" required>
                                    </div>
                                    
                                    <div class="mb-3">
                                        <label class="form-label">Fecha *</label>
                                        <input type="date" class="form-control" name="fecha_asignacion" 
                                               value="<?php echo date('Y-m-d'); ?>" required>
                                    </div>
                                    
                                    <div class="mb-3">
                                        <label class="form-label">Descripción</label>
                                        <textarea class="form-control" name="descripcion" rows="3" 
                                                  placeholder="Descripción de la asignación"></textarea>
                                    </div>
                                    
                                    <div class="mb-3">
                                        <label class="form-label">Comprobante Fotográfico</label>
                                        <input type="file" class="form-control" name="comprobante" 
                                               accept="image/*" capture="environment">
                                        <small class="text-muted">
                                            Puede tomar una foto o seleccionar desde la galería (JPG, PNG, GIF)
                                        </small>
                                    </div>
                                    
                                    <div class="d-grid">
                                        <button type="submit" class="btn btn-primary">
                                            <i class="bi bi-save"></i> Asignar Fondos
                                        </button>
                                    </div>
                                </form>
                            </div>
                        </div>
                    </div>
                    
                    <div class="col-md-6 mb-4">
                        <div class="card">
                            <div class="card-header">
                                <h5 class="mb-0"><i class="bi bi-clock-history"></i> Últimas Asignaciones</h5>
                            </div>
                            <div class="card-body">
                                <div class="table-responsive">
                                    <table class="table table-sm">
                                        <thead>
                                            <tr>
                                                <th>Fecha</th>
                                                <th>Usuario</th>
                                                <th>Rol</th>
                                                <th>Monto</th>
                                                <th>Comprobante</th>
                                            </tr>
                                        </thead>
                                        <tbody>
                                            <?php if (!empty($asignaciones)): ?>
                                                <?php foreach ($asignaciones as $asig): ?>
                                                    <tr>
                                                        <td><?php echo date('d/m/Y', strtotime($asig['fecha_asignacion'])); ?></td>
                                                        <td><?php echo htmlspecialchars($asig['asignado_a_nombre']); ?></td>
                                                        <td>
                                                            <span class="badge bg-<?php echo $asig['rol_asignado'] === 'Supervisor' ? 'warning' : 'info'; ?>">
                                                                <?php echo htmlspecialchars($asig['rol_asignado']); ?>
                                                            </span>
                                                        </td>
                                                        <td><?php echo formatCurrency($asig['monto']); ?></td>
                                                        <td>
                                                            <?php if ($asig['comprobante']): ?>
                                                                <a href="uploads/<?php echo htmlspecialchars($asig['comprobante']); ?>" 
                                                                   target="_blank" class="btn btn-sm btn-info">
                                                                    <i class="bi bi-image"></i>
                                                                </a>
                                                            <?php else: ?>
                                                                <span class="text-muted">-</span>
                                                            <?php endif; ?>
                                                        </td>
                                                    </tr>
                                                <?php endforeach; ?>
                                            <?php else: ?>
                                                <tr>
                                                    <td colspan="5" class="text-center text-muted">No hay asignaciones</td>
                                                </tr>
                                            <?php endif; ?>
                                        </tbody>
                                    </table>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </main>
        </div>
    </div>
    
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Formatear monto con separadores de miles
        document.getElementById('monto').addEventListener('input', function(e) {
            let value = e.target.value.replace(/\D/g, '');
            e.target.value = value.replace(/\B(?=(\d{3})+(?!\d))/g, '.');
        });
    </script>
</body>
</html>

