<?php
/**
 * Sistema Médico - Ver Detalle de Factura
 */
require_once '../includes/config.php';
require_once '../includes/database.php';
require_once '../includes/auth.php';
require_once '../includes/permissions.php';

Permissions::requirePermission('CreateInvoices');

$page_title = 'Detalle de Factura';

$database = new Database();
$db = $database->getConnection();

$factura_id = $_GET['id'] ?? 0;

// Obtener factura
$query = "SELECT f.*, 
          CONCAT(p.nombre, ' ', p.apellido) as paciente_nombre,
          p.cedula as paciente_cedula,
          p.telefono as paciente_telefono,
          p.domicilio as paciente_domicilio,
          u.nombre_completo as usuario_nombre
          FROM facturas f
          INNER JOIN pacientes p ON f.paciente_id = p.id
          INNER JOIN usuarios u ON f.usuario_id = u.id
          WHERE f.id = :id";

$stmt = $db->prepare($query);
$stmt->bindParam(':id', $factura_id);
$stmt->execute();
$factura = $stmt->fetch();

if (!$factura) {
    header('Location: facturas.php?error=no_encontrada');
    exit();
}

// Obtener productos de la factura
$query = "SELECT fp.*, p.nombre, p.codigo
          FROM factura_productos fp
          INNER JOIN productos p ON fp.producto_id = p.id
          WHERE fp.factura_id = :factura_id";
$stmt = $db->prepare($query);
$stmt->bindParam(':factura_id', $factura_id);
$stmt->execute();
$productos = $stmt->fetchAll();

// Obtener servicios de la factura
$query = "SELECT fs.*, s.nombre, s.codigo
          FROM factura_servicios fs
          INNER JOIN servicios s ON fs.servicio_id = s.id
          WHERE fs.factura_id = :factura_id";
$stmt = $db->prepare($query);
$stmt->bindParam(':factura_id', $factura_id);
$stmt->execute();
$servicios = $stmt->fetchAll();

include '../includes/header.php';
?>

<div class="card">
    <div class="card-header" style="display: flex; justify-content: space-between; align-items: center; flex-wrap: wrap;">
        <h2>Factura <?php echo htmlspecialchars($factura['numero_factura']); ?></h2>
        <div style="display: flex; gap: 10px;">
            <button onclick="window.print()" class="btn btn-primary">🖨️ Imprimir</button>
            <a href="facturas.php" class="btn btn-secondary">← Volver</a>
        </div>
    </div>
    <div class="card-body">
        <?php if (isset($_GET['success'])): ?>
            <div class="alert alert-success">
                ✓ Factura creada exitosamente
            </div>
        <?php endif; ?>

        <!-- Información de la factura -->
        <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(300px, 1fr)); gap: 20px; margin-bottom: 30px;">
            <div>
                <h3 style="color: var(--medical-blue); margin-bottom: 15px;">Información del Paciente</h3>
                <p><strong>Nombre:</strong> <?php echo htmlspecialchars($factura['paciente_nombre']); ?></p>
                <p><strong>Cédula:</strong> <?php echo htmlspecialchars($factura['paciente_cedula']); ?></p>
                <p><strong>Teléfono:</strong> <?php echo htmlspecialchars($factura['paciente_telefono'] ?? '-'); ?></p>
                <p><strong>Domicilio:</strong> <?php echo htmlspecialchars($factura['paciente_domicilio'] ?? '-'); ?></p>
            </div>
            
            <div>
                <h3 style="color: var(--medical-blue); margin-bottom: 15px;">Información de la Factura</h3>
                <p><strong>Número:</strong> <?php echo htmlspecialchars($factura['numero_factura']); ?></p>
                <p><strong>Fecha Emisión:</strong> <?php echo date('d/m/Y H:i', strtotime($factura['fecha_emision'])); ?></p>
                <p><strong>Emitida por:</strong> <?php echo htmlspecialchars($factura['usuario_nombre']); ?></p>
                <p><strong>Estado:</strong> 
                    <?php
                    $color = 'gray';
                    if ($factura['estado'] == 'pagada') $color = 'green';
                    elseif ($factura['estado'] == 'pendiente') $color = 'orange';
                    elseif ($factura['estado'] == 'anulada') $color = 'red';
                    ?>
                    <span style="color: <?php echo $color; ?>; font-weight: bold;">
                        <?php echo strtoupper($factura['estado']); ?>
                    </span>
                </p>
                <p><strong>Método de Pago:</strong> <?php echo ucfirst($factura['metodo_pago']); ?></p>
                <?php if ($factura['fecha_pago']): ?>
                    <p><strong>Fecha de Pago:</strong> <?php echo date('d/m/Y H:i', strtotime($factura['fecha_pago'])); ?></p>
                <?php endif; ?>
            </div>
        </div>

        <!-- Detalle de productos -->
        <?php if (count($productos) > 0): ?>
            <h3 style="color: var(--medical-blue); margin-top: 30px; margin-bottom: 15px;">Productos</h3>
            <div class="table-responsive">
                <table>
                    <thead>
                        <tr>
                            <th>Código</th>
                            <th>Producto</th>
                            <th>Cantidad</th>
                            <th>Precio Unitario</th>
                            <th>Subtotal</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($productos as $producto): ?>
                            <tr>
                                <td><?php echo htmlspecialchars($producto['codigo']); ?></td>
                                <td><?php echo htmlspecialchars($producto['nombre']); ?></td>
                                <td><?php echo $producto['cantidad']; ?></td>
                                <td>₲ <?php echo number_format($producto['precio_unitario'], 0, ',', '.'); ?></td>
                                <td><strong>₲ <?php echo number_format($producto['subtotal'], 0, ',', '.'); ?></strong></td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        <?php endif; ?>

        <!-- Detalle de servicios -->
        <?php if (count($servicios) > 0): ?>
            <h3 style="color: var(--medical-blue); margin-top: 30px; margin-bottom: 15px;">Servicios</h3>
            <div class="table-responsive">
                <table>
                    <thead>
                        <tr>
                            <th>Código</th>
                            <th>Servicio</th>
                            <th>Cantidad</th>
                            <th>Precio Unitario</th>
                            <th>Subtotal</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($servicios as $servicio): ?>
                            <tr>
                                <td><?php echo htmlspecialchars($servicio['codigo']); ?></td>
                                <td><?php echo htmlspecialchars($servicio['nombre']); ?></td>
                                <td><?php echo $servicio['cantidad']; ?></td>
                                <td>₲ <?php echo number_format($servicio['precio_unitario'], 0, ',', '.'); ?></td>
                                <td><strong>₲ <?php echo number_format($servicio['subtotal'], 0, ',', '.'); ?></strong></td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        <?php endif; ?>

        <!-- Observaciones -->
        <?php if ($factura['observaciones']): ?>
            <div style="margin-top: 30px;">
                <h3 style="color: var(--medical-blue); margin-bottom: 10px;">Observaciones</h3>
                <p style="background: var(--light-color); padding: 15px; border-radius: 5px;">
                    <?php echo nl2br(htmlspecialchars($factura['observaciones'])); ?>
                </p>
            </div>
        <?php endif; ?>

        <!-- Totales -->
        <div style="background: var(--medical-light-blue); padding: 25px; border-radius: 8px; margin-top: 30px;">
            <div style="display: flex; justify-content: space-between; margin-bottom: 10px;">
                <span style="font-size: 18px;">Subtotal:</span>
                <span style="font-size: 18px; font-weight: bold;">₲ <?php echo number_format($factura['subtotal'], 0, ',', '.'); ?></span>
            </div>
            <?php if ($factura['descuento'] > 0): ?>
                <div style="display: flex; justify-content: space-between; margin-bottom: 10px;">
                    <span style="font-size: 18px;">Descuento:</span>
                    <span style="font-size: 18px; font-weight: bold; color: red;">- ₲ <?php echo number_format($factura['descuento'], 0, ',', '.'); ?></span>
                </div>
            <?php endif; ?>
            <?php if ($factura['impuesto'] > 0): ?>
                <div style="display: flex; justify-content: space-between; margin-bottom: 10px;">
                    <span style="font-size: 18px;">Impuesto:</span>
                    <span style="font-size: 18px; font-weight: bold;">₲ <?php echo number_format($factura['impuesto'], 0, ',', '.'); ?></span>
                </div>
            <?php endif; ?>
            <hr style="border: 1px solid var(--medical-blue); margin: 15px 0;">
            <div style="display: flex; justify-content: space-between;">
                <span style="font-size: 24px; font-weight: bold; color: var(--medical-blue);">TOTAL:</span>
                <span style="font-size: 24px; font-weight: bold; color: var(--medical-blue);">₲ <?php echo number_format($factura['total'], 0, ',', '.'); ?></span>
            </div>
        </div>
    </div>
</div>

<style>
@media print {
    .sidebar, .menu-toggle, .sidebar-overlay, .btn, .alert {
        display: none !important;
    }
    .main-content {
        margin-left: 0 !important;
        width: 100% !important;
    }
    .card {
        box-shadow: none !important;
        border: 1px solid #000;
    }
}
</style>

<?php include '../includes/footer.php'; ?>
