<?php
/**
 * Sistema Médico - Citas de Urgencia
 */
require_once '../includes/config.php';
require_once '../includes/database.php';
require_once '../includes/auth.php';

$page_title = 'Citas de Urgencia';

$database = new Database();
$db = $database->getConnection();

// Obtener médicos
$query = "SELECT id, nombre_completo FROM usuarios WHERE activo = 1 ORDER BY nombre_completo";
$stmt = $db->query($query);
$medicos = $stmt->fetchAll();

$mensaje = '';
$tipo_mensaje = '';
$busqueda_realizada = false;
$paciente = null;

// Buscar paciente
if ($_SERVER['REQUEST_METHOD'] === 'GET' && isset($_GET['buscar'])) {
    $termino = trim($_GET['termino'] ?? '');
    
    if (!empty($termino)) {
        $query = "SELECT * FROM pacientes 
                  WHERE cedula = :termino 
                  OR nombre LIKE :termino_like 
                  OR apellido LIKE :termino_like
                  LIMIT 1";
        
        $stmt = $db->prepare($query);
        $stmt->bindParam(':termino', $termino);
        $termino_like = "%{$termino}%";
        $stmt->bindParam(':termino_like', $termino_like);
        $stmt->execute();
        
        $paciente = $stmt->fetch();
        $busqueda_realizada = true;
    }
}

// Registrar cita de urgencia
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['registrar_urgencia'])) {
    $paciente_id = $_POST['paciente_id'] ?? 0;
    $medico_id = $_POST['medico_id'] ?? 0;
    $motivo = trim($_POST['motivo'] ?? '');
    $observaciones = trim($_POST['observaciones'] ?? '');

    try {
        $current_user = Auth::getCurrentUser();
        
        // Fecha y hora actual para urgencia
        $fecha_hora_cita = date('Y-m-d H:i:s');

        $query = "INSERT INTO citas (paciente_id, medico_id, fecha_cita, tipo, motivo, observaciones, estado) 
                  VALUES (:paciente_id, :medico_id, :fecha_cita, 'urgencia', :motivo, :observaciones, 'confirmada')";
        
        $stmt = $db->prepare($query);
        $stmt->bindParam(':paciente_id', $paciente_id);
        $stmt->bindParam(':medico_id', $medico_id);
        $stmt->bindParam(':fecha_cita', $fecha_hora_cita);
        $stmt->bindParam(':motivo', $motivo);
        $stmt->bindParam(':observaciones', $observaciones);
        $stmt->execute();

        $mensaje = 'Cita de urgencia registrada exitosamente';
        $tipo_mensaje = 'success';

        // Redirigir a la ficha del paciente después de 2 segundos
        header("refresh:2;url=ficha_paciente.php?id=$paciente_id");

    } catch (Exception $e) {
        $mensaje = 'Error al registrar cita de urgencia: ' . $e->getMessage();
        $tipo_mensaje = 'error';
    }
}

// Obtener citas de urgencia recientes
$query = "SELECT c.*, p.nombre, p.apellido, p.cedula, u.nombre_completo as medico
          FROM citas c
          INNER JOIN pacientes p ON c.paciente_id = p.id
          INNER JOIN usuarios u ON c.medico_id = u.id
          WHERE c.tipo = 'urgencia'
          ORDER BY c.fecha_cita DESC
          LIMIT 20";

$stmt = $db->query($query);
$urgencias_recientes = $stmt->fetchAll();

include '../includes/header.php';
?>

<div class="alert alert-warning">
    <strong>⚠️ ATENCIÓN:</strong> Esta sección es para registrar citas de urgencia que requieren atención inmediata.
</div>

<div class="card">
    <div class="card-header">
        <h2>Registrar Cita de Urgencia</h2>
    </div>
    <div class="card-body">
        <?php if ($mensaje): ?>
            <div class="alert alert-<?php echo $tipo_mensaje; ?>">
                <?php echo htmlspecialchars($mensaje); ?>
            </div>
        <?php endif; ?>

        <!-- Buscar paciente -->
        <h3 style="color: var(--medical-blue); margin-bottom: 15px;">1. Buscar Paciente</h3>
        <form method="GET" action="" class="search-box">
            <input type="text" 
                   name="termino" 
                   placeholder="Buscar por cédula, nombre o apellido..." 
                   value="<?php echo htmlspecialchars($_GET['termino'] ?? ''); ?>"
                   required>
            <button type="submit" name="buscar" class="btn btn-primary">🔍 Buscar</button>
        </form>

        <?php if ($busqueda_realizada): ?>
            <?php if ($paciente): ?>
                <!-- Información del paciente encontrado -->
                <div class="patient-info" style="margin-top: 20px;">
                    <h3>Paciente Encontrado</h3>
                    <div class="patient-details">
                        <div class="detail-item">
                            <span class="detail-label">Nombre Completo</span>
                            <span class="detail-value"><?php echo htmlspecialchars($paciente['nombre'] . ' ' . $paciente['apellido']); ?></span>
                        </div>
                        <div class="detail-item">
                            <span class="detail-label">Cédula</span>
                            <span class="detail-value"><?php echo htmlspecialchars($paciente['cedula']); ?></span>
                        </div>
                        <div class="detail-item">
                            <span class="detail-label">Teléfono</span>
                            <span class="detail-value"><?php echo htmlspecialchars($paciente['telefono']); ?></span>
                        </div>
                    </div>
                </div>

                <?php
                // Verificar alertas
                $query = "SELECT * FROM alertas_medicas WHERE paciente_id = :id AND activo = 1";
                $stmt = $db->prepare($query);
                $stmt->bindParam(':id', $paciente['id']);
                $stmt->execute();
                $alertas = $stmt->fetchAll();

                if (count($alertas) > 0):
                ?>
                    <div class="alert-emergency" style="margin-top: 20px;">
                        <strong>⚠️ ALERTAS MÉDICAS IMPORTANTES ⚠️</strong>
                        <ul style="margin: 10px 0 0 20px;">
                            <?php foreach ($alertas as $alerta): ?>
                                <li>
                                    <strong><?php echo strtoupper(str_replace('_', ' ', $alerta['tipo'])); ?>:</strong>
                                    <?php echo htmlspecialchars($alerta['descripcion']); ?>
                                </li>
                            <?php endforeach; ?>
                        </ul>
                    </div>
                <?php endif; ?>

                <!-- Formulario de urgencia -->
                <h3 style="color: var(--medical-blue); margin: 30px 0 15px;">2. Registrar Urgencia</h3>
                <form method="POST" action="">
                    <input type="hidden" name="paciente_id" value="<?php echo $paciente['id']; ?>">

                    <div class="form-group">
                        <label for="medico_id">Médico que Atenderá *</label>
                        <select id="medico_id" name="medico_id" required>
                            <option value="">Seleccione un médico...</option>
                            <?php foreach ($medicos as $medico): ?>
                                <option value="<?php echo $medico['id']; ?>">
                                    <?php echo htmlspecialchars($medico['nombre_completo']); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>

                    <div class="form-group">
                        <label for="motivo">Motivo de la Urgencia *</label>
                        <textarea id="motivo" name="motivo" rows="4" required placeholder="Describa la situación de urgencia..."></textarea>
                    </div>

                    <div class="form-group">
                        <label for="observaciones">Observaciones</label>
                        <textarea id="observaciones" name="observaciones" rows="3"></textarea>
                    </div>

                    <div style="display: flex; gap: 15px; margin-top: 20px;">
                        <button type="submit" name="registrar_urgencia" class="btn btn-danger">🚨 Registrar Urgencia</button>
                        <a href="citas_urgencia.php" class="btn btn-secondary">Cancelar</a>
                    </div>
                </form>

            <?php else: ?>
                <div class="alert alert-error" style="margin-top: 20px;">
                    No se encontró ningún paciente con el término de búsqueda: <strong><?php echo htmlspecialchars($_GET['termino']); ?></strong>
                </div>
                <a href="nuevo_paciente.php" class="btn btn-success">➕ Registrar Nuevo Paciente</a>
            <?php endif; ?>
        <?php endif; ?>
    </div>
</div>

<!-- Urgencias recientes -->
<div class="card">
    <div class="card-header">
        <h2>Urgencias Recientes</h2>
    </div>
    <div class="card-body">
        <?php if (count($urgencias_recientes) > 0): ?>
            <div class="table-responsive">
                <table>
                    <thead>
                        <tr>
                            <th>Fecha y Hora</th>
                            <th>Paciente</th>
                            <th>Cédula</th>
                            <th>Médico</th>
                            <th>Motivo</th>
                            <th>Estado</th>
                            <th>Acciones</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($urgencias_recientes as $urgencia): ?>
                            <tr>
                                <td><?php echo date('d/m/Y H:i', strtotime($urgencia['fecha_cita'])); ?></td>
                                <td><?php echo htmlspecialchars($urgencia['nombre'] . ' ' . $urgencia['apellido']); ?></td>
                                <td><?php echo htmlspecialchars($urgencia['cedula']); ?></td>
                                <td><?php echo htmlspecialchars($urgencia['medico']); ?></td>
                                <td><?php echo htmlspecialchars(substr($urgencia['motivo'], 0, 50)) . (strlen($urgencia['motivo']) > 50 ? '...' : ''); ?></td>
                                <td>
                                    <?php
                                    $estado_class = '';
                                    switch ($urgencia['estado']) {
                                        case 'confirmada':
                                            $estado_class = 'color: var(--info-color);';
                                            break;
                                        case 'completada':
                                            $estado_class = 'color: var(--success-color);';
                                            break;
                                        default:
                                            $estado_class = 'color: var(--warning-color);';
                                    }
                                    echo '<span style="' . $estado_class . ' font-weight: bold;">' . ucfirst($urgencia['estado']) . '</span>';
                                    ?>
                                </td>
                                <td>
                                    <a href="ficha_paciente.php?id=<?php echo $urgencia['paciente_id']; ?>" class="btn btn-primary" style="padding: 6px 12px; font-size: 12px;">
                                        Ver Paciente
                                    </a>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        <?php else: ?>
            <p>No hay urgencias registradas recientemente.</p>
        <?php endif; ?>
    </div>
</div>

<?php include '../includes/footer.php'; ?>

