<?php
/**
 * Sistema Médico - Gestión de Servicios (Solo Administradores)
 */
require_once '../../includes/config.php';
require_once '../../includes/database.php';
require_once '../../includes/auth.php';
require_once '../../includes/permissions.php';

// Verificar permisos de administrador
Permissions::requirePermission('ManageProductsServices');

$page_title = 'Gestión de Servicios';

$database = new Database();
$db = $database->getConnection();

$mensaje = '';
$tipo_mensaje = '';

// Procesar acciones
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['crear_servicio'])) {
        // Crear servicio
        $categoria_id = $_POST['categoria_id'] ?? null;
        $codigo = trim($_POST['codigo'] ?? '');
        $nombre = trim($_POST['nombre'] ?? '');
        $descripcion = trim($_POST['descripcion'] ?? '');
        $precio = $_POST['precio'] ?? 0;
        $duracion_estimada = $_POST['duracion_estimada'] ?? null;

        try {
            $query = "INSERT INTO servicios (categoria_id, codigo, nombre, descripcion, precio, duracion_estimada) 
                      VALUES (:categoria_id, :codigo, :nombre, :descripcion, :precio, :duracion_estimada)";
            
            $stmt = $db->prepare($query);
            $stmt->bindParam(':categoria_id', $categoria_id);
            $stmt->bindParam(':codigo', $codigo);
            $stmt->bindParam(':nombre', $nombre);
            $stmt->bindParam(':descripcion', $descripcion);
            $stmt->bindParam(':precio', $precio);
            $stmt->bindParam(':duracion_estimada', $duracion_estimada);
            $stmt->execute();

            $mensaje = 'Servicio creado exitosamente';
            $tipo_mensaje = 'success';
        } catch (Exception $e) {
            $mensaje = 'Error al crear servicio: ' . $e->getMessage();
            $tipo_mensaje = 'error';
        }
    }
    elseif (isset($_POST['editar_servicio'])) {
        // Editar servicio
        $id = $_POST['servicio_id'] ?? 0;
        $categoria_id = $_POST['categoria_id'] ?? null;
        $codigo = trim($_POST['codigo'] ?? '');
        $nombre = trim($_POST['nombre'] ?? '');
        $descripcion = trim($_POST['descripcion'] ?? '');
        $precio_nuevo = $_POST['precio'] ?? 0;
        $duracion_estimada = $_POST['duracion_estimada'] ?? null;

        try {
            // Obtener precio anterior para historial
            $query = "SELECT precio FROM servicios WHERE id = :id";
            $stmt = $db->prepare($query);
            $stmt->bindParam(':id', $id);
            $stmt->execute();
            $servicio_actual = $stmt->fetch();

            $db->beginTransaction();

            // Actualizar servicio
            $query = "UPDATE servicios 
                      SET categoria_id = :categoria_id, codigo = :codigo, nombre = :nombre, 
                          descripcion = :descripcion, precio = :precio, duracion_estimada = :duracion_estimada
                      WHERE id = :id";
            
            $stmt = $db->prepare($query);
            $stmt->bindParam(':categoria_id', $categoria_id);
            $stmt->bindParam(':codigo', $codigo);
            $stmt->bindParam(':nombre', $nombre);
            $stmt->bindParam(':descripcion', $descripcion);
            $stmt->bindParam(':precio', $precio_nuevo);
            $stmt->bindParam(':duracion_estimada', $duracion_estimada);
            $stmt->bindParam(':id', $id);
            $stmt->execute();

            // Registrar cambio de precio si cambió
            if ($servicio_actual && $servicio_actual['precio'] != $precio_nuevo) {
                $current_user = Auth::getCurrentUser();
                $query = "INSERT INTO historial_precios (tipo, item_id, precio_anterior, precio_nuevo, usuario_id) 
                          VALUES ('servicio', :item_id, :precio_anterior, :precio_nuevo, :usuario_id)";
                $stmt = $db->prepare($query);
                $stmt->bindParam(':item_id', $id);
                $stmt->bindParam(':precio_anterior', $servicio_actual['precio']);
                $stmt->bindParam(':precio_nuevo', $precio_nuevo);
                $stmt->bindParam(':usuario_id', $current_user['id']);
                $stmt->execute();
            }

            $db->commit();

            $mensaje = 'Servicio actualizado exitosamente';
            $tipo_mensaje = 'success';
        } catch (Exception $e) {
            $db->rollBack();
            $mensaje = 'Error al actualizar servicio: ' . $e->getMessage();
            $tipo_mensaje = 'error';
        }
    }
    elseif (isset($_POST['eliminar_servicio'])) {
        // Desactivar servicio
        $id = $_POST['servicio_id'] ?? 0;

        try {
            $query = "UPDATE servicios SET activo = 0 WHERE id = :id";
            $stmt = $db->prepare($query);
            $stmt->bindParam(':id', $id);
            $stmt->execute();

            $mensaje = 'Servicio desactivado exitosamente';
            $tipo_mensaje = 'success';
        } catch (Exception $e) {
            $mensaje = 'Error al desactivar servicio: ' . $e->getMessage();
            $tipo_mensaje = 'error';
        }
    }
    elseif (isset($_POST['activar_servicio'])) {
        // Activar servicio
        $id = $_POST['servicio_id'] ?? 0;

        try {
            $query = "UPDATE servicios SET activo = 1 WHERE id = :id";
            $stmt = $db->prepare($query);
            $stmt->bindParam(':id', $id);
            $stmt->execute();

            $mensaje = 'Servicio activado exitosamente';
            $tipo_mensaje = 'success';
        } catch (Exception $e) {
            $mensaje = 'Error al activar servicio: ' . $e->getMessage();
            $tipo_mensaje = 'error';
        }
    }
}

// Obtener categorías de servicios
$query = "SELECT * FROM categorias WHERE tipo = 'servicio' AND activo = 1 ORDER BY nombre";
$stmt = $db->query($query);
$categorias = $stmt->fetchAll();

// Obtener servicios
$filtro = $_GET['filtro'] ?? 'activos';
$where_clause = $filtro == 'todos' ? '' : 'WHERE s.activo = 1';

$query = "SELECT s.*, c.nombre as categoria_nombre
          FROM servicios s
          LEFT JOIN categorias c ON s.categoria_id = c.id
          $where_clause
          ORDER BY s.nombre";

$stmt = $db->query($query);
$servicios = $stmt->fetchAll();

include '../../includes/header.php';
?>

<div class="card">
    <div class="card-header">
        <h2>Gestión de Servicios</h2>
    </div>
    <div class="card-body">
        <?php if ($mensaje): ?>
            <div class="alert alert-<?php echo $tipo_mensaje; ?>">
                <?php echo htmlspecialchars($mensaje); ?>
            </div>
        <?php endif; ?>

        <!-- Botones de acción -->
        <div style="display: flex; gap: 10px; margin-bottom: 20px; flex-wrap: wrap;">
            <button onclick="document.getElementById('modalCrear').style.display='block'" class="btn btn-success">
                ➕ Nuevo Servicio
            </button>
            <a href="?filtro=activos" class="btn <?php echo $filtro == 'activos' ? 'btn-primary' : 'btn-secondary'; ?>">
                Activos
            </a>
            <a href="?filtro=todos" class="btn <?php echo $filtro == 'todos' ? 'btn-primary' : 'btn-secondary'; ?>">
                Todos
            </a>
        </div>

        <!-- Tabla de servicios -->
        <div class="table-responsive">
            <table>
                <thead>
                    <tr>
                        <th>Código</th>
                        <th>Nombre</th>
                        <th>Categoría</th>
                        <th>Precio</th>
                        <th>Duración (min)</th>
                        <th>Estado</th>
                        <th>Acciones</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($servicios as $servicio): ?>
                        <tr style="<?php echo !$servicio['activo'] ? 'opacity: 0.5;' : ''; ?>">
                            <td><?php echo htmlspecialchars($servicio['codigo']); ?></td>
                            <td><?php echo htmlspecialchars($servicio['nombre']); ?></td>
                            <td><?php echo htmlspecialchars($servicio['categoria_nombre'] ?? 'Sin categoría'); ?></td>
                            <td>₲ <?php echo number_format($servicio['precio'], 0, ',', '.'); ?></td>
                            <td><?php echo $servicio['duracion_estimada'] ?? '-'; ?></td>
                            <td>
                                <?php if ($servicio['activo']): ?>
                                    <span style="color: green;">✓ Activo</span>
                                <?php else: ?>
                                    <span style="color: red;">✗ Inactivo</span>
                                <?php endif; ?>
                            </td>
                            <td>
                                <button onclick="editarServicio(<?php echo htmlspecialchars(json_encode($servicio)); ?>)" 
                                        class="btn btn-primary" style="padding: 6px 12px; font-size: 12px;">
                                    Editar
                                </button>
                                <?php if ($servicio['activo']): ?>
                                    <form method="POST" style="display: inline;">
                                        <input type="hidden" name="servicio_id" value="<?php echo $servicio['id']; ?>">
                                        <button type="submit" name="eliminar_servicio" class="btn btn-danger" 
                                                style="padding: 6px 12px; font-size: 12px;"
                                                onclick="return confirm('¿Desactivar este servicio?')">
                                            Desactivar
                                        </button>
                                    </form>
                                <?php else: ?>
                                    <form method="POST" style="display: inline;">
                                        <input type="hidden" name="servicio_id" value="<?php echo $servicio['id']; ?>">
                                        <button type="submit" name="activar_servicio" class="btn btn-success" 
                                                style="padding: 6px 12px; font-size: 12px;">
                                            Activar
                                        </button>
                                    </form>
                                <?php endif; ?>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
    </div>
</div>

<!-- Modal Crear Servicio -->
<div id="modalCrear" style="display: none; position: fixed; z-index: 1000; left: 0; top: 0; width: 100%; height: 100%; background-color: rgba(0,0,0,0.5); overflow: auto;">
    <div style="background-color: white; margin: 5% auto; padding: 30px; width: 90%; max-width: 600px; border-radius: 10px;">
        <h2 style="color: var(--medical-blue); margin-bottom: 20px;">Nuevo Servicio</h2>
        <form method="POST">
            <div class="form-group">
                <label>Categoría</label>
                <select name="categoria_id">
                    <option value="">Sin categoría</option>
                    <?php foreach ($categorias as $cat): ?>
                        <option value="<?php echo $cat['id']; ?>"><?php echo htmlspecialchars($cat['nombre']); ?></option>
                    <?php endforeach; ?>
                </select>
            </div>
            <div class="form-group">
                <label>Código *</label>
                <input type="text" name="codigo" required>
            </div>
            <div class="form-group">
                <label>Nombre *</label>
                <input type="text" name="nombre" required>
            </div>
            <div class="form-group">
                <label>Descripción</label>
                <textarea name="descripcion" rows="3"></textarea>
            </div>
            <div class="form-group">
                <label>Precio (₲) *</label>
                <input type="number" name="precio" step="0.01" required>
            </div>
            <div class="form-group">
                <label>Duración Estimada (minutos)</label>
                <input type="number" name="duracion_estimada">
            </div>
            <div style="display: flex; gap: 10px; margin-top: 20px;">
                <button type="submit" name="crear_servicio" class="btn btn-success">Crear Servicio</button>
                <button type="button" onclick="document.getElementById('modalCrear').style.display='none'" class="btn btn-secondary">Cancelar</button>
            </div>
        </form>
    </div>
</div>

<!-- Modal Editar Servicio -->
<div id="modalEditar" style="display: none; position: fixed; z-index: 1000; left: 0; top: 0; width: 100%; height: 100%; background-color: rgba(0,0,0,0.5); overflow: auto;">
    <div style="background-color: white; margin: 5% auto; padding: 30px; width: 90%; max-width: 600px; border-radius: 10px;">
        <h2 style="color: var(--medical-blue); margin-bottom: 20px;">Editar Servicio</h2>
        <form method="POST" id="formEditar">
            <input type="hidden" name="servicio_id" id="edit_id">
            <div class="form-group">
                <label>Categoría</label>
                <select name="categoria_id" id="edit_categoria_id">
                    <option value="">Sin categoría</option>
                    <?php foreach ($categorias as $cat): ?>
                        <option value="<?php echo $cat['id']; ?>"><?php echo htmlspecialchars($cat['nombre']); ?></option>
                    <?php endforeach; ?>
                </select>
            </div>
            <div class="form-group">
                <label>Código *</label>
                <input type="text" name="codigo" id="edit_codigo" required>
            </div>
            <div class="form-group">
                <label>Nombre *</label>
                <input type="text" name="nombre" id="edit_nombre" required>
            </div>
            <div class="form-group">
                <label>Descripción</label>
                <textarea name="descripcion" id="edit_descripcion" rows="3"></textarea>
            </div>
            <div class="form-group">
                <label>Precio (₲) *</label>
                <input type="number" name="precio" id="edit_precio" step="0.01" required>
            </div>
            <div class="form-group">
                <label>Duración Estimada (minutos)</label>
                <input type="number" name="duracion_estimada" id="edit_duracion_estimada">
            </div>
            <div style="display: flex; gap: 10px; margin-top: 20px;">
                <button type="submit" name="editar_servicio" class="btn btn-success">Guardar Cambios</button>
                <button type="button" onclick="document.getElementById('modalEditar').style.display='none'" class="btn btn-secondary">Cancelar</button>
            </div>
        </form>
    </div>
</div>

<script>
function editarServicio(servicio) {
    document.getElementById('edit_id').value = servicio.id;
    document.getElementById('edit_categoria_id').value = servicio.categoria_id || '';
    document.getElementById('edit_codigo').value = servicio.codigo;
    document.getElementById('edit_nombre').value = servicio.nombre;
    document.getElementById('edit_descripcion').value = servicio.descripcion || '';
    document.getElementById('edit_precio').value = servicio.precio;
    document.getElementById('edit_duracion_estimada').value = servicio.duracion_estimada || '';
    document.getElementById('modalEditar').style.display = 'block';
}
</script>

<?php include '../../includes/footer.php'; ?>
