// Tutorial Interactivo - Sistema Médico Centro Santa Salud
// JavaScript para navegación y funcionalidades interactivas

document.addEventListener('DOMContentLoaded', function() {
    // Variables globales
    let currentStep = 1;
    const totalSteps = 12;
    const completedSteps = new Set();

    // Elementos del DOM
    const navItems = document.querySelectorAll('.nav-item');
    const stepContents = document.querySelectorAll('.step-content');
    const btnPrev = document.getElementById('btnPrev');
    const btnNext = document.getElementById('btnNext');
    const btnFinish = document.getElementById('btnFinish');
    const progressBar = document.getElementById('progressBar');
    const currentStepEl = document.getElementById('currentStep');
    const totalStepsEl = document.getElementById('totalSteps');

    // Inicializar
    init();

    function init() {
        // Establecer total de pasos
        totalStepsEl.textContent = totalSteps;
        
        // Event listeners para navegación lateral
        navItems.forEach(item => {
            item.addEventListener('click', function(e) {
                e.preventDefault();
                const step = parseInt(this.dataset.step);
                goToStep(step);
            });
        });

        // Event listeners para botones de navegación
        btnPrev.addEventListener('click', () => goToStep(currentStep - 1));
        btnNext.addEventListener('click', () => goToStep(currentStep + 1));
        btnFinish.addEventListener('click', finishTutorial);

        // Demo de login
        const demoLoginForm = document.getElementById('demoLoginForm');
        if (demoLoginForm) {
            demoLoginForm.addEventListener('submit', handleDemoLogin);
        }

        // Actualizar UI inicial
        updateUI();
    }

    function goToStep(step) {
        if (step < 1 || step > totalSteps) return;

        // Marcar paso actual como completado
        completedSteps.add(currentStep);

        // Actualizar paso actual
        currentStep = step;

        // Actualizar UI
        updateUI();

        // Scroll suave al inicio
        window.scrollTo({ top: 0, behavior: 'smooth' });
    }

    function updateUI() {
        // Actualizar contenido visible
        stepContents.forEach((content, index) => {
            if (index + 1 === currentStep) {
                content.classList.add('active');
            } else {
                content.classList.remove('active');
            }
        });

        // Actualizar navegación lateral
        navItems.forEach((item, index) => {
            const stepNum = index + 1;
            
            // Remover todas las clases
            item.classList.remove('active', 'completed');
            
            // Agregar clase activa
            if (stepNum === currentStep) {
                item.classList.add('active');
            }
            
            // Agregar clase completada
            if (completedSteps.has(stepNum)) {
                item.classList.add('completed');
            }
        });

        // Actualizar botones de navegación
        btnPrev.disabled = currentStep === 1;
        
        if (currentStep === totalSteps) {
            btnNext.style.display = 'none';
            btnFinish.style.display = 'flex';
        } else {
            btnNext.style.display = 'flex';
            btnFinish.style.display = 'none';
        }

        // Actualizar barra de progreso
        const progress = (currentStep / totalSteps) * 100;
        progressBar.style.width = progress + '%';
        currentStepEl.textContent = currentStep;

        // Cargar contenido dinámico si es necesario
        loadStepContent(currentStep);
    }

    function loadStepContent(step) {
        // Cargar contenido específico para cada paso
        switch(step) {
            case 4:
                loadPatientsModule();
                break;
            case 5:
                loadConsultationsModule();
                break;
            case 6:
                loadAppointmentsModule();
                break;
            case 7:
                loadBillingModule();
                break;
            case 8:
                loadProductsModule();
                break;
            case 9:
                loadServicesModule();
                break;
            case 10:
                loadReportsModule();
                break;
            case 11:
                loadExportModule();
                break;
            case 12:
                loadTipsModule();
                break;
        }
    }

    function handleDemoLogin(e) {
        e.preventDefault();
        
        const username = document.getElementById('demoUsername').value;
        const password = document.getElementById('demoPassword').value;
        const resultDiv = document.getElementById('loginDemoResult');

        // Simular validación
        if (username && password) {
            resultDiv.innerHTML = `
                <i class="fas fa-check-circle"></i>
                <strong>¡Inicio de sesión exitoso!</strong><br>
                Bienvenido, ${username}. Redirigiendo al Dashboard...
            `;
            resultDiv.className = 'login-demo-result success';

            // Simular redirección
            setTimeout(() => {
                resultDiv.style.display = 'none';
            }, 3000);
        }
    }

    function loadPatientsModule() {
        const stepContent = document.getElementById('step-4');
        if (stepContent.querySelector('.content-loaded')) return;

        const content = `
            <div class="content-loaded">
                <h3>Buscar Paciente</h3>
                <p>El sistema permite buscar pacientes de tres formas diferentes:</p>

                <div class="search-demo">
                    <div class="search-box">
                        <input type="text" placeholder="Buscar por cédula, nombre o apellido..." class="search-input">
                        <button class="btn-search"><i class="fas fa-search"></i> Buscar</button>
                    </div>
                </div>

                <div class="features-grid" style="margin-top: 30px;">
                    <div class="feature-card" style="background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);">
                        <i class="fas fa-id-card"></i>
                        <h4>Por Cédula</h4>
                        <p>Búsqueda exacta por número de documento</p>
                    </div>
                    <div class="feature-card" style="background: linear-gradient(135deg, #f093fb 0%, #f5576c 100%);">
                        <i class="fas fa-user"></i>
                        <h4>Por Nombre</h4>
                        <p>Búsqueda parcial por nombre del paciente</p>
                    </div>
                    <div class="feature-card" style="background: linear-gradient(135deg, #4facfe 0%, #00f2fe 100%);">
                        <i class="fas fa-users"></i>
                        <h4>Por Apellido</h4>
                        <p>Búsqueda parcial por apellido del paciente</p>
                    </div>
                </div>

                <h3 style="margin-top: 40px;">Nuevo Paciente</h3>
                <p>Para registrar un nuevo paciente, completa el formulario con los siguientes datos:</p>

                <div class="form-demo">
                    <div class="form-grid">
                        <div class="form-field">
                            <label><i class="fas fa-user"></i> Nombre</label>
                            <input type="text" placeholder="Juan" readonly>
                        </div>
                        <div class="form-field">
                            <label><i class="fas fa-user"></i> Apellido</label>
                            <input type="text" placeholder="Pérez" readonly>
                        </div>
                        <div class="form-field">
                            <label><i class="fas fa-id-card"></i> Cédula</label>
                            <input type="text" placeholder="1234567" readonly>
                        </div>
                        <div class="form-field">
                            <label><i class="fas fa-calendar"></i> Fecha de Nacimiento</label>
                            <input type="date" readonly>
                        </div>
                        <div class="form-field">
                            <label><i class="fas fa-phone"></i> Teléfono</label>
                            <input type="text" placeholder="+595 21 123-4567" readonly>
                        </div>
                        <div class="form-field">
                            <label><i class="fas fa-home"></i> Domicilio</label>
                            <input type="text" placeholder="Av. Principal 123" readonly>
                        </div>
                    </div>
                </div>

                <div class="warning-box" style="margin-top: 30px;">
                    <i class="fas fa-exclamation-triangle"></i>
                    <div>
                        <h4>Sistema de Alertas</h4>
                        <p>Si el paciente tiene alergias, resistencias a medicamentos o medicamentos permanentes, aparecerá una <strong>alerta naranja parpadeante</strong> en su ficha para advertir al personal médico.</p>
                    </div>
                </div>
            </div>
        `;

        const contentCard = stepContent.querySelector('.content-card');
        contentCard.innerHTML = content;
    }

    function loadConsultationsModule() {
        const stepContent = document.getElementById('step-5');
        if (stepContent.querySelector('.content-loaded')) return;

        const content = `
            <div class="content-loaded">
                <h3>Registro de Consulta Médica</h3>
                <p>Cada consulta médica incluye la siguiente información:</p>

                <div class="consultation-demo">
                    <div class="demo-section">
                        <h4><i class="fas fa-clipboard-list"></i> Motivo de la Visita</h4>
                        <textarea readonly style="width: 100%; padding: 10px; border-radius: 8px; border: 1px solid #ddd;">Dolor de cabeza persistente desde hace 3 días</textarea>
                    </div>

                    <div class="vitals-grid">
                        <div class="vital-card">
                            <i class="fas fa-weight"></i>
                            <h4>Peso</h4>
                            <input type="number" value="75" readonly> <span>kg</span>
                        </div>
                        <div class="vital-card">
                            <i class="fas fa-heartbeat"></i>
                            <h4>Presión Arterial</h4>
                            <input type="text" value="120/80" readonly> <span>mmHg</span>
                        </div>
                        <div class="vital-card">
                            <i class="fas fa-dumbbell"></i>
                            <h4>Masa Muscular</h4>
                            <input type="number" value="35" readonly> <span>%</span>
                        </div>
                    </div>

                    <div class="demo-section">
                        <h4><i class="fas fa-stethoscope"></i> Diagnóstico</h4>
                        <textarea readonly style="width: 100%; padding: 10px; border-radius: 8px; border: 1px solid #ddd;">Cefalea tensional. Se recomienda descanso y medicación analgésica.</textarea>
                    </div>

                    <div class="demo-section">
                        <h4><i class="fas fa-pills"></i> Tratamiento y Medicamentos</h4>
                        <div class="medication-list">
                            <div class="medication-item">
                                <i class="fas fa-prescription-bottle"></i>
                                <div>
                                    <strong>Paracetamol 500mg</strong>
                                    <p>1 comprimido cada 8 horas por 5 días</p>
                                </div>
                            </div>
                            <div class="medication-item">
                                <i class="fas fa-prescription-bottle"></i>
                                <div>
                                    <strong>Ibuprofeno 400mg</strong>
                                    <p>1 comprimido cada 12 horas si persiste el dolor</p>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

                <div class="tip-box" style="margin-top: 30px;">
                    <i class="fas fa-lightbulb"></i>
                    <div>
                        <h4>Historial Completo</h4>
                        <p>Todas las consultas quedan registradas en el historial del paciente. En la primera línea de la ficha aparecen los datos de la última visita (peso, presión, masa muscular) para referencia rápida.</p>
                    </div>
                </div>
            </div>
        `;

        const contentCard = stepContent.querySelector('.content-card');
        contentCard.innerHTML = content;
    }

    function loadAppointmentsModule() {
        const stepContent = document.getElementById('step-6');
        if (stepContent.querySelector('.content-loaded')) return;

        const content = `
            <div class="content-loaded">
                <h3>Agendar Cita</h3>
                <p>El sistema permite programar dos tipos de citas:</p>

                <div class="appointment-types">
                    <div class="appointment-card regular">
                        <div class="appointment-icon">
                            <i class="fas fa-calendar-check"></i>
                        </div>
                        <h4>Cita Regular</h4>
                        <p>Programada con anticipación para consultas de rutina</p>
                        <ul>
                            <li>Seleccionar fecha y hora</li>
                            <li>Elegir tipo de consulta</li>
                            <li>Agregar observaciones</li>
                        </ul>
                    </div>

                    <div class="appointment-card urgent">
                        <div class="appointment-icon">
                            <i class="fas fa-ambulance"></i>
                        </div>
                        <h4>Cita de Urgencia</h4>
                        <p>Atención inmediata para casos urgentes</p>
                        <ul>
                            <li>Registro instantáneo</li>
                            <li>Prioridad alta</li>
                            <li>Sin necesidad de fecha futura</li>
                        </ul>
                    </div>
                </div>

                <h3 style="margin-top: 40px;">Estados de Citas</h3>
                <div class="status-list">
                    <div class="status-item pending">
                        <i class="fas fa-clock"></i>
                        <div>
                            <strong>Pendiente</strong>
                            <p>Cita programada, esperando confirmación</p>
                        </div>
                    </div>
                    <div class="status-item confirmed">
                        <i class="fas fa-check-circle"></i>
                        <div>
                            <strong>Confirmada</strong>
                            <p>Cita confirmada por el paciente</p>
                        </div>
                    </div>
                    <div class="status-item completed">
                        <i class="fas fa-check-double"></i>
                        <div>
                            <strong>Completada</strong>
                            <p>Consulta realizada exitosamente</p>
                        </div>
                    </div>
                    <div class="status-item cancelled">
                        <i class="fas fa-times-circle"></i>
                        <div>
                            <strong>Cancelada</strong>
                            <p>Cita cancelada por el paciente o el centro</p>
                        </div>
                    </div>
                </div>

                <div class="info-box" style="margin-top: 30px;">
                    <i class="fas fa-info-circle"></i>
                    <div>
                        <h4>Gestión de Citas</h4>
                        <p>Puedes ver todas las citas en una lista filtrable por estado, fecha y paciente. También puedes actualizar el estado de las citas según avance el día.</p>
                    </div>
                </div>
            </div>
        `;

        const contentCard = stepContent.querySelector('.content-card');
        contentCard.innerHTML = content;
    }

    function loadBillingModule() {
        const stepContent = document.getElementById('step-7');
        if (stepContent.querySelector('.content-loaded')) return;

        const content = `
            <div class="content-loaded">
                <h3>Proceso de Facturación</h3>
                <p>La facturación es un proceso completo que incluye:</p>

                <div class="billing-steps">
                    <div class="billing-step">
                        <div class="step-num">1</div>
                        <div class="step-content-billing">
                            <h4>Seleccionar Paciente</h4>
                            <p>Buscar y seleccionar el paciente a facturar</p>
                        </div>
                    </div>
                    <div class="billing-step">
                        <div class="step-num">2</div>
                        <div class="step-content-billing">
                            <h4>Agregar Productos</h4>
                            <p>Seleccionar productos con control automático de stock</p>
                        </div>
                    </div>
                    <div class="billing-step">
                        <div class="step-num">3</div>
                        <div class="step-content-billing">
                            <h4>Agregar Servicios</h4>
                            <p>Seleccionar servicios médicos prestados</p>
                        </div>
                    </div>
                    <div class="billing-step">
                        <div class="step-num">4</div>
                        <div class="step-content-billing">
                            <h4>Método de Pago</h4>
                            <p>Elegir: Efectivo, Tarjeta, Transferencia o Cheque</p>
                        </div>
                    </div>
                    <div class="billing-step">
                        <div class="step-num">5</div>
                        <div class="step-content-billing">
                            <h4>Generar Factura</h4>
                            <p>El sistema genera automáticamente el número de factura</p>
                        </div>
                    </div>
                </div>

                <h3 style="margin-top: 40px;">Ejemplo de Factura</h3>
                <div class="invoice-demo">
                    <div class="invoice-header">
                        <h4>Factura N° FAC-20251201-0001</h4>
                        <p>Fecha: 01/12/2025</p>
                    </div>
                    <div class="invoice-patient">
                        <strong>Paciente:</strong> Juan Pérez<br>
                        <strong>Cédula:</strong> 1234567
                    </div>
                    <table class="invoice-table">
                        <thead>
                            <tr>
                                <th>Descripción</th>
                                <th>Cantidad</th>
                                <th>Precio Unit.</th>
                                <th>Subtotal</th>
                            </tr>
                        </thead>
                        <tbody>
                            <tr>
                                <td>Consulta General</td>
                                <td>1</td>
                                <td>₲ 150,000</td>
                                <td>₲ 150,000</td>
                            </tr>
                            <tr>
                                <td>Paracetamol 500mg</td>
                                <td>10</td>
                                <td>₲ 5,000</td>
                                <td>₲ 50,000</td>
                            </tr>
                        </tbody>
                        <tfoot>
                            <tr>
                                <td colspan="3"><strong>TOTAL</strong></td>
                                <td><strong>₲ 200,000</strong></td>
                            </tr>
                        </tfoot>
                    </table>
                    <div class="invoice-footer">
                        <strong>Método de Pago:</strong> Efectivo<br>
                        <strong>Estado:</strong> <span style="color: #4caf50;">Pagada</span>
                    </div>
                </div>

                <div class="warning-box" style="margin-top: 30px;">
                    <i class="fas fa-exclamation-triangle"></i>
                    <div>
                        <h4>Control de Stock</h4>
                        <p>Al generar una factura con productos, el stock se descuenta automáticamente. Si anulass una factura, el stock se restaura.</p>
                    </div>
                </div>
            </div>
        `;

        const contentCard = stepContent.querySelector('.content-card');
        contentCard.innerHTML = content;
    }

    function loadProductsModule() {
        const stepContent = document.getElementById('step-8');
        if (stepContent.querySelector('.content-loaded')) return;

        const content = `
            <div class="content-loaded">
                <h3>Gestión de Productos (Solo Administrador)</h3>
                <p>Los administradores pueden gestionar el catálogo completo de productos médicos:</p>

                <div class="crud-operations">
                    <div class="crud-card create">
                        <i class="fas fa-plus-circle"></i>
                        <h4>Crear Producto</h4>
                        <p>Agregar nuevos productos al inventario</p>
                    </div>
                    <div class="crud-card read">
                        <i class="fas fa-eye"></i>
                        <h4>Ver Productos</h4>
                        <p>Lista completa con filtros y búsqueda</p>
                    </div>
                    <div class="crud-card update">
                        <i class="fas fa-edit"></i>
                        <h4>Editar Producto</h4>
                        <p>Actualizar precios, stock y datos</p>
                    </div>
                    <div class="crud-card delete">
                        <i class="fas fa-trash-alt"></i>
                        <h4>Desactivar Producto</h4>
                        <p>Desactivación lógica (no elimina)</p>
                    </div>
                </div>

                <h3 style="margin-top: 40px;">Datos del Producto</h3>
                <div class="product-fields">
                    <div class="field-item">
                        <i class="fas fa-barcode"></i>
                        <div>
                            <strong>Código</strong>
                            <p>Código único del producto (ej: MED-001)</p>
                        </div>
                    </div>
                    <div class="field-item">
                        <i class="fas fa-tag"></i>
                        <div>
                            <strong>Nombre</strong>
                            <p>Nombre comercial del producto</p>
                        </div>
                    </div>
                    <div class="field-item">
                        <i class="fas fa-dollar-sign"></i>
                        <div>
                            <strong>Precio</strong>
                            <p>Precio unitario en Guaraníes (₲)</p>
                        </div>
                    </div>
                    <div class="field-item">
                        <i class="fas fa-boxes"></i>
                        <div>
                            <strong>Stock</strong>
                            <p>Cantidad disponible en inventario</p>
                        </div>
                    </div>
                    <div class="field-item">
                        <i class="fas fa-folder"></i>
                        <div>
                            <strong>Categoría</strong>
                            <p>Clasificación del producto</p>
                        </div>
                    </div>
                </div>

                <div class="tip-box" style="margin-top: 30px;">
                    <i class="fas fa-lightbulb"></i>
                    <div>
                        <h4>Historial de Precios</h4>
                        <p>El sistema mantiene un registro automático de todos los cambios de precios para auditoría y control.</p>
                    </div>
                </div>
            </div>
        `;

        const contentCard = stepContent.querySelector('.content-card');
        contentCard.innerHTML = content;
    }

    function loadServicesModule() {
        const stepContent = document.getElementById('step-9');
        if (stepContent.querySelector('.content-loaded')) return;

        const content = `
            <div class="content-loaded">
                <h3>Gestión de Servicios (Solo Administrador)</h3>
                <p>Similar a los productos, los servicios médicos se gestionan desde el panel de administrador:</p>

                <div class="services-examples">
                    <div class="service-card">
                        <i class="fas fa-user-md"></i>
                        <h4>Consulta General</h4>
                        <p class="service-price">₲ 150,000</p>
                        <p class="service-duration">30 minutos</p>
                    </div>
                    <div class="service-card">
                        <i class="fas fa-microscope"></i>
                        <h4>Análisis de Sangre</h4>
                        <p class="service-price">₲ 120,000</p>
                        <p class="service-duration">15 minutos</p>
                    </div>
                    <div class="service-card">
                        <i class="fas fa-heartbeat"></i>
                        <h4>Electrocardiograma</h4>
                        <p class="service-price">₲ 180,000</p>
                        <p class="service-duration">20 minutos</p>
                    </div>
                    <div class="service-card">
                        <i class="fas fa-x-ray"></i>
                        <h4>Ecografía</h4>
                        <p class="service-price">₲ 275,000</p>
                        <p class="service-duration">45 minutos</p>
                    </div>
                </div>

                <h3 style="margin-top: 40px;">Diferencias con Productos</h3>
                <div class="comparison-table">
                    <table>
                        <thead>
                            <tr>
                                <th>Característica</th>
                                <th>Productos</th>
                                <th>Servicios</th>
                            </tr>
                        </thead>
                        <tbody>
                            <tr>
                                <td><strong>Stock</strong></td>
                                <td><i class="fas fa-check" style="color: #4caf50;"></i> Sí</td>
                                <td><i class="fas fa-times" style="color: #f44336;"></i> No</td>
                            </tr>
                            <tr>
                                <td><strong>Duración</strong></td>
                                <td><i class="fas fa-times" style="color: #f44336;"></i> No</td>
                                <td><i class="fas fa-check" style="color: #4caf50;"></i> Sí</td>
                            </tr>
                            <tr>
                                <td><strong>Precio</strong></td>
                                <td><i class="fas fa-check" style="color: #4caf50;"></i> Sí</td>
                                <td><i class="fas fa-check" style="color: #4caf50;"></i> Sí</td>
                            </tr>
                            <tr>
                                <td><strong>Categoría</strong></td>
                                <td><i class="fas fa-check" style="color: #4caf50;"></i> Sí</td>
                                <td><i class="fas fa-check" style="color: #4caf50;"></i> Sí</td>
                            </tr>
                        </tbody>
                    </table>
                </div>

                <div class="info-box" style="margin-top: 30px;">
                    <i class="fas fa-info-circle"></i>
                    <div>
                        <h4>Facturación Combinada</h4>
                        <p>En una misma factura puedes incluir tanto productos como servicios. El sistema calcula automáticamente el total.</p>
                    </div>
                </div>
            </div>
        `;

        const contentCard = stepContent.querySelector('.content-card');
        contentCard.innerHTML = content;
    }

    function loadReportsModule() {
        const stepContent = document.getElementById('step-10');
        if (stepContent.querySelector('.content-loaded')) return;

        const content = `
            <div class="content-loaded">
                <h3>Reportes y Recaudación (Solo Administrador)</h3>
                <p>El módulo de reportes permite analizar la recaudación del centro médico en diferentes períodos:</p>

                <div class="period-selector">
                    <button class="period-btn active">Diario</button>
                    <button class="period-btn">Mensual</button>
                    <button class="period-btn">Anual</button>
                    <button class="period-btn">Personalizado</button>
                </div>

                <h3 style="margin-top: 40px;">Estadísticas Principales</h3>
                <div class="stats-grid">
                    <div class="stat-card purple">
                        <div class="stat-icon">
                            <i class="fas fa-file-invoice"></i>
                        </div>
                        <div class="stat-info">
                            <h4>Total Facturas</h4>
                            <p class="stat-number">156</p>
                        </div>
                    </div>
                    <div class="stat-card green">
                        <div class="stat-icon">
                            <i class="fas fa-check-circle"></i>
                        </div>
                        <div class="stat-info">
                            <h4>Facturas Pagadas</h4>
                            <p class="stat-number">142</p>
                        </div>
                    </div>
                    <div class="stat-card orange">
                        <div class="stat-icon">
                            <i class="fas fa-clock"></i>
                        </div>
                        <div class="stat-info">
                            <h4>Facturas Pendientes</h4>
                            <p class="stat-number">14</p>
                        </div>
                    </div>
                    <div class="stat-card blue">
                        <div class="stat-icon">
                            <i class="fas fa-dollar-sign"></i>
                        </div>
                        <div class="stat-info">
                            <h4>Total Recaudado</h4>
                            <p class="stat-number">₲ 12.5M</p>
                        </div>
                    </div>
                </div>

                <h3 style="margin-top: 40px;">Análisis por Método de Pago</h3>
                <div class="payment-methods">
                    <div class="payment-item">
                        <div class="payment-bar" style="width: 35%; background: #667eea;"></div>
                        <div class="payment-info">
                            <strong>Tarjeta</strong>
                            <span>35% - ₲ 4.375M</span>
                        </div>
                    </div>
                    <div class="payment-item">
                        <div class="payment-bar" style="width: 30%; background: #4caf50;"></div>
                        <div class="payment-info">
                            <strong>Efectivo</strong>
                            <span>30% - ₲ 3.750M</span>
                        </div>
                    </div>
                    <div class="payment-item">
                        <div class="payment-bar" style="width: 25%; background: #ff9800;"></div>
                        <div class="payment-info">
                            <strong>Transferencia</strong>
                            <span>25% - ₲ 3.125M</span>
                        </div>
                    </div>
                    <div class="payment-item">
                        <div class="payment-bar" style="width: 10%; background: #9c27b0;"></div>
                        <div class="payment-info">
                            <strong>Cheque</strong>
                            <span>10% - ₲ 1.250M</span>
                        </div>
                    </div>
                </div>

                <div class="tip-box" style="margin-top: 30px;">
                    <i class="fas fa-lightbulb"></i>
                    <div>
                        <h4>Reportes Detallados</h4>
                        <p>Además de estas estadísticas, puedes ver reportes detallados de productos más vendidos, servicios más prestados y recaudación diaria.</p>
                    </div>
                </div>
            </div>
        `;

        const contentCard = stepContent.querySelector('.content-card');
        contentCard.innerHTML = content;
    }

    function loadExportModule() {
        const stepContent = document.getElementById('step-11');
        if (stepContent.querySelector('.content-loaded')) return;

        const content = `
            <div class="content-loaded">
                <h3>Exportación de Datos</h3>
                <p>El sistema permite exportar todos los reportes en dos formatos:</p>

                <div class="export-options">
                    <div class="export-card excel">
                        <div class="export-icon">
                            <i class="fas fa-file-excel"></i>
                        </div>
                        <h4>Exportar a Excel</h4>
                        <p>Formato CSV compatible con Microsoft Excel y Google Sheets</p>
                        <ul>
                            <li>Fácil de editar</li>
                            <li>Compatible con fórmulas</li>
                            <li>Análisis adicional</li>
                        </ul>
                        <button class="btn-export excel-btn">
                            <i class="fas fa-download"></i> Descargar Excel
                        </button>
                    </div>

                    <div class="export-card pdf">
                        <div class="export-icon">
                            <i class="fas fa-file-pdf"></i>
                        </div>
                        <h4>Exportar a PDF</h4>
                        <p>Documento profesional listo para imprimir o compartir</p>
                        <ul>
                            <li>Formato profesional</li>
                            <li>Listo para imprimir</li>
                            <li>No editable</li>
                        </ul>
                        <button class="btn-export pdf-btn">
                            <i class="fas fa-download"></i> Descargar PDF
                        </button>
                    </div>
                </div>

                <h3 style="margin-top: 40px;">Datos que se Exportan</h3>
                <div class="export-data-list">
                    <div class="data-item">
                        <i class="fas fa-check-circle"></i>
                        <span>Resumen de recaudación del período</span>
                    </div>
                    <div class="data-item">
                        <i class="fas fa-check-circle"></i>
                        <span>Desglose por método de pago</span>
                    </div>
                    <div class="data-item">
                        <i class="fas fa-check-circle"></i>
                        <span>Recaudación diaria detallada</span>
                    </div>
                    <div class="data-item">
                        <i class="fas fa-check-circle"></i>
                        <span>Productos más vendidos</span>
                    </div>
                    <div class="data-item">
                        <i class="fas fa-check-circle"></i>
                        <span>Servicios más prestados</span>
                    </div>
                    <div class="data-item">
                        <i class="fas fa-check-circle"></i>
                        <span>Lista completa de facturas</span>
                    </div>
                </div>

                <div class="info-box" style="margin-top: 30px;">
                    <i class="fas fa-info-circle"></i>
                    <div>
                        <h4>Uso Recomendado</h4>
                        <p><strong>Excel:</strong> Para análisis adicional, gráficos personalizados y reportes contables.<br>
                        <strong>PDF:</strong> Para presentaciones a gerencia, archivo físico y documentación oficial.</p>
                    </div>
                </div>
            </div>
        `;

        const contentCard = stepContent.querySelector('.content-card');
        contentCard.innerHTML = content;
    }

    function loadTipsModule() {
        const stepContent = document.getElementById('step-12');
        if (stepContent.querySelector('.content-loaded')) return;

        const content = `
            <div class="content-loaded">
                <h3>Consejos y Mejores Prácticas</h3>
                <p>Para aprovechar al máximo el sistema, sigue estas recomendaciones:</p>

                <div class="tips-grid">
                    <div class="tip-card security">
                        <i class="fas fa-shield-alt"></i>
                        <h4>Seguridad</h4>
                        <ul>
                            <li>Nunca compartas tu contraseña</li>
                            <li>Cierra sesión al terminar</li>
                            <li>Verifica los datos antes de guardar</li>
                            <li>Realiza backups periódicos</li>
                        </ul>
                    </div>

                    <div class="tip-card efficiency">
                        <i class="fas fa-rocket"></i>
                        <h4>Eficiencia</h4>
                        <ul>
                            <li>Usa la búsqueda rápida de pacientes</li>
                            <li>Mantén actualizados los datos</li>
                            <li>Revisa reportes regularmente</li>
                            <li>Aprovecha las acciones rápidas</li>
                        </ul>
                    </div>

                    <div class="tip-card quality">
                        <i class="fas fa-star"></i>
                        <h4>Calidad de Datos</h4>
                        <ul>
                            <li>Verifica la cédula del paciente</li>
                            <li>Registra alergias correctamente</li>
                            <li>Actualiza stock de productos</li>
                            <li>Documenta consultas completamente</li>
                        </ul>
                    </div>

                    <div class="tip-card support">
                        <i class="fas fa-life-ring"></i>
                        <h4>Soporte</h4>
                        <ul>
                            <li>Consulta este tutorial cuando lo necesites</li>
                            <li>Contacta al administrador si tienes dudas</li>
                            <li>Reporta errores inmediatamente</li>
                            <li>Sugiere mejoras al sistema</li>
                        </ul>
                    </div>
                </div>

                <h3 style="margin-top: 40px;">Atajos de Teclado</h3>
                <div class="shortcuts-list">
                    <div class="shortcut-item">
                        <kbd>Ctrl</kbd> + <kbd>B</kbd>
                        <span>Buscar paciente</span>
                    </div>
                    <div class="shortcut-item">
                        <kbd>Ctrl</kbd> + <kbd>N</kbd>
                        <span>Nuevo paciente</span>
                    </div>
                    <div class="shortcut-item">
                        <kbd>Ctrl</kbd> + <kbd>F</kbd>
                        <span>Nueva factura</span>
                    </div>
                    <div class="shortcut-item">
                        <kbd>Ctrl</kbd> + <kbd>C</kbd>
                        <span>Nueva cita</span>
                    </div>
                    <div class="shortcut-item">
                        <kbd>Esc</kbd>
                        <span>Cancelar acción</span>
                    </div>
                </div>

                <div class="congratulations-box">
                    <i class="fas fa-trophy"></i>
                    <div>
                        <h4>¡Felicitaciones!</h4>
                        <p>Has completado el tutorial interactivo del Sistema Médico Centro Santa Salud. Ahora estás listo para usar todas las funcionalidades del sistema de manera eficiente y profesional.</p>
                    </div>
                </div>
            </div>
        `;

        const contentCard = stepContent.querySelector('.content-card');
        contentCard.innerHTML = content;
    }

    function finishTutorial() {
        alert('¡Felicitaciones! Has completado el tutorial.\n\nAhora puedes acceder al sistema real y comenzar a trabajar.\n\nURL del sistema: https://80-ij1yesyhewpq0wvcvo0eq-a076d47a.manusvm.computer/');
        
        // Marcar todos los pasos como completados
        for (let i = 1; i <= totalSteps; i++) {
            completedSteps.add(i);
        }
        
        updateUI();
    }

    // Agregar estilos dinámicos adicionales
    addDynamicStyles();

    function addDynamicStyles() {
        const style = document.createElement('style');
        style.textContent = `
            .search-demo {
                margin: 30px 0;
            }

            .search-box {
                display: flex;
                gap: 15px;
                max-width: 600px;
            }

            .search-input {
                flex: 1;
                padding: 15px;
                border: 2px solid #ddd;
                border-radius: 10px;
                font-size: 16px;
            }

            .btn-search {
                padding: 15px 30px;
                background: linear-gradient(135deg, var(--primary-color) 0%, #1e3c72 100%);
                color: white;
                border: none;
                border-radius: 10px;
                cursor: pointer;
                font-weight: 600;
                transition: var(--transition);
            }

            .btn-search:hover {
                transform: translateY(-2px);
                box-shadow: var(--shadow);
            }

            .form-demo {
                background: var(--light-color);
                padding: 30px;
                border-radius: 15px;
                margin: 30px 0;
            }

            .form-grid {
                display: grid;
                grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
                gap: 20px;
            }

            .form-field label {
                display: block;
                margin-bottom: 8px;
                font-weight: 600;
                color: var(--primary-color);
            }

            .form-field input {
                width: 100%;
                padding: 12px;
                border: 1px solid #ddd;
                border-radius: 8px;
                font-size: 14px;
            }

            .consultation-demo {
                background: var(--light-color);
                padding: 30px;
                border-radius: 15px;
                margin: 30px 0;
            }

            .demo-section {
                margin: 25px 0;
            }

            .demo-section h4 {
                color: var(--primary-color);
                margin-bottom: 15px;
                display: flex;
                align-items: center;
                gap: 10px;
            }

            .vitals-grid {
                display: grid;
                grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
                gap: 20px;
                margin: 25px 0;
            }

            .vital-card {
                background: white;
                padding: 20px;
                border-radius: 10px;
                text-align: center;
                box-shadow: var(--shadow);
            }

            .vital-card i {
                font-size: 32px;
                color: var(--primary-color);
                margin-bottom: 10px;
            }

            .vital-card h4 {
                font-size: 14px;
                color: #666;
                margin-bottom: 10px;
            }

            .vital-card input {
                width: 80px;
                padding: 8px;
                border: 1px solid #ddd;
                border-radius: 5px;
                text-align: center;
                font-size: 18px;
                font-weight: 700;
            }

            .medication-list {
                display: flex;
                flex-direction: column;
                gap: 15px;
            }

            .medication-item {
                display: flex;
                gap: 15px;
                align-items: flex-start;
                background: white;
                padding: 15px;
                border-radius: 10px;
                box-shadow: var(--shadow);
            }

            .medication-item i {
                font-size: 32px;
                color: var(--secondary-color);
            }

            .appointment-types {
                display: grid;
                grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
                gap: 30px;
                margin: 30px 0;
            }

            .appointment-card {
                padding: 30px;
                border-radius: 15px;
                color: white;
                box-shadow: var(--shadow-lg);
            }

            .appointment-card.regular {
                background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            }

            .appointment-card.urgent {
                background: linear-gradient(135deg, #f093fb 0%, #f5576c 100%);
            }

            .appointment-icon {
                text-align: center;
                margin-bottom: 20px;
            }

            .appointment-icon i {
                font-size: 64px;
            }

            .appointment-card h4 {
                text-align: center;
                font-size: 24px;
                margin-bottom: 15px;
            }

            .appointment-card p {
                text-align: center;
                margin-bottom: 20px;
                opacity: 0.9;
                color: white;
            }

            .appointment-card ul {
                list-style: none;
            }

            .appointment-card li {
                padding: 8px 0;
                padding-left: 25px;
                position: relative;
            }

            .appointment-card li:before {
                content: "✓";
                position: absolute;
                left: 0;
                font-weight: bold;
            }

            .status-list {
                display: flex;
                flex-direction: column;
                gap: 15px;
                margin: 30px 0;
            }

            .status-item {
                display: flex;
                gap: 15px;
                align-items: flex-start;
                padding: 20px;
                border-radius: 10px;
                box-shadow: var(--shadow);
            }

            .status-item.pending {
                background: #fff3e0;
                border-left: 5px solid #ff9800;
            }

            .status-item.confirmed {
                background: #e8f5e9;
                border-left: 5px solid #4caf50;
            }

            .status-item.completed {
                background: #e3f2fd;
                border-left: 5px solid #2196f3;
            }

            .status-item.cancelled {
                background: #ffebee;
                border-left: 5px solid #f44336;
            }

            .status-item i {
                font-size: 32px;
                flex-shrink: 0;
            }

            .status-item.pending i {
                color: #ff9800;
            }

            .status-item.confirmed i {
                color: #4caf50;
            }

            .status-item.completed i {
                color: #2196f3;
            }

            .status-item.cancelled i {
                color: #f44336;
            }

            .billing-steps {
                display: flex;
                flex-direction: column;
                gap: 20px;
                margin: 30px 0;
            }

            .billing-step {
                display: flex;
                gap: 20px;
                align-items: center;
            }

            .step-num {
                width: 50px;
                height: 50px;
                border-radius: 50%;
                background: linear-gradient(135deg, var(--primary-color) 0%, #1e3c72 100%);
                color: white;
                display: flex;
                align-items: center;
                justify-content: center;
                font-size: 24px;
                font-weight: 700;
                flex-shrink: 0;
            }

            .step-content-billing {
                flex: 1;
                background: var(--light-color);
                padding: 20px;
                border-radius: 10px;
            }

            .step-content-billing h4 {
                color: var(--primary-color);
                margin-bottom: 8px;
            }

            .invoice-demo {
                background: white;
                padding: 30px;
                border-radius: 15px;
                box-shadow: var(--shadow-lg);
                margin: 30px 0;
            }

            .invoice-header {
                text-align: center;
                padding-bottom: 20px;
                border-bottom: 2px solid var(--light-color);
                margin-bottom: 20px;
            }

            .invoice-patient {
                background: var(--light-color);
                padding: 15px;
                border-radius: 10px;
                margin-bottom: 20px;
            }

            .invoice-table {
                width: 100%;
                border-collapse: collapse;
                margin: 20px 0;
            }

            .invoice-table th,
            .invoice-table td {
                padding: 12px;
                text-align: left;
                border-bottom: 1px solid #ddd;
            }

            .invoice-table th {
                background: var(--primary-color);
                color: white;
            }

            .invoice-table tfoot td {
                font-size: 18px;
                background: var(--light-color);
            }

            .invoice-footer {
                background: var(--light-color);
                padding: 15px;
                border-radius: 10px;
                margin-top: 20px;
            }

            .crud-operations {
                display: grid;
                grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
                gap: 20px;
                margin: 30px 0;
            }

            .crud-card {
                padding: 30px;
                border-radius: 15px;
                color: white;
                text-align: center;
                box-shadow: var(--shadow-lg);
                transition: var(--transition);
            }

            .crud-card:hover {
                transform: translateY(-5px);
            }

            .crud-card.create {
                background: linear-gradient(135deg, #4caf50 0%, #45a049 100%);
            }

            .crud-card.read {
                background: linear-gradient(135deg, #2196f3 0%, #1976d2 100%);
            }

            .crud-card.update {
                background: linear-gradient(135deg, #ff9800 0%, #f57c00 100%);
            }

            .crud-card.delete {
                background: linear-gradient(135deg, #f44336 0%, #d32f2f 100%);
            }

            .crud-card i {
                font-size: 48px;
                margin-bottom: 15px;
            }

            .crud-card h4 {
                font-size: 18px;
                margin-bottom: 10px;
            }

            .crud-card p {
                font-size: 14px;
                opacity: 0.9;
                color: white;
            }

            .product-fields {
                display: flex;
                flex-direction: column;
                gap: 15px;
                margin: 30px 0;
            }

            .field-item {
                display: flex;
                gap: 15px;
                align-items: flex-start;
                background: var(--light-color);
                padding: 20px;
                border-radius: 10px;
            }

            .field-item i {
                font-size: 32px;
                color: var(--primary-color);
                flex-shrink: 0;
            }

            .services-examples {
                display: grid;
                grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
                gap: 20px;
                margin: 30px 0;
            }

            .service-card {
                background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
                color: white;
                padding: 25px;
                border-radius: 15px;
                text-align: center;
                box-shadow: var(--shadow-lg);
                transition: var(--transition);
            }

            .service-card:hover {
                transform: translateY(-5px);
            }

            .service-card i {
                font-size: 48px;
                margin-bottom: 15px;
            }

            .service-card h4 {
                font-size: 18px;
                margin-bottom: 10px;
            }

            .service-price {
                font-size: 24px;
                font-weight: 700;
                margin: 10px 0;
                color: white !important;
            }

            .service-duration {
                font-size: 14px;
                opacity: 0.9;
                color: white !important;
            }

            .comparison-table {
                margin: 30px 0;
            }

            .comparison-table table {
                width: 100%;
                border-collapse: collapse;
                background: white;
                border-radius: 10px;
                overflow: hidden;
                box-shadow: var(--shadow);
            }

            .comparison-table th,
            .comparison-table td {
                padding: 15px;
                text-align: center;
                border-bottom: 1px solid #ddd;
            }

            .comparison-table th {
                background: var(--primary-color);
                color: white;
            }

            .period-selector {
                display: flex;
                gap: 15px;
                margin: 30px 0;
                flex-wrap: wrap;
            }

            .period-btn {
                padding: 12px 25px;
                border: 2px solid var(--primary-color);
                background: white;
                color: var(--primary-color);
                border-radius: 10px;
                cursor: pointer;
                font-weight: 600;
                transition: var(--transition);
            }

            .period-btn:hover,
            .period-btn.active {
                background: var(--primary-color);
                color: white;
            }

            .payment-methods {
                margin: 30px 0;
            }

            .payment-item {
                margin: 15px 0;
            }

            .payment-bar {
                height: 40px;
                border-radius: 10px;
                margin-bottom: 8px;
                transition: width 0.5s ease;
            }

            .payment-info {
                display: flex;
                justify-content: space-between;
                padding: 0 10px;
            }

            .export-options {
                display: grid;
                grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
                gap: 30px;
                margin: 30px 0;
            }

            .export-card {
                background: white;
                padding: 30px;
                border-radius: 15px;
                box-shadow: var(--shadow-lg);
                text-align: center;
            }

            .export-card.excel {
                border-top: 5px solid #217346;
            }

            .export-card.pdf {
                border-top: 5px solid #f44336;
            }

            .export-icon {
                margin-bottom: 20px;
            }

            .export-icon i {
                font-size: 64px;
            }

            .export-card.excel .export-icon i {
                color: #217346;
            }

            .export-card.pdf .export-icon i {
                color: #f44336;
            }

            .export-card h4 {
                color: var(--primary-color);
                margin-bottom: 15px;
            }

            .export-card ul {
                list-style: none;
                margin: 20px 0;
                text-align: left;
            }

            .export-card li {
                padding: 8px 0;
                padding-left: 25px;
                position: relative;
            }

            .export-card li:before {
                content: "✓";
                position: absolute;
                left: 0;
                color: var(--secondary-color);
                font-weight: bold;
            }

            .btn-export {
                width: 100%;
                padding: 15px;
                border: none;
                border-radius: 10px;
                color: white;
                font-weight: 600;
                cursor: pointer;
                transition: var(--transition);
                display: flex;
                align-items: center;
                justify-content: center;
                gap: 10px;
            }

            .excel-btn {
                background: #217346;
            }

            .excel-btn:hover {
                background: #1a5c37;
                transform: translateY(-2px);
            }

            .pdf-btn {
                background: #f44336;
            }

            .pdf-btn:hover {
                background: #d32f2f;
                transform: translateY(-2px);
            }

            .export-data-list {
                display: flex;
                flex-direction: column;
                gap: 12px;
                margin: 30px 0;
            }

            .data-item {
                display: flex;
                align-items: center;
                gap: 15px;
                padding: 15px;
                background: var(--light-color);
                border-radius: 10px;
            }

            .data-item i {
                color: var(--secondary-color);
                font-size: 20px;
            }

            .tips-grid {
                display: grid;
                grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
                gap: 25px;
                margin: 30px 0;
            }

            .tip-card {
                padding: 25px;
                border-radius: 15px;
                color: white;
                box-shadow: var(--shadow-lg);
            }

            .tip-card.security {
                background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            }

            .tip-card.efficiency {
                background: linear-gradient(135deg, #4caf50 0%, #45a049 100%);
            }

            .tip-card.quality {
                background: linear-gradient(135deg, #ff9800 0%, #f57c00 100%);
            }

            .tip-card.support {
                background: linear-gradient(135deg, #2196f3 0%, #1976d2 100%);
            }

            .tip-card i {
                font-size: 48px;
                margin-bottom: 15px;
                display: block;
            }

            .tip-card h4 {
                font-size: 20px;
                margin-bottom: 15px;
            }

            .tip-card ul {
                list-style: none;
            }

            .tip-card li {
                padding: 8px 0;
                padding-left: 20px;
                position: relative;
            }

            .tip-card li:before {
                content: "→";
                position: absolute;
                left: 0;
            }

            .shortcuts-list {
                display: flex;
                flex-direction: column;
                gap: 15px;
                margin: 30px 0;
            }

            .shortcut-item {
                display: flex;
                align-items: center;
                justify-content: space-between;
                padding: 15px 20px;
                background: var(--light-color);
                border-radius: 10px;
            }

            kbd {
                padding: 5px 10px;
                background: white;
                border: 2px solid #ddd;
                border-radius: 5px;
                font-family: monospace;
                font-weight: 700;
                box-shadow: 0 2px 0 #ddd;
            }

            .congratulations-box {
                display: flex;
                gap: 20px;
                padding: 30px;
                background: linear-gradient(135deg, #4caf50 0%, #45a049 100%);
                color: white;
                border-radius: 15px;
                margin-top: 40px;
                box-shadow: var(--shadow-lg);
            }

            .congratulations-box i {
                font-size: 64px;
                flex-shrink: 0;
            }

            .congratulations-box h4 {
                font-size: 24px;
                margin-bottom: 10px;
            }

            .congratulations-box p {
                font-size: 16px;
                line-height: 1.6;
                color: white;
            }
        `;
        document.head.appendChild(style);
    }
});
