<?php
/**
 * Script de Pruebas del Sistema
 * Sistema de Gestión de Agua Mineral
 */

// Incluir configuración
require_once 'includes/init.php';

// Solo permitir acceso desde localhost o IP específica para seguridad
$allowedIPs = ['127.0.0.1', '::1', 'localhost'];
$clientIP = $_SERVER['REMOTE_ADDR'] ?? '';

if (!in_array($clientIP, $allowedIPs) && !isset($_GET['force'])) {
    die('Acceso no autorizado. Este script solo puede ejecutarse desde localhost.');
}

$tests = [];
$errors = [];
$warnings = [];

/**
 * Función para ejecutar una prueba
 */
function runTest($name, $callback) {
    global $tests;
    
    try {
        $start = microtime(true);
        $result = $callback();
        $time = round((microtime(true) - $start) * 1000, 2);
        
        $tests[] = [
            'name' => $name,
            'status' => $result === true ? 'PASS' : 'FAIL',
            'message' => $result === true ? 'OK' : $result,
            'time' => $time . 'ms'
        ];
        
        return $result === true;
    } catch (Exception $e) {
        $tests[] = [
            'name' => $name,
            'status' => 'ERROR',
            'message' => $e->getMessage(),
            'time' => '0ms'
        ];
        return false;
    }
}

// Prueba 1: Verificar extensiones PHP requeridas
runTest('Extensiones PHP', function() {
    $required = ['mysqli', 'pdo', 'json', 'session', 'gd', 'curl'];
    $missing = [];
    
    foreach ($required as $ext) {
        if (!extension_loaded($ext)) {
            $missing[] = $ext;
        }
    }
    
    return empty($missing) ? true : 'Extensiones faltantes: ' . implode(', ', $missing);
});

// Prueba 2: Verificar versión de PHP
runTest('Versión PHP', function() {
    $minVersion = '7.4.0';
    return version_compare(PHP_VERSION, $minVersion, '>=') ? true : "Se requiere PHP $minVersion o superior. Actual: " . PHP_VERSION;
});

// Prueba 3: Verificar conexión a base de datos
runTest('Conexión Base de Datos', function() {
    try {
        $db = Database::getInstance();
        $result = $db->select("SELECT 1 as test");
        return !empty($result) ? true : 'No se pudo ejecutar consulta de prueba';
    } catch (Exception $e) {
        return 'Error de conexión: ' . $e->getMessage();
    }
});

// Prueba 4: Verificar tablas de la base de datos
runTest('Tablas Base de Datos', function() {
    try {
        $db = Database::getInstance();
        $requiredTables = ['usuarios', 'clientes', 'ventas', 'pagos', 'zonas', 'configuracion', 'actividad_log'];
        $existingTables = [];
        
        $result = $db->select("SHOW TABLES");
        foreach ($result as $row) {
            $existingTables[] = array_values($row)[0];
        }
        
        $missing = array_diff($requiredTables, $existingTables);
        return empty($missing) ? true : 'Tablas faltantes: ' . implode(', ', $missing);
    } catch (Exception $e) {
        return 'Error al verificar tablas: ' . $e->getMessage();
    }
});

// Prueba 5: Verificar usuario administrador por defecto
runTest('Usuario Administrador', function() {
    try {
        $db = Database::getInstance();
        $result = $db->select("SELECT * FROM usuarios WHERE usuario = 'agua' AND rol = 'administrador'");
        return !empty($result) ? true : 'Usuario administrador por defecto no encontrado';
    } catch (Exception $e) {
        return 'Error al verificar usuario: ' . $e->getMessage();
    }
});

// Prueba 6: Verificar permisos de archivos
runTest('Permisos de Archivos', function() {
    $paths = [
        'config/' => 'Directorio de configuración',
        'assets/' => 'Directorio de assets',
        'uploads/' => 'Directorio de uploads (si existe)'
    ];
    
    $errors = [];
    foreach ($paths as $path => $description) {
        if (file_exists($path)) {
            if (!is_readable($path)) {
                $errors[] = "$description no es legible";
            }
            if (is_dir($path) && !is_writable($path)) {
                $errors[] = "$description no es escribible";
            }
        }
    }
    
    return empty($errors) ? true : implode(', ', $errors);
});

// Prueba 7: Verificar configuración de sesiones
runTest('Configuración Sesiones', function() {
    if (!function_exists('session_start')) {
        return 'Soporte de sesiones no disponible';
    }
    
    $sessionPath = session_save_path();
    if (empty($sessionPath)) {
        return 'Ruta de sesiones no configurada';
    }
    
    if (!is_writable($sessionPath)) {
        return 'Directorio de sesiones no escribible: ' . $sessionPath;
    }
    
    return true;
});

// Prueba 8: Verificar funciones de hash
runTest('Funciones Hash', function() {
    if (!function_exists('password_hash') || !function_exists('password_verify')) {
        return 'Funciones de hash de contraseñas no disponibles';
    }
    
    $test = 'test123';
    $hash = password_hash($test, PASSWORD_DEFAULT);
    
    return password_verify($test, $hash) ? true : 'Verificación de hash falló';
});

// Prueba 9: Verificar configuración de uploads
runTest('Configuración Uploads', function() {
    if (!ini_get('file_uploads')) {
        return 'Uploads de archivos deshabilitados';
    }
    
    $maxSize = ini_get('upload_max_filesize');
    $postSize = ini_get('post_max_size');
    
    if (intval($maxSize) < 5) {
        return "Tamaño máximo de upload muy pequeño: $maxSize";
    }
    
    return true;
});

// Prueba 10: Verificar funciones de fecha
runTest('Funciones Fecha', function() {
    try {
        $date = new DateTime();
        $formatted = $date->format('Y-m-d H:i:s');
        
        if (empty($formatted)) {
            return 'Error al formatear fecha';
        }
        
        return true;
    } catch (Exception $e) {
        return 'Error en funciones de fecha: ' . $e->getMessage();
    }
});

// Prueba 11: Verificar funciones de JSON
runTest('Funciones JSON', function() {
    $test = ['test' => 'value', 'number' => 123];
    $json = json_encode($test);
    
    if (json_last_error() !== JSON_ERROR_NONE) {
        return 'Error al codificar JSON: ' . json_last_error_msg();
    }
    
    $decoded = json_decode($json, true);
    
    if (json_last_error() !== JSON_ERROR_NONE) {
        return 'Error al decodificar JSON: ' . json_last_error_msg();
    }
    
    return ($decoded['test'] === 'value') ? true : 'Datos JSON no coinciden';
});

// Prueba 12: Verificar configuración de memoria
runTest('Configuración Memoria', function() {
    $memoryLimit = ini_get('memory_limit');
    $memoryBytes = return_bytes($memoryLimit);
    
    if ($memoryBytes < 64 * 1024 * 1024) { // 64MB mínimo
        return "Límite de memoria muy bajo: $memoryLimit";
    }
    
    return true;
});

// Función auxiliar para convertir límite de memoria a bytes
function return_bytes($val) {
    $val = trim($val);
    $last = strtolower($val[strlen($val)-1]);
    $val = intval($val);
    
    switch($last) {
        case 'g':
            $val *= 1024;
        case 'm':
            $val *= 1024;
        case 'k':
            $val *= 1024;
    }
    
    return $val;
}

// Calcular estadísticas
$totalTests = count($tests);
$passedTests = count(array_filter($tests, function($test) { return $test['status'] === 'PASS'; }));
$failedTests = count(array_filter($tests, function($test) { return $test['status'] === 'FAIL'; }));
$errorTests = count(array_filter($tests, function($test) { return $test['status'] === 'ERROR'; }));

?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Pruebas del Sistema - Agua Mineral</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <style>
        .test-pass { color: #28a745; }
        .test-fail { color: #dc3545; }
        .test-error { color: #fd7e14; }
        .system-info { background-color: #f8f9fa; }
    </style>
</head>
<body>
    <div class="container mt-4">
        <div class="row">
            <div class="col-12">
                <h1 class="mb-4">
                    <i class="fas fa-vial me-2"></i>
                    Pruebas del Sistema - Agua Mineral
                </h1>
                
                <!-- Resumen -->
                <div class="row mb-4">
                    <div class="col-md-3">
                        <div class="card bg-primary text-white">
                            <div class="card-body text-center">
                                <h3><?= $totalTests ?></h3>
                                <small>Total Pruebas</small>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="card bg-success text-white">
                            <div class="card-body text-center">
                                <h3><?= $passedTests ?></h3>
                                <small>Exitosas</small>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="card bg-danger text-white">
                            <div class="card-body text-center">
                                <h3><?= $failedTests ?></h3>
                                <small>Fallidas</small>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="card bg-warning text-white">
                            <div class="card-body text-center">
                                <h3><?= $errorTests ?></h3>
                                <small>Errores</small>
                            </div>
                        </div>
                    </div>
                </div>
                
                <!-- Estado general -->
                <div class="alert <?= ($failedTests + $errorTests) === 0 ? 'alert-success' : 'alert-warning' ?> mb-4">
                    <h5>
                        <i class="fas <?= ($failedTests + $errorTests) === 0 ? 'fa-check-circle' : 'fa-exclamation-triangle' ?> me-2"></i>
                        Estado General: <?= ($failedTests + $errorTests) === 0 ? 'SISTEMA LISTO' : 'REQUIERE ATENCIÓN' ?>
                    </h5>
                    <p class="mb-0">
                        <?= ($failedTests + $errorTests) === 0 
                            ? 'Todas las pruebas han pasado exitosamente. El sistema está listo para usar.' 
                            : 'Algunas pruebas han fallado. Revise los errores antes de usar el sistema en producción.' ?>
                    </p>
                </div>
                
                <!-- Resultados de pruebas -->
                <div class="card">
                    <div class="card-header">
                        <h5 class="mb-0">Resultados Detallados</h5>
                    </div>
                    <div class="card-body">
                        <div class="table-responsive">
                            <table class="table table-striped">
                                <thead>
                                    <tr>
                                        <th>Prueba</th>
                                        <th>Estado</th>
                                        <th>Mensaje</th>
                                        <th>Tiempo</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($tests as $test): ?>
                                        <tr>
                                            <td><?= htmlspecialchars($test['name']) ?></td>
                                            <td>
                                                <span class="test-<?= strtolower($test['status']) ?>">
                                                    <i class="fas fa-<?= $test['status'] === 'PASS' ? 'check' : ($test['status'] === 'FAIL' ? 'times' : 'exclamation') ?>"></i>
                                                    <?= $test['status'] ?>
                                                </span>
                                            </td>
                                            <td><?= htmlspecialchars($test['message']) ?></td>
                                            <td><small class="text-muted"><?= $test['time'] ?></small></td>
                                        </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
                
                <!-- Información del sistema -->
                <div class="card mt-4">
                    <div class="card-header">
                        <h5 class="mb-0">Información del Sistema</h5>
                    </div>
                    <div class="card-body system-info">
                        <div class="row">
                            <div class="col-md-6">
                                <p><strong>PHP Version:</strong> <?= PHP_VERSION ?></p>
                                <p><strong>Sistema Operativo:</strong> <?= PHP_OS ?></p>
                                <p><strong>Servidor Web:</strong> <?= $_SERVER['SERVER_SOFTWARE'] ?? 'Desconocido' ?></p>
                                <p><strong>Límite de Memoria:</strong> <?= ini_get('memory_limit') ?></p>
                            </div>
                            <div class="col-md-6">
                                <p><strong>Tiempo Máximo Ejecución:</strong> <?= ini_get('max_execution_time') ?>s</p>
                                <p><strong>Tamaño Máximo Upload:</strong> <?= ini_get('upload_max_filesize') ?></p>
                                <p><strong>Tamaño Máximo POST:</strong> <?= ini_get('post_max_size') ?></p>
                                <p><strong>Zona Horaria:</strong> <?= date_default_timezone_get() ?></p>
                            </div>
                        </div>
                    </div>
                </div>
                
                <!-- Acciones -->
                <div class="mt-4 text-center">
                    <a href="index.php" class="btn btn-primary">
                        <i class="fas fa-home me-2"></i>Ir al Sistema
                    </a>
                    <a href="?refresh=1" class="btn btn-secondary">
                        <i class="fas fa-sync me-2"></i>Ejecutar Pruebas Nuevamente
                    </a>
                </div>
            </div>
        </div>
    </div>
    
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>
