<?php
/**
 * Mapa de Clientes - Vendedor
 * Sistema de Gestión de Agua Mineral
 */

require_once '../../includes/init.php';

// Verificar que sea vendedor
if ($_SESSION['user_role'] !== 'vendedor') {
    redirect('../admin/dashboard.php');
}

$pageTitle = 'Mapa de Clientes - Vendedor';

$db = Database::getInstance();
$vendedorId = $_SESSION['user_id'];

// Obtener clientes con ubicación GPS
$clientes = $db->select("
    SELECT 
        c.id,
        c.nombre,
        c.direccion,
        c.latitud,
        c.longitud,
        c.telefono,
        (SELECT COUNT(*) FROM ventas v WHERE v.cliente_id = c.id AND MONTH(v.fecha) = MONTH(CURDATE()) AND YEAR(v.fecha) = YEAR(CURDATE())) as entregas_mes,
        (SELECT SUM(v.cantidad_bidones) FROM ventas v WHERE v.cliente_id = c.id AND MONTH(v.fecha) = MONTH(CURDATE()) AND YEAR(v.fecha) = YEAR(CURDATE())) as bidones_mes,
        (SELECT SUM(v.total) FROM ventas v WHERE v.cliente_id = c.id) as total_ventas,
        (SELECT SUM(p.monto) FROM pagos p WHERE p.cliente_id = c.id) as total_pagos,
        (SELECT MAX(v.fecha) FROM ventas v WHERE v.cliente_id = c.id) as ultima_entrega
    FROM clientes c
    WHERE c.vendedor_id = ? 
    AND c.activo = 1 
    AND c.latitud IS NOT NULL 
    AND c.longitud IS NOT NULL
    ORDER BY c.nombre
", [$vendedorId]);

// Calcular saldos
foreach ($clientes as &$cliente) {
    $cliente['saldo'] = ($cliente['total_ventas'] ?? 0) - ($cliente['total_pagos'] ?? 0);
    $cliente['entregas_mes'] = $cliente['entregas_mes'] ?? 0;
    $cliente['bidones_mes'] = $cliente['bidones_mes'] ?? 0;
}

// Clientes sin ubicación GPS
$clientesSinUbicacion = $db->select("
    SELECT id, nombre, direccion
    FROM clientes 
    WHERE vendedor_id = ? 
    AND activo = 1 
    AND (latitud IS NULL OR longitud IS NULL)
    ORDER BY nombre
", [$vendedorId]);

include '../../includes/header.php';
?>

<div class="row">
    <div class="col-12">
        <div class="d-flex justify-content-between align-items-center mb-4">
            <h1 class="h3">
                <i class="fas fa-map-marker-alt me-2"></i>Mapa de Clientes
            </h1>
            <div>
                <button type="button" class="btn btn-info me-2" data-bs-toggle="modal" data-bs-target="#leyendaModal">
                    <i class="fas fa-info-circle me-1"></i>Leyenda
                </button>
                <a href="mis-clientes.php" class="btn btn-secondary">
                    <i class="fas fa-list me-1"></i>Vista Lista
                </a>
            </div>
        </div>
    </div>
</div>

<!-- Estadísticas rápidas -->
<div class="row mb-4">
    <div class="col-md-3">
        <div class="card bg-primary text-white">
            <div class="card-body text-center">
                <i class="fas fa-map-marker-alt fa-2x mb-2"></i>
                <h4><?= count($clientes) ?></h4>
                <small>Con Ubicación GPS</small>
            </div>
        </div>
    </div>
    <div class="col-md-3">
        <div class="card bg-warning text-white">
            <div class="card-body text-center">
                <i class="fas fa-exclamation-triangle fa-2x mb-2"></i>
                <h4><?= count($clientesSinUbicacion) ?></h4>
                <small>Sin Ubicación GPS</small>
            </div>
        </div>
    </div>
    <div class="col-md-3">
        <div class="card bg-success text-white">
            <div class="card-body text-center">
                <i class="fas fa-check-circle fa-2x mb-2"></i>
                <h4><?= count(array_filter($clientes, function($c) { return $c['entregas_mes'] > 0; })) ?></h4>
                <small>Activos este Mes</small>
            </div>
        </div>
    </div>
    <div class="col-md-3">
        <div class="card bg-danger text-white">
            <div class="card-body text-center">
                <i class="fas fa-dollar-sign fa-2x mb-2"></i>
                <h4><?= count(array_filter($clientes, function($c) { return $c['saldo'] > 0; })) ?></h4>
                <small>Con Saldo Pendiente</small>
            </div>
        </div>
    </div>
</div>

<div class="row">
    <!-- Mapa -->
    <div class="col-lg-9">
        <div class="card">
            <div class="card-header d-flex justify-content-between align-items-center">
                <h5 class="mb-0">Ubicación de Clientes</h5>
                <div class="btn-group btn-group-sm">
                    <button type="button" class="btn btn-outline-primary" onclick="centrarMapa()">
                        <i class="fas fa-crosshairs me-1"></i>Centrar
                    </button>
                    <button type="button" class="btn btn-outline-success" onclick="optimizarRuta()">
                        <i class="fas fa-route me-1"></i>Optimizar Ruta
                    </button>
                </div>
            </div>
            <div class="card-body p-0">
                <div id="map" style="height: 600px; width: 100%;"></div>
            </div>
        </div>
    </div>
    
    <!-- Panel lateral -->
    <div class="col-lg-3">
        <!-- Filtros -->
        <div class="card mb-3">
            <div class="card-header">
                <h6 class="mb-0">Filtros</h6>
            </div>
            <div class="card-body">
                <div class="form-check mb-2">
                    <input class="form-check-input" type="checkbox" id="mostrarTodos" checked onchange="filtrarMarcadores()">
                    <label class="form-check-label" for="mostrarTodos">
                        Mostrar todos
                    </label>
                </div>
                <div class="form-check mb-2">
                    <input class="form-check-input" type="checkbox" id="mostrarActivos" checked onchange="filtrarMarcadores()">
                    <label class="form-check-label" for="mostrarActivos">
                        Activos este mes
                    </label>
                </div>
                <div class="form-check mb-2">
                    <input class="form-check-input" type="checkbox" id="mostrarSaldo" checked onchange="filtrarMarcadores()">
                    <label class="form-check-label" for="mostrarSaldo">
                        Con saldo pendiente
                    </label>
                </div>
                <div class="form-check">
                    <input class="form-check-input" type="checkbox" id="mostrarInactivos" checked onchange="filtrarMarcadores()">
                    <label class="form-check-label" for="mostrarInactivos">
                        Sin entregas este mes
                    </label>
                </div>
            </div>
        </div>
        
        <!-- Lista de clientes -->
        <div class="card">
            <div class="card-header">
                <h6 class="mb-0">Lista de Clientes</h6>
            </div>
            <div class="card-body" style="max-height: 400px; overflow-y: auto;">
                <?php if (empty($clientes)): ?>
                    <p class="text-muted text-center">No hay clientes con ubicación GPS</p>
                <?php else: ?>
                    <?php foreach ($clientes as $cliente): ?>
                        <?php
                        $saldoClass = $cliente['saldo'] > 0 ? 'text-danger' : ($cliente['saldo'] < 0 ? 'text-success' : 'text-muted');
                        $activoMes = $cliente['entregas_mes'] > 0;
                        ?>
                        <div class="cliente-item mb-2 p-2 border rounded" 
                             data-cliente-id="<?= $cliente['id'] ?>"
                             data-activo="<?= $activoMes ? 1 : 0 ?>"
                             data-saldo="<?= $cliente['saldo'] > 0 ? 1 : 0 ?>"
                             style="cursor: pointer;"
                             onclick="centrarEnCliente(<?= $cliente['latitud'] ?>, <?= $cliente['longitud'] ?>, <?= $cliente['id'] ?>)">
                            <div class="d-flex justify-content-between align-items-start">
                                <div class="flex-grow-1">
                                    <div class="fw-bold"><?= htmlspecialchars($cliente['nombre']) ?></div>
                                    <small class="text-muted"><?= htmlspecialchars($cliente['direccion']) ?></small>
                                    <div class="mt-1">
                                        <span class="badge bg-<?= $activoMes ? 'success' : 'secondary' ?> me-1">
                                            <?= $cliente['entregas_mes'] ?> entregas
                                        </span>
                                        <?php if ($cliente['saldo'] != 0): ?>
                                            <span class="badge bg-<?= $cliente['saldo'] > 0 ? 'danger' : 'info' ?>">
                                                <?= formatCurrency(abs($cliente['saldo'])) ?>
                                            </span>
                                        <?php endif; ?>
                                    </div>
                                </div>
                                <div class="text-end">
                                    <i class="fas fa-map-marker-alt text-primary"></i>
                                </div>
                            </div>
                        </div>
                    <?php endforeach; ?>
                <?php endif; ?>
            </div>
        </div>
        
        <!-- Clientes sin ubicación -->
        <?php if (!empty($clientesSinUbicacion)): ?>
        <div class="card mt-3">
            <div class="card-header bg-warning text-dark">
                <h6 class="mb-0">
                    <i class="fas fa-exclamation-triangle me-1"></i>
                    Sin Ubicación GPS
                </h6>
            </div>
            <div class="card-body">
                <small class="text-muted">Los siguientes clientes no tienen ubicación GPS configurada:</small>
                <?php foreach ($clientesSinUbicacion as $cliente): ?>
                    <div class="mt-2 p-2 bg-light rounded">
                        <div class="fw-bold"><?= htmlspecialchars($cliente['nombre']) ?></div>
                        <small class="text-muted"><?= htmlspecialchars($cliente['direccion']) ?></small>
                    </div>
                <?php endforeach; ?>
            </div>
        </div>
        <?php endif; ?>
    </div>
</div>

<!-- Modal de leyenda -->
<div class="modal fade" id="leyendaModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Leyenda del Mapa</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <div class="row">
                    <div class="col-6 mb-3">
                        <div class="d-flex align-items-center">
                            <div class="me-2" style="width: 20px; height: 20px; background-color: #28a745; border-radius: 50%;"></div>
                            <span>Activo este mes</span>
                        </div>
                    </div>
                    <div class="col-6 mb-3">
                        <div class="d-flex align-items-center">
                            <div class="me-2" style="width: 20px; height: 20px; background-color: #6c757d; border-radius: 50%;"></div>
                            <span>Sin entregas este mes</span>
                        </div>
                    </div>
                    <div class="col-6 mb-3">
                        <div class="d-flex align-items-center">
                            <div class="me-2" style="width: 20px; height: 20px; background-color: #dc3545; border-radius: 50%;"></div>
                            <span>Con saldo pendiente</span>
                        </div>
                    </div>
                    <div class="col-6 mb-3">
                        <div class="d-flex align-items-center">
                            <div class="me-2" style="width: 20px; height: 20px; background-color: #007bff; border-radius: 50%;"></div>
                            <span>Saldo al día</span>
                        </div>
                    </div>
                </div>
                <hr>
                <h6>Funciones del Mapa:</h6>
                <ul class="list-unstyled">
                    <li><i class="fas fa-mouse-pointer me-2"></i>Click en marcador: Ver información del cliente</li>
                    <li><i class="fas fa-crosshairs me-2"></i>Centrar: Ajustar vista a todos los clientes</li>
                    <li><i class="fas fa-route me-2"></i>Optimizar Ruta: Calcular ruta más eficiente</li>
                </ul>
            </div>
        </div>
    </div>
</div>

<?php
$extraJS = '
<script>
let map;
let markers = [];
let clientesData = ' . json_encode($clientes) . ';

// Inicializar mapa
function initMap() {
    // Centro por defecto (Asunción, Paraguay)
    const defaultCenter = { lat: -25.2637, lng: -57.5759 };
    
    map = new google.maps.Map(document.getElementById("map"), {
        zoom: 12,
        center: defaultCenter,
        mapTypeId: google.maps.MapTypeId.ROADMAP
    });
    
    // Agregar marcadores
    agregarMarcadores();
    
    // Centrar mapa en los marcadores
    if (markers.length > 0) {
        centrarMapa();
    }
}

function agregarMarcadores() {
    clientesData.forEach(cliente => {
        const position = {
            lat: parseFloat(cliente.latitud),
            lng: parseFloat(cliente.longitud)
        };
        
        // Determinar color del marcador
        let color = "#007bff"; // Azul por defecto
        if (cliente.saldo > 0) {
            color = "#dc3545"; // Rojo para saldo pendiente
        } else if (cliente.entregas_mes > 0) {
            color = "#28a745"; // Verde para activos
        } else {
            color = "#6c757d"; // Gris para inactivos
        }
        
        const marker = new google.maps.Marker({
            position: position,
            map: map,
            title: cliente.nombre,
            icon: {
                path: google.maps.SymbolPath.CIRCLE,
                scale: 8,
                fillColor: color,
                fillOpacity: 0.8,
                strokeColor: "#ffffff",
                strokeWeight: 2
            }
        });
        
        // Información del cliente
        const infoContent = `
            <div style="max-width: 300px;">
                <h6>${cliente.nombre}</h6>
                <p class="mb-1"><i class="fas fa-map-marker-alt"></i> ${cliente.direccion}</p>
                ${cliente.telefono ? `<p class="mb-1"><i class="fas fa-phone"></i> ${cliente.telefono}</p>` : ""}
                <hr>
                <div class="row text-center">
                    <div class="col-4">
                        <strong>${cliente.entregas_mes}</strong><br>
                        <small>Entregas</small>
                    </div>
                    <div class="col-4">
                        <strong>${cliente.bidones_mes}</strong><br>
                        <small>Bidones</small>
                    </div>
                    <div class="col-4">
                        <strong style="color: ${cliente.saldo > 0 ? "#dc3545" : cliente.saldo < 0 ? "#28a745" : "#6c757d"}">
                            $${Math.abs(cliente.saldo).toLocaleString("es-AR", {minimumFractionDigits: 2})}
                        </strong><br>
                        <small>Saldo</small>
                    </div>
                </div>
                ${cliente.ultima_entrega ? `<p class="mt-2 mb-1"><small>Última entrega: ${new Date(cliente.ultima_entrega).toLocaleDateString("es-ES")}</small></p>` : ""}
                <div class="mt-2">
                    <a href="nueva-venta.php?cliente_id=${cliente.id}" class="btn btn-primary btn-sm">
                        <i class="fas fa-plus"></i> Nueva Venta
                    </a>
                    <a href="cliente-detalle.php?id=${cliente.id}" class="btn btn-info btn-sm">
                        <i class="fas fa-eye"></i> Detalle
                    </a>
                </div>
            </div>
        `;
        
        const infoWindow = new google.maps.InfoWindow({
            content: infoContent
        });
        
        marker.addListener("click", () => {
            // Cerrar otras ventanas de información
            markers.forEach(m => {
                if (m.infoWindow) {
                    m.infoWindow.close();
                }
            });
            
            infoWindow.open(map, marker);
        });
        
        // Guardar referencia
        marker.clienteData = cliente;
        marker.infoWindow = infoWindow;
        markers.push(marker);
    });
}

function centrarMapa() {
    if (markers.length === 0) return;
    
    const bounds = new google.maps.LatLngBounds();
    markers.forEach(marker => {
        if (marker.getVisible()) {
            bounds.extend(marker.getPosition());
        }
    });
    
    map.fitBounds(bounds);
    
    // Ajustar zoom si hay un solo marcador
    if (markers.filter(m => m.getVisible()).length === 1) {
        map.setZoom(15);
    }
}

function centrarEnCliente(lat, lng, clienteId) {
    const position = { lat: parseFloat(lat), lng: parseFloat(lng) };
    map.setCenter(position);
    map.setZoom(16);
    
    // Abrir info window del cliente
    const marker = markers.find(m => m.clienteData.id == clienteId);
    if (marker && marker.infoWindow) {
        marker.infoWindow.open(map, marker);
    }
}

function filtrarMarcadores() {
    const mostrarTodos = document.getElementById("mostrarTodos").checked;
    const mostrarActivos = document.getElementById("mostrarActivos").checked;
    const mostrarSaldo = document.getElementById("mostrarSaldo").checked;
    const mostrarInactivos = document.getElementById("mostrarInactivos").checked;
    
    markers.forEach(marker => {
        const cliente = marker.clienteData;
        let mostrar = false;
        
        if (mostrarTodos) {
            mostrar = true;
        } else {
            if (mostrarActivos && cliente.entregas_mes > 0) mostrar = true;
            if (mostrarSaldo && cliente.saldo > 0) mostrar = true;
            if (mostrarInactivos && cliente.entregas_mes === 0) mostrar = true;
        }
        
        marker.setVisible(mostrar);
        
        // Actualizar lista lateral
        const clienteItem = document.querySelector(`[data-cliente-id="${cliente.id}"]`);
        if (clienteItem) {
            clienteItem.style.display = mostrar ? "block" : "none";
        }
    });
    
    // Recentrar mapa
    setTimeout(centrarMapa, 100);
}

function optimizarRuta() {
    const marcadoresVisibles = markers.filter(m => m.getVisible());
    
    if (marcadoresVisibles.length < 2) {
        alert("Se necesitan al menos 2 clientes visibles para optimizar la ruta.");
        return;
    }
    
    // Aquí se podría implementar la optimización de ruta usando Google Directions API
    alert(`Función de optimización de ruta en desarrollo. Clientes seleccionados: ${marcadoresVisibles.length}`);
}

// Manejar cambios en "Mostrar todos"
document.getElementById("mostrarTodos").addEventListener("change", function() {
    const otrosCheckboxes = ["mostrarActivos", "mostrarSaldo", "mostrarInactivos"];
    
    if (this.checked) {
        otrosCheckboxes.forEach(id => {
            document.getElementById(id).checked = false;
        });
    }
});

// Manejar cambios en otros checkboxes
["mostrarActivos", "mostrarSaldo", "mostrarInactivos"].forEach(id => {
    document.getElementById(id).addEventListener("change", function() {
        if (this.checked) {
            document.getElementById("mostrarTodos").checked = false;
        }
    });
});
</script>

<script async defer src="https://maps.googleapis.com/maps/api/js?key=' . GOOGLE_MAPS_API_KEY . '&callback=initMap"></script>
';

include '../../includes/footer.php';
?>
