<?php
/**
 * Nuevo Cliente - Vendedor
 * Sistema de Gestión de Agua Mineral
 */

require_once '../../includes/init.php';

// Verificar que sea vendedor
checkPermission('vendedor');

$pageTitle = 'Nuevo Cliente - Vendedor';

$db = Database::getInstance();
$vendedorId = $_SESSION['user_id'];

// Procesar formulario
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $nombre = sanitize($_POST['nombre']);
    $direccion = sanitize($_POST['direccion']);
    $telefono = sanitize($_POST['telefono']);
    $email = sanitize($_POST['email']);
    $latitud = !empty($_POST['latitud']) ? (float)$_POST['latitud'] : null;
    $longitud = !empty($_POST['longitud']) ? (float)$_POST['longitud'] : null;
    $observaciones = sanitize($_POST['observaciones']);
    
    // Validaciones
    $errors = [];
    
    if (empty($nombre)) {
        $errors[] = 'El nombre es obligatorio.';
    }
    
    if (empty($direccion)) {
        $errors[] = 'La dirección es obligatoria.';
    }
    
    if (!empty($email) && !validateEmail($email)) {
        $errors[] = 'El email no es válido.';
    }
    
    if ($latitud && $longitud && !validarCoordenadas($latitud, $longitud)) {
        $errors[] = 'Las coordenadas GPS no son válidas para Paraguay.';
    }
    
    if (empty($errors)) {
        try {
            $query = "INSERT INTO clientes (nombre, direccion, telefono, email, vendedor_id, latitud, longitud, observaciones, fecha_registro) 
                      VALUES (?, ?, ?, ?, ?, ?, ?, ?, NOW())";
            
            $params = [$nombre, $direccion, $telefono, $email, $vendedorId, $latitud, $longitud, $observaciones];
            
            if ($db->execute($query, $params)) {
                // Log de actividad
                logActivity($vendedorId, 'create_client', "Cliente creado por vendedor: $nombre (ID: " . $db->lastInsertId() . ")");
                
                setFlashMessage('Cliente creado exitosamente y asignado a usted.', 'success');
                redirect('mis-clientes.php');
            } else {
                $errors[] = 'Error al crear el cliente. Intente nuevamente.';
            }
        } catch (Exception $e) {
            $errors[] = 'Error en la base de datos: ' . $e->getMessage();
        }
    }
    
    // Si hay errores, recargar los datos del POST para que no se pierdan
    $formData = $_POST;
} else {
    $formData = [];
}

include '../../includes/header.php';
?>

<div class="row">
    <div class="col-12">
        <div class="d-flex justify-content-between align-items-center mb-4">
            <h1 class="h3">
                <i class="fas fa-user-plus me-2"></i>Nuevo Cliente
            </h1>
            <a href="mis-clientes.php" class="btn btn-secondary">
                <i class="fas fa-arrow-left me-2"></i>Volver a Mis Clientes
            </a>
        </div>
    </div>
</div>

<?php if (!empty($errors)): ?>
    <div class="alert alert-danger">
        <ul class="mb-0">
            <?php foreach ($errors as $error): ?>
                <li><?= htmlspecialchars($error) ?></li>
            <?php endforeach; ?>
        </ul>
    </div>
<?php endif; ?>

<div class="row">
    <div class="col-lg-8">
        <div class="card">
            <div class="card-header">
                <h5 class="mb-0">Información del Nuevo Cliente</h5>
            </div>
            <div class="card-body">
                <form method="POST" class="needs-validation" novalidate>
                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="nombre" class="form-label">Nombre Completo *</label>
                                <input type="text" name="nombre" id="nombre" class="form-control" 
                                       value="<?= htmlspecialchars($formData['nombre'] ?? '') ?>" required>
                                <div class="invalid-feedback">
                                    Por favor ingrese el nombre del cliente.
                                </div>
                            </div>
                        </div>
                        
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="telefono" class="form-label">Teléfono</label>
                                <input type="tel" name="telefono" id="telefono" class="form-control" 
                                       value="<?= htmlspecialchars($formData['telefono'] ?? '') ?>" 
                                       placeholder="0981-123-456">
                            </div>
                        </div>
                    </div>
                    
                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="email" class="form-label">Email</label>
                                <input type="email" name="email" id="email" class="form-control" 
                                       value="<?= htmlspecialchars($formData['email'] ?? '') ?>">
                                <div class="invalid-feedback">
                                    Por favor ingrese un email válido.
                                </div>
                            </div>
                        </div>
                        
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="vendedor_id" class="form-label">Vendedor Asignado</label>
                                <input type="text" class="form-control" value="<?= htmlspecialchars($_SESSION['user_name']) ?>" disabled>
                                <div class="form-text">El cliente será asignado automáticamente a usted.</div>
                            </div>
                        </div>
                    </div>
                    
                    <div class="mb-3">
                        <label for="direccion" class="form-label">Dirección Completa *</label>
                        <textarea name="direccion" id="direccion" class="form-control" rows="2" required><?= htmlspecialchars($formData['direccion'] ?? '') ?></textarea>
                        <div class="invalid-feedback">
                            Por favor ingrese la dirección del cliente.
                        </div>
                    </div>
                    
                    <div class="mb-3">
                        <label for="observaciones" class="form-label">Observaciones</label>
                        <textarea name="observaciones" id="observaciones" class="form-control" rows="3"><?= htmlspecialchars($formData['observaciones'] ?? '') ?></textarea>
                        <div class="form-text">Información adicional sobre el cliente (opcional)</div>
                    </div>
                    
                    <hr>
                    
                    <h6 class="mb-3">
                        <i class="fas fa-map-marker-alt me-2"></i>Ubicación GPS (Opcional)
                    </h6>
                    
                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="latitud" class="form-label">Latitud</label>
                                <input type="number" name="latitud" id="latitud" class="form-control" 
                                       step="0.000001" min="-27.6" max="-19.3" 
                                       value="<?= htmlspecialchars($formData['latitud'] ?? '') ?>" 
                                       placeholder="-25.263740">
                                <div class="form-text">Coordenada de latitud para Paraguay</div>
                            </div>
                        </div>
                        
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="longitud" class="form-label">Longitud</label>
                                <input type="number" name="longitud" id="longitud" class="form-control" 
                                       step="0.000001" min="-62.6" max="-54.3" 
                                       value="<?= htmlspecialchars($formData['longitud'] ?? '') ?>" 
                                       placeholder="-57.575926">
                                <div class="form-text">Coordenada de longitud para Paraguay</div>
                            </div>
                        </div>
                    </div>
                    
                    <div class="mb-3">
                        <button type="button" class="btn btn-info btn-sm" onclick="buscarUbicacion()">
                            <i class="fas fa-search me-1"></i>Buscar Ubicación por Dirección
                        </button>
                        <button type="button" class="btn btn-success btn-sm" onclick="obtenerUbicacionActual()">
                            <i class="fas fa-crosshairs me-1"></i>Usar Mi Ubicación
                        </button>
                    </div>
                    
                    <div class="d-grid gap-2 d-md-flex justify-content-md-end">
                        <a href="mis-clientes.php" class="btn btn-secondary me-md-2">
                            <i class="fas fa-times me-1"></i>Cancelar
                        </a>
                        <button type="submit" class="btn btn-primary">
                            <i class="fas fa-save me-1"></i>Guardar Cliente
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>
</div>

<?php
$extraJS = '
<script>
// Validación del formulario
(function() {
    "use strict";
    window.addEventListener("load", function() {
        var forms = document.getElementsByClassName("needs-validation");
        var validation = Array.prototype.filter.call(forms, function(form) {
            form.addEventListener("submit", function(event) {
                if (form.checkValidity() === false) {
                    event.preventDefault();
                    event.stopPropagation();
                }
                form.classList.add("was-validated");
            }, false);
        });
    }, false);
})();

// Buscar ubicación por dirección
function buscarUbicacion() {
    const direccion = document.getElementById("direccion").value;
    
    if (!direccion.trim()) {
        alert("Por favor, ingrese una dirección para buscar.");
        return;
    }
    
    // Usar servicio de geocodificación (requiere API key de Google Maps)
    if (typeof google !== "undefined" && google.maps) {
        const geocoder = new google.maps.Geocoder();
        
        geocoder.geocode({ address: direccion + ", Paraguay" }, (results, status) => {
            if (status === "OK" && results[0]) {
                const location = results[0].geometry.location;
                
                document.getElementById("latitud").value = location.lat().toFixed(6);
                document.getElementById("longitud").value = location.lng().toFixed(6);
                
                alert("Ubicación encontrada y coordenadas actualizadas.");
            } else {
                alert("No se pudo encontrar la ubicación. Intente con una dirección más específica.");
            }
        });
    } else {
        alert("Servicio de geocodificación no disponible. Ingrese las coordenadas manualmente.");
    }
}

// Obtener ubicación actual
function obtenerUbicacionActual() {
    if (navigator.geolocation) {
        navigator.geolocation.getCurrentPosition(
            function(position) {
                const lat = position.coords.latitude;
                const lng = position.coords.longitude;
                
                // Validar que esté en Paraguay
                if (lat >= -27.6 && lat <= -19.3 && lng >= -62.6 && lng <= -54.3) {
                    document.getElementById("latitud").value = lat.toFixed(6);
                    document.getElementById("longitud").value = lng.toFixed(6);
                    alert("Ubicación actual obtenida correctamente.");
                } else {
                    alert("Su ubicación actual no está en Paraguay.");
                }
            },
            function(error) {
                alert("No se pudo obtener su ubicación: " + error.message);
            }
        );
    } else {
        alert("Su navegador no soporta geolocalización.");
    }
}

// Validación de coordenadas en tiempo real
document.getElementById("latitud").addEventListener("blur", function() {
    const lat = parseFloat(this.value);
    if (this.value && (lat < -27.6 || lat > -19.3)) {
        this.setCustomValidity("La latitud debe estar entre -27.6 y -19.3 para Paraguay");
    } else {
        this.setCustomValidity("");
    }
});

document.getElementById("longitud").addEventListener("blur", function() {
    const lng = parseFloat(this.value);
    if (this.value && (lng < -62.6 || lng > -54.3)) {
        this.setCustomValidity("La longitud debe estar entre -62.6 y -54.3 para Paraguay");
    } else {
        this.setCustomValidity("");
    }
});
</script>
';

include '../../includes/footer.php';
?>
