<?php
/**
 * Editar Vendedor - Administrador
 * Sistema de Gestión de Agua Mineral
 */

require_once '../../includes/init.php';

// Verificar que sea administrador
checkPermission('administrador');

$pageTitle = 'Editar Vendedor - Administrador';

$db = Database::getInstance();

$vendedorId = (int)($_GET['id'] ?? 0);

// 1. Obtener datos del vendedor
$vendedor = $db->select("SELECT * FROM usuarios WHERE id = ? AND rol = 'vendedor'", [$vendedorId]);

if (empty($vendedor)) {
    setFlashMessage('Vendedor no encontrado.', 'error');
    redirect('vendedores.php');
}

$vendedor = $vendedor[0];

// Obtener zonas para el formulario
$zonas = $db->select("SELECT id, nombre FROM zonas ORDER BY nombre");

// Procesar formulario
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $nombre = sanitize($_POST['nombre']);
    $usuario = sanitize($_POST['usuario']);
    $password = $_POST['password'];
    $zonaId = (int)$_POST['zona_id'];
    $telefono = sanitize($_POST['telefono']);
    $email = sanitize($_POST['email']);
    
    // Validaciones
    $errors = [];
    
    if (empty($nombre)) {
        $errors[] = 'El nombre es obligatorio.';
    }
    
    if (empty($usuario)) {
        $errors[] = 'El usuario es obligatorio.';
    }
    
    // Verificar que el usuario no exista en otro registro
    if ($usuario) {
        $existing = $db->select("SELECT id FROM usuarios WHERE usuario = ? AND id != ?", [$usuario, $vendedorId]);
        if (!empty($existing)) {
            $errors[] = 'El nombre de usuario ya existe en otro registro.';
        }
    }
    
    if (!empty($password) && strlen($password) < 6) {
        $errors[] = 'La contraseña debe tener al menos 6 caracteres.';
    }
    
    if (empty($errors)) {
        try {
            $query = "UPDATE usuarios SET 
                        nombre = ?, 
                        usuario = ?, 
                        zona_id = ?, 
                        telefono = ?, 
                        email = ?, 
                        updated_at = NOW()";
            
            $params = [$nombre, $usuario, $zonaId, $telefono, $email];
            
            // Si se proporciona una nueva contraseña, la actualizamos
            if (!empty($password)) {
                $hashedPassword = hashPassword($password);
                $query .= ", password = ?";
                $params[] = $hashedPassword;
            }
            
            $query .= " WHERE id = ? AND rol = 'vendedor'";
            $params[] = $vendedorId;
            
            if ($db->execute($query, $params)) {
                // Log de actividad
                logActivity($_SESSION['user_id'], 'update_seller', "Vendedor actualizado: $nombre (ID: $vendedorId)");
                
                setFlashMessage('Vendedor actualizado exitosamente.', 'success');
                redirect('vendedores.php');
            } else {
                $errors[] = 'Error al actualizar el vendedor. Intente nuevamente.';
            }
        } catch (Exception $e) {
            $errors[] = 'Error en la base de datos: ' . $e->getMessage();
        }
    }
    
    // Si hay errores, recargar los datos del POST para que no se pierdan
    $vendedor = array_merge($vendedor, $_POST);
}

include '../../includes/header.php';
?>

<div class="row">
    <div class="col-12">
        <div class="d-flex justify-content-between align-items-center mb-4">
            <h1 class="h3">
                <i class="fas fa-user-edit me-2"></i>Editar Vendedor: <?= htmlspecialchars($vendedor['nombre']) ?>
            </h1>
            <a href="vendedores.php" class="btn btn-secondary">
                <i class="fas fa-arrow-left me-2"></i>Volver a Vendedores
            </a>
        </div>
    </div>
</div>

<?php if (!empty($errors)): ?>
    <div class="alert alert-danger">
        <ul class="mb-0">
            <?php foreach ($errors as $error): ?>
                <li><?= htmlspecialchars($error) ?></li>
            <?php endforeach; ?>
        </ul>
    </div>
<?php endif; ?>

<div class="row">
    <div class="col-lg-8">
        <div class="card">
            <div class="card-header">
                <h5 class="mb-0">Información del Vendedor</h5>
            </div>
            <div class="card-body">
                <form method="POST" class="needs-validation" novalidate>
                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="nombre" class="form-label">Nombre Completo *</label>
                                <input type="text" name="nombre" id="nombre" class="form-control" 
                                       value="<?= htmlspecialchars($vendedor['nombre'] ?? '') ?>" required>
                                <div class="invalid-feedback">
                                    Por favor ingrese el nombre completo.
                                </div>
                            </div>
                        </div>
                        
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="usuario" class="form-label">Usuario *</label>
                                <input type="text" name="usuario" id="usuario" class="form-control" 
                                       value="<?= htmlspecialchars($vendedor['usuario'] ?? '') ?>" required>
                                <div class="invalid-feedback">
                                    Por favor ingrese un nombre de usuario.
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="telefono" class="form-label">Teléfono</label>
                                <input type="tel" name="telefono" id="telefono" class="form-control" 
                                       value="<?= htmlspecialchars($vendedor['telefono'] ?? '') ?>" 
                                       placeholder="0981-123-456">
                            </div>
                        </div>
                        
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="email" class="form-label">Email</label>
                                <input type="email" name="email" id="email" class="form-control" 
                                       value="<?= htmlspecialchars($vendedor['email'] ?? '') ?>">
                                <div class="invalid-feedback">
                                    Por favor ingrese un email válido.
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="zona_id" class="form-label">Zona Asignada</label>
                                <select name="zona_id" id="zona_id" class="form-select">
                                    <option value="">Sin asignar</option>
                                    <?php foreach ($zonas as $zona): ?>
                                        <option value="<?= $zona['id'] ?>" 
                                                <?= (($vendedor['zona_id'] ?? '') == $zona['id']) ? 'selected' : '' ?>>
                                            <?= htmlspecialchars($zona['nombre']) ?>
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                        </div>
                        
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="password" class="form-label">Nueva Contraseña (Dejar vacío para no cambiar)</label>
                                <input type="password" name="password" id="password" class="form-control" 
                                       minlength="6">
                                <div class="form-text">Mínimo 6 caracteres.</div>
                                <div class="invalid-feedback">
                                    La contraseña debe tener al menos 6 caracteres.
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <div class="d-grid gap-2 d-md-flex justify-content-md-end mt-4">
                        <a href="vendedores.php" class="btn btn-secondary me-md-2">
                            <i class="fas fa-times me-1"></i>Cancelar
                        </a>
                        <button type="submit" class="btn btn-primary">
                            <i class="fas fa-save me-1"></i>Guardar Cambios
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>
    
    <div class="col-lg-4">
        <div class="card">
            <div class="card-header">
                <h6 class="mb-0">Información Adicional</h6>
            </div>
            <div class="card-body">
                <p><strong>ID Vendedor:</strong> <?= $vendedor['id'] ?></p>
                <p><strong>Rol:</strong> <span class="badge bg-info">Vendedor</span></p>
                <p><strong>Estado:</strong> 
                    <span class="badge bg-<?= $vendedor['activo'] ? 'success' : 'secondary' ?>">
                        <?= $vendedor['activo'] ? 'Activo' : 'Inactivo' ?>
                    </span>
                </p>
                <p><strong>Registrado:</strong> <?= formatDate($vendedor['fecha_registro']) ?></p>
                <p><strong>Última Actualización:</strong> <?= formatDate($vendedor['updated_at'], 'd/m/Y H:i') ?></p>
                
                <hr>
                
                <?php
                // Obtener estadísticas rápidas
                $stats = $db->select("
                    SELECT 
                        COUNT(DISTINCT c.id) as total_clientes,
                        COALESCE(SUM(v.total), 0) as total_facturado
                    FROM clientes c
                    LEFT JOIN ventas v ON c.id = v.cliente_id
                    WHERE c.vendedor_id = ?
                ", [$vendedorId])[0];
                ?>
                <p><strong>Clientes Asignados:</strong> <?= $stats['total_clientes'] ?></p>
                <p><strong>Total Facturado:</strong> <?= safeCurrency($stats['total_facturado']) ?></p>
                
                <hr>
                
                <a href="vendedores.php" class="btn btn-sm btn-outline-secondary w-100">
                    <i class="fas fa-list me-1"></i>Ver Lista de Vendedores
                </a>
            </div>
        </div>
    </div>
</div>

<?php
$extraJS = '
<script>
// Validación del formulario
(function() {
    "use strict";
    window.addEventListener("load", function() {
        var forms = document.getElementsByClassName("needs-validation");
        var validation = Array.prototype.filter.call(forms, function(form) {
            form.addEventListener("submit", function(event) {
                if (form.checkValidity() === false) {
                    event.preventDefault();
                    event.stopPropagation();
                }
                form.classList.add("was-validated");
            }, false);
        });
    }, false);
})();
</script>
';

include '../../includes/footer.php';
?>
