<?php
/**
 * Gestión de Usuarios (Administradores) - Administrador
 * Sistema de Gestión de Agua Mineral
 */

require_once '../../includes/init.php';

// Verificar que sea administrador
checkPermission('administrador');

$pageTitle = 'Gestión de Usuarios - Administrador';

$db = Database::getInstance();

// Procesar acciones
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    
    if ($action === 'create') {
        $nombre = sanitize($_POST['nombre']);
        $usuario = sanitize($_POST['usuario']);
        $password = $_POST['password'];
        $rol = sanitize($_POST['rol']);
        
        // Validaciones
        $errors = [];
        
        if (empty($nombre)) $errors[] = 'El nombre es obligatorio.';
        if (empty($usuario)) $errors[] = 'El usuario es obligatorio.';
        if (empty($password)) $errors[] = 'La contraseña es obligatoria.';
        if (strlen($password) < 6) $errors[] = 'La contraseña debe tener al menos 6 caracteres.';
        if (!in_array($rol, ['administrador', 'vendedor'])) $errors[] = 'Rol no válido.';
        
        // Verificar que el usuario no exista
        if ($usuario) {
            $existing = $db->select("SELECT id FROM usuarios WHERE usuario = ?", [$usuario]);
            if (!empty($existing)) {
                $errors[] = 'El nombre de usuario ya existe.';
            }
        }
        
        if (empty($errors)) {
            $hashedPassword = hashPassword($password);
            $query = "INSERT INTO usuarios (nombre, usuario, password, rol, fecha_registro) 
                      VALUES (?, ?, ?, ?, NOW())";
            
            if ($db->execute($query, [$nombre, $usuario, $hashedPassword, $rol])) {
                logActivity($_SESSION['user_id'], 'create_user', "Usuario creado: $nombre (Rol: $rol)");
                setFlashMessage('Usuario creado exitosamente.', 'success');
                redirect('usuarios.php');
            } else {
                setFlashMessage('Error al crear el usuario.', 'error');
            }
        } else {
            foreach ($errors as $error) {
                setFlashMessage($error, 'error');
            }
        }
    }
    
    if ($action === 'toggle_status') {
        $userId = (int)$_POST['user_id'];
        $newStatus = (int)$_POST['new_status'];
        
        // No permitir desactivar al propio usuario
        if ($userId === $_SESSION['user_id']) {
            setFlashMessage('No puedes desactivar tu propia cuenta.', 'error');
            redirect('usuarios.php');
        }
        
        $query = "UPDATE usuarios SET activo = ? WHERE id = ?";
        if ($db->execute($query, [$newStatus, $userId])) {
            $status = $newStatus ? 'activado' : 'desactivado';
            logActivity($_SESSION['user_id'], 'toggle_user_status', "Usuario $status: ID $userId");
            setFlashMessage("Usuario $status exitosamente.", 'success');
        } else {
            setFlashMessage('Error al cambiar el estado del usuario.', 'error');
        }
        redirect('usuarios.php');
    }
    
    if ($action === 'delete_user') {
        $userId = (int)$_POST['user_id'];
        
        // No permitir eliminar al propio usuario
        if ($userId === $_SESSION['user_id']) {
            setFlashMessage('No puedes eliminar tu propia cuenta.', 'error');
            redirect('usuarios.php');
        }
        
        // Verificar si es vendedor con clientes asignados
        $user = $db->select("SELECT rol FROM usuarios WHERE id = ?", [$userId]);
        if (!empty($user) && $user[0]['rol'] === 'vendedor') {
            $hasClients = $db->select("SELECT COUNT(*) as count FROM clientes WHERE vendedor_id = ?", [$userId])[0]['count'];
            if ($hasClients > 0) {
                setFlashMessage('No se puede eliminar este vendedor porque tiene clientes asignados. Desactívelo en su lugar.', 'error');
                redirect('usuarios.php');
            }
        }
        
        if ($db->execute("DELETE FROM usuarios WHERE id = ?", [$userId])) {
            logActivity($_SESSION['user_id'], 'delete_user', "Usuario eliminado permanentemente: ID $userId");
            setFlashMessage('Usuario eliminado permanentemente.', 'success');
        } else {
            setFlashMessage('Error al eliminar el usuario.', 'error');
        }
        redirect('usuarios.php');
    }
}

// Obtener todos los usuarios (administradores y otros)
$usuarios = $db->select("
    SELECT 
        u.*,
        z.nombre as zona_nombre
    FROM usuarios u
    LEFT JOIN zonas z ON u.zona_id = z.id
    ORDER BY u.rol ASC, u.nombre ASC
");

// Obtener zonas para el formulario
$zonas = $db->select("SELECT id, nombre FROM zonas ORDER BY nombre");

include '../../includes/header.php';
?>

<div class="row">
    <div class="col-12">
        <div class="d-flex justify-content-between align-items-center mb-4">
            <h1 class="h3">
                <i class="fas fa-users-cog me-2"></i>Gestión de Usuarios
            </h1>
            <button type="button" class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#nuevoUsuarioModal">
                <i class="fas fa-plus me-2"></i>Nuevo Usuario
            </button>
        </div>
    </div>
</div>

<!-- Lista de usuarios -->
<div class="card">
    <div class="card-header">
        <h5 class="mb-0">Lista de Usuarios del Sistema</h5>
    </div>
    <div class="card-body">
        <div class="table-responsive">
            <table class="table table-striped table-hover" id="usuariosTable">
                <thead class="table-dark">
                    <tr>
                        <th>Usuario</th>
                        <th>Rol</th>
                        <th>Contacto</th>
                        <th>Zona</th>
                        <th>Estado</th>
                        <th>Acciones</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($usuarios as $user): ?>
                        <tr class="<?= $user['activo'] ? '' : 'table-secondary' ?>">
                            <td>
                                <div class="fw-bold"><?= htmlspecialchars($user['nombre']) ?></div>
                                <small class="text-muted">@<?= htmlspecialchars($user['usuario']) ?></small>
                            </td>
                            <td>
                                <span class="badge bg-<?= $user['rol'] === 'administrador' ? 'danger' : 'info' ?>">
                                    <?= ucfirst($user['rol']) ?>
                                </span>
                            </td>
                            <td>
                                <?php if ($user['telefono']): ?>
                                    <div><i class="fas fa-phone me-1"></i><?= htmlspecialchars($user['telefono']) ?></div>
                                <?php endif; ?>
                                <?php if ($user['email']): ?>
                                    <div><i class="fas fa-envelope me-1"></i><?= htmlspecialchars($user['email']) ?></div>
                                <?php endif; ?>
                            </td>
                            <td>
                                <?= htmlspecialchars($user['zona_nombre'] ?? 'N/A') ?>
                            </td>
                            <td>
                                <span class="badge bg-<?= $user['activo'] ? 'success' : 'secondary' ?>">
                                    <?= $user['activo'] ? 'Activo' : 'Inactivo' ?>
                                </span>
                            </td>
                            <td>
                                <div class="btn-group btn-group-sm">
                                    <button type="button" class="btn btn-outline-primary" 
                                            onclick="editarUsuario(<?= $user['id'] ?>)" 
                                            title="Editar">
                                        <i class="fas fa-edit"></i>
                                    </button>
                                    
                                    <?php if ($user['id'] !== $_SESSION['user_id']): ?>
                                        <button type="button" class="btn btn-outline-danger" 
                                                onclick="deleteUser(<?= $user['id'] ?>, '<?= htmlspecialchars($user['nombre']) ?>')"
                                                title="Eliminar permanentemente">
                                            <i class="fas fa-trash"></i>
                                        </button>
                                        
                                        <button type="button" class="btn btn-outline-<?= $user['activo'] ? 'warning' : 'success' ?>" 
                                                onclick="toggleUserStatus(<?= $user['id'] ?>, <?= $user['activo'] ? 0 : 1 ?>)"
                                                title="<?= $user['activo'] ? 'Desactivar' : 'Activar' ?>">
                                            <i class="fas fa-<?= $user['activo'] ? 'pause' : 'play' ?>"></i>
                                        </button>
                                    <?php else: ?>
                                        <button type="button" class="btn btn-outline-secondary" disabled title="No puedes modificar tu propia cuenta">
                                            <i class="fas fa-lock"></i>
                                        </button>
                                    <?php endif; ?>
                                </div>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
    </div>
</div>

<!-- Modal Nuevo Usuario -->
<div class="modal fade" id="nuevoUsuarioModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Nuevo Usuario</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <form method="POST" class="needs-validation" novalidate>
                <input type="hidden" name="action" value="create">
                <div class="modal-body">
                    <div class="mb-3">
                        <label for="nombre" class="form-label">Nombre Completo *</label>
                        <input type="text" name="nombre" id="nombre" class="form-control" required>
                        <div class="invalid-feedback">
                            Por favor ingrese el nombre completo.
                        </div>
                    </div>
                    
                    <div class="mb-3">
                        <label for="usuario" class="form-label">Usuario *</label>
                        <input type="text" name="usuario" id="usuario" class="form-control" required>
                        <div class="invalid-feedback">
                            Por favor ingrese un nombre de usuario.
                        </div>
                    </div>
                    
                    <div class="mb-3">
                        <label for="password" class="form-label">Contraseña *</label>
                        <input type="password" name="password" id="password" class="form-control" 
                               minlength="6" required>
                        <div class="invalid-feedback">
                            La contraseña debe tener al menos 6 caracteres.
                        </div>
                    </div>
                    
                    <div class="mb-3">
                        <label for="rol" class="form-label">Rol *</label>
                        <select name="rol" id="rol" class="form-select" required>
                            <option value="administrador">Administrador</option>
                            <option value="vendedor">Vendedor</option>
                        </select>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancelar</button>
                    <button type="submit" class="btn btn-primary">Crear Usuario</button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Modal de confirmación de estado -->
<div class="modal fade" id="confirmStatusModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Confirmar Acción</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <p id="confirmMessage"></p>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancelar</button>
                <form method="POST" style="display: inline;">
                    <input type="hidden" name="action" value="toggle_status">
                    <input type="hidden" name="user_id" id="confirmUserId">
                    <input type="hidden" name="new_status" id="confirmNewStatus">
                    <button type="submit" class="btn btn-primary">Confirmar</button>
                </form>
            </div>
        </div>
    </div>
</div>

<!-- Modal de confirmación de eliminación -->
<div class="modal fade" id="confirmDeleteModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title text-danger">Eliminar Usuario</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <p class="text-danger fw-bold">ADVERTENCIA: Esta acción es irreversible.</p>
                <p>¿Está seguro de que desea eliminar permanentemente al usuario <strong id="deleteUserName"></strong>?</p>
                <p class="small text-muted">Si es un vendedor con clientes, no se podrá eliminar.</p>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancelar</button>
                <form method="POST" style="display: inline;">
                    <input type="hidden" name="action" value="delete_user">
                    <input type="hidden" name="user_id" id="deleteUserId">
                    <button type="submit" class="btn btn-danger">Eliminar Permanentemente</button>
                </form>
            </div>
        </div>
    </div>
</div>

<?php
$extraJS = '
<script>
function editarUsuario(userId) {
    // Implementar edición
    alert("Función de edición en desarrollo. Usuario ID: " + userId);
}

function toggleUserStatus(userId, newStatus) {
    const action = newStatus ? "activar" : "desactivar";
    document.getElementById("confirmMessage").textContent = 
        `¿Está seguro que desea ${action} este usuario?`;
    document.getElementById("confirmUserId").value = userId;
    document.getElementById("confirmNewStatus").value = newStatus;
    
    const modal = new bootstrap.Modal(document.getElementById("confirmStatusModal"));
    modal.show();
}

function deleteUser(userId, userName) {
    document.getElementById("deleteUserId").value = userId;
    document.getElementById("deleteUserName").textContent = userName;
    
    const modal = new bootstrap.Modal(document.getElementById("confirmDeleteModal"));
    modal.show();
}

// DataTable para mejor navegación
$(document).ready(function() {
    $("#usuariosTable").DataTable({
        "language": {
            "url": "//cdn.datatables.net/plug-ins/1.10.24/i18n/Spanish.json"
        },
        "pageLength": 25,
        "order": [[0, "asc"]],
        "columnDefs": [
            { "orderable": false, "targets": [5] }
        ]
    });
});
</script>
<link rel="stylesheet" type="text/css" href="https://cdn.datatables.net/1.10.24/css/dataTables.bootstrap5.min.css">
<script type="text/javascript" charset="utf8" src="https://cdn.datatables.net/1.10.24/js/jquery.dataTables.js"></script>
<script type="text/javascript" charset="utf8" src="https://cdn.datatables.net/1.10.24/js/dataTables.bootstrap5.min.js"></script>
';

include '../../includes/footer.php';
?>
