<?php
/**
 * Configuración del Sistema - Administrador
 * Sistema de Gestión de Agua Mineral
 */

require_once '../../includes/init.php';

// Verificar que sea administrador
checkPermission('administrador');

$pageTitle = 'Configuración del Sistema - Administrador';

$db = Database::getInstance();

// Procesar formularios
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    
    if ($action === 'update_prices') {
        $productoId = (int)$_POST['producto_id'];
        $nuevoPrecio = (float)$_POST['nuevo_precio'];
        
        if ($nuevoPrecio > 0) {
            $query = "UPDATE productos SET precio = ? WHERE id = ?";
            if ($db->execute($query, [$nuevoPrecio, $productoId])) {
                setFlashMessage('Precio actualizado correctamente.', 'success');
                logActivity($_SESSION['user_id'], 'update_price', "Precio actualizado a $nuevoPrecio para producto ID $productoId");
            } else {
                setFlashMessage('Error al actualizar el precio.', 'error');
            }
        } else {
            setFlashMessage('El precio debe ser mayor a 0.', 'error');
        }
        redirect('configuracion.php');
    }
    
    if ($action === 'create_zone') {
        $nombreZona = sanitize($_POST['nombre_zona']);
        
        if (!empty($nombreZona)) {
            $query = "INSERT INTO zonas (nombre) VALUES (?)";
            if ($db->execute($query, [$nombreZona])) {
                setFlashMessage('Zona creada correctamente.', 'success');
                logActivity($_SESSION['user_id'], 'create_zone', "Zona creada: $nombreZona");
            } else {
                setFlashMessage('Error al crear la zona.', 'error');
            }
        } else {
            setFlashMessage('El nombre de la zona es requerido.', 'error');
        }
        redirect('configuracion.php');
    }
    
    if ($action === 'update_zone') {
        $zonaId = (int)$_POST['zona_id'];
        $nombreZona = sanitize($_POST['nombre_zona']);
        
        if (!empty($nombreZona)) {
            $query = "UPDATE zonas SET nombre = ? WHERE id = ?";
            if ($db->execute($query, [$nombreZona, $zonaId])) {
                setFlashMessage('Zona actualizada correctamente.', 'success');
                logActivity($_SESSION['user_id'], 'update_zone', "Zona actualizada: $nombreZona");
            } else {
                setFlashMessage('Error al actualizar la zona.', 'error');
            }
        } else {
            setFlashMessage('El nombre de la zona es requerido.', 'error');
        }
        redirect('configuracion.php');
    }
    
    if ($action === 'create_user') {
        $nombre = sanitize($_POST['nombre']);
        $usuario = sanitize($_POST['usuario']);
        $password = $_POST['password'];
        $rol = $_POST['rol'];
        $zonaId = !empty($_POST['zona_id']) ? (int)$_POST['zona_id'] : null;
        
        if (!empty($nombre) && !empty($usuario) && !empty($password)) {
            // Verificar que el usuario no exista
            $checkQuery = "SELECT COUNT(*) as count FROM usuarios WHERE usuario = ?";
            $checkResult = $db->select($checkQuery, [$usuario]);
            
            if ($checkResult[0]['count'] == 0) {
                $hashedPassword = hashPassword($password);
                $query = "INSERT INTO usuarios (nombre, usuario, password, rol, zona_id) VALUES (?, ?, ?, ?, ?)";
                
                if ($db->execute($query, [$nombre, $usuario, $hashedPassword, $rol, $zonaId])) {
                    setFlashMessage('Usuario creado correctamente.', 'success');
                    logActivity($_SESSION['user_id'], 'create_user', "Usuario creado: $usuario ($rol)");
                } else {
                    setFlashMessage('Error al crear el usuario.', 'error');
                }
            } else {
                setFlashMessage('El nombre de usuario ya existe.', 'error');
            }
        } else {
            setFlashMessage('Todos los campos son requeridos.', 'error');
        }
        redirect('configuracion.php');
    }
    
    if ($action === 'toggle_user') {
        $userId = (int)$_POST['user_id'];
        $nuevoEstado = (int)$_POST['nuevo_estado'];
        
        // No permitir desactivar al usuario actual
        if ($userId != $_SESSION['user_id']) {
            $query = "UPDATE usuarios SET activo = ? WHERE id = ?";
            if ($db->execute($query, [$nuevoEstado, $userId])) {
                $estado = $nuevoEstado ? 'activado' : 'desactivado';
                setFlashMessage("Usuario $estado correctamente.", 'success');
                logActivity($_SESSION['user_id'], 'toggle_user', "Usuario $estado: ID $userId");
            } else {
                setFlashMessage('Error al cambiar el estado del usuario.', 'error');
            }
        } else {
            setFlashMessage('No puede desactivar su propio usuario.', 'error');
        }
        redirect('configuracion.php');
    }
}

// Obtener datos para mostrar
$productos = $db->select("SELECT * FROM productos ORDER BY nombre");
$zonas = $db->select("SELECT * FROM zonas ORDER BY nombre");
$usuarios = $db->select("
    SELECT u.*, z.nombre as zona_nombre 
    FROM usuarios u 
    LEFT JOIN zonas z ON u.zona_id = z.id 
    ORDER BY u.rol, u.nombre
");

include '../../includes/header.php';
?>

<div class="row">
    <div class="col-12">
        <h1 class="h3 mb-4">
            <i class="fas fa-cog me-2"></i>Configuración del Sistema
        </h1>
    </div>
</div>

<!-- Tabs de configuración -->
<ul class="nav nav-tabs" id="configTabs" role="tablist">
    <li class="nav-item" role="presentation">
        <button class="nav-link active" id="precios-tab" data-bs-toggle="tab" data-bs-target="#precios" type="button">
            <i class="fas fa-dollar-sign me-2"></i>Precios
        </button>
    </li>
    <li class="nav-item" role="presentation">
        <button class="nav-link" id="zonas-tab" data-bs-toggle="tab" data-bs-target="#zonas" type="button">
            <i class="fas fa-map me-2"></i>Zonas
        </button>
    </li>
    <li class="nav-item" role="presentation">
        <button class="nav-link" id="usuarios-tab" data-bs-toggle="tab" data-bs-target="#usuarios" type="button">
            <i class="fas fa-users me-2"></i>Usuarios
        </button>
    </li>
</ul>

<div class="tab-content" id="configTabsContent">
    <!-- Tab de Precios -->
    <div class="tab-pane fade show active" id="precios" role="tabpanel">
        <div class="card mt-3">
            <div class="card-header">
                <h5 class="mb-0">Gestión de Precios</h5>
            </div>
            <div class="card-body">
                <div class="table-responsive">
                    <table class="table table-striped">
                        <thead>
                            <tr>
                                <th>Producto</th>
                                <th>Precio Actual</th>
                                <th>Nuevo Precio</th>
                                <th>Acciones</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($productos as $producto): ?>
                                <tr>
                                    <td><?= htmlspecialchars($producto['nombre']) ?></td>
                                    <td>
                                        <span class="badge bg-primary fs-6">
                                            <?= formatCurrency($producto['precio']) ?>
                                        </span>
                                    </td>
                                    <td>
                                        <form method="POST" class="d-inline-flex align-items-center">
                                            <input type="hidden" name="action" value="update_prices">
                                            <input type="hidden" name="producto_id" value="<?= $producto['id'] ?>">
                                            <div class="input-group input-group-sm" style="width: 150px;">
                                                <span class="input-group-text">$</span>
                                                <input type="number" name="nuevo_precio" class="form-control" 
                                                       step="0.01" min="0.01" value="<?= $producto['precio'] ?>" required>
                                            </div>
                                    </td>
                                    <td>
                                            <button type="submit" class="btn btn-sm btn-success">
                                                <i class="fas fa-save me-1"></i>Actualizar
                                            </button>
                                        </form>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </div>

    <!-- Tab de Zonas -->
    <div class="tab-pane fade" id="zonas" role="tabpanel">
        <div class="row mt-3">
            <div class="col-md-6">
                <div class="card">
                    <div class="card-header">
                        <h5 class="mb-0">Crear Nueva Zona</h5>
                    </div>
                    <div class="card-body">
                        <form method="POST">
                            <input type="hidden" name="action" value="create_zone">
                            <div class="mb-3">
                                <label for="nombre_zona" class="form-label">Nombre de la Zona</label>
                                <input type="text" name="nombre_zona" id="nombre_zona" class="form-control" required>
                            </div>
                            <button type="submit" class="btn btn-primary">
                                <i class="fas fa-plus me-2"></i>Crear Zona
                            </button>
                        </form>
                    </div>
                </div>
            </div>
            
            <div class="col-md-6">
                <div class="card">
                    <div class="card-header">
                        <h5 class="mb-0">Zonas Existentes</h5>
                    </div>
                    <div class="card-body">
                        <?php if (empty($zonas)): ?>
                            <p class="text-muted">No hay zonas creadas.</p>
                        <?php else: ?>
                            <?php foreach ($zonas as $zona): ?>
                                <div class="d-flex justify-content-between align-items-center mb-2 p-2 border rounded">
                                    <form method="POST" class="d-flex align-items-center flex-grow-1">
                                        <input type="hidden" name="action" value="update_zone">
                                        <input type="hidden" name="zona_id" value="<?= $zona['id'] ?>">
                                        <input type="text" name="nombre_zona" class="form-control form-control-sm me-2" 
                                               value="<?= htmlspecialchars($zona['nombre']) ?>" required>
                                        <button type="submit" class="btn btn-sm btn-outline-primary">
                                            <i class="fas fa-save"></i>
                                        </button>
                                    </form>
                                </div>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Tab de Usuarios -->
    <div class="tab-pane fade" id="usuarios" role="tabpanel">
        <div class="row mt-3">
            <div class="col-md-4">
                <div class="card">
                    <div class="card-header">
                        <h5 class="mb-0">Crear Nuevo Usuario</h5>
                    </div>
                    <div class="card-body">
                        <form method="POST">
                            <input type="hidden" name="action" value="create_user">
                            
                            <div class="mb-3">
                                <label for="nombre" class="form-label">Nombre Completo</label>
                                <input type="text" name="nombre" id="nombre" class="form-control" required>
                            </div>
                            
                            <div class="mb-3">
                                <label for="usuario" class="form-label">Usuario</label>
                                <input type="text" name="usuario" id="usuario" class="form-control" required>
                            </div>
                            
                            <div class="mb-3">
                                <label for="password" class="form-label">Contraseña</label>
                                <input type="password" name="password" id="password" class="form-control" required>
                            </div>
                            
                            <div class="mb-3">
                                <label for="rol" class="form-label">Rol</label>
                                <select name="rol" id="rol" class="form-select" required>
                                    <option value="">Seleccionar rol</option>
                                    <option value="administrador">Administrador</option>
                                    <option value="vendedor">Vendedor</option>
                                </select>
                            </div>
                            
                            <div class="mb-3">
                                <label for="zona_id" class="form-label">Zona (solo vendedores)</label>
                                <select name="zona_id" id="zona_id" class="form-select">
                                    <option value="">Sin zona asignada</option>
                                    <?php foreach ($zonas as $zona): ?>
                                        <option value="<?= $zona['id'] ?>"><?= htmlspecialchars($zona['nombre']) ?></option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                            
                            <button type="submit" class="btn btn-primary">
                                <i class="fas fa-user-plus me-2"></i>Crear Usuario
                            </button>
                        </form>
                    </div>
                </div>
            </div>
            
            <div class="col-md-8">
                <div class="card">
                    <div class="card-header">
                        <h5 class="mb-0">Usuarios del Sistema</h5>
                    </div>
                    <div class="card-body">
                        <div class="table-responsive">
                            <table class="table table-striped">
                                <thead>
                                    <tr>
                                        <th>Nombre</th>
                                        <th>Usuario</th>
                                        <th>Rol</th>
                                        <th>Zona</th>
                                        <th>Estado</th>
                                        <th>Acciones</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($usuarios as $usuario): ?>
                                        <tr>
                                            <td><?= htmlspecialchars($usuario['nombre']) ?></td>
                                            <td><?= htmlspecialchars($usuario['usuario']) ?></td>
                                            <td>
                                                <span class="badge bg-<?= $usuario['rol'] === 'administrador' ? 'danger' : 'info' ?>">
                                                    <?= ucfirst($usuario['rol']) ?>
                                                </span>
                                            </td>
                                            <td><?= htmlspecialchars($usuario['zona_nombre'] ?? 'Sin zona') ?></td>
                                            <td>
                                                <?php if ($usuario['activo']): ?>
                                                    <span class="badge bg-success">Activo</span>
                                                <?php else: ?>
                                                    <span class="badge bg-secondary">Inactivo</span>
                                                <?php endif; ?>
                                            </td>
                                            <td>
                                                <?php if ($usuario['id'] != $_SESSION['user_id']): ?>
                                                    <button type="button" class="btn btn-sm btn-outline-<?= $usuario['activo'] ? 'warning' : 'success' ?>" 
                                                            onclick="toggleUserStatus(<?= $usuario['id'] ?>, <?= $usuario['activo'] ? 0 : 1 ?>)">
                                                        <i class="fas fa-<?= $usuario['activo'] ? 'pause' : 'play' ?>"></i>
                                                        <?= $usuario['activo'] ? 'Desactivar' : 'Activar' ?>
                                                    </button>
                                                <?php else: ?>
                                                    <span class="text-muted">Usuario actual</span>
                                                <?php endif; ?>
                                            </td>
                                        </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Modal de confirmación para usuarios -->
<div class="modal fade" id="confirmUserModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Confirmar Acción</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <p id="confirmUserMessage"></p>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancelar</button>
                <form method="POST" style="display: inline;">
                    <input type="hidden" name="action" value="toggle_user">
                    <input type="hidden" name="user_id" id="confirmUserId">
                    <input type="hidden" name="nuevo_estado" id="confirmUserEstado">
                    <button type="submit" class="btn btn-primary">Confirmar</button>
                </form>
            </div>
        </div>
    </div>
</div>

<?php
$extraJS = '
<script>
function toggleUserStatus(userId, nuevoEstado) {
    const action = nuevoEstado ? "activar" : "desactivar";
    document.getElementById("confirmUserMessage").textContent = 
        `¿Está seguro que desea ${action} este usuario?`;
    document.getElementById("confirmUserId").value = userId;
    document.getElementById("confirmUserEstado").value = nuevoEstado;
    
    const modal = new bootstrap.Modal(document.getElementById("confirmUserModal"));
    modal.show();
}

// Mostrar/ocultar zona según el rol seleccionado
document.getElementById("rol").addEventListener("change", function() {
    const zonaField = document.getElementById("zona_id").closest(".mb-3");
    if (this.value === "vendedor") {
        zonaField.style.display = "block";
    } else {
        zonaField.style.display = "none";
        document.getElementById("zona_id").value = "";
    }
});
</script>
';

include '../../includes/footer.php';
?>
