<?php
/**
 * Gestión de Clientes - Administrador
 * Sistema de Gestión de Agua Mineral
 */

require_once '../../includes/init.php';

// Verificar que sea administrador
checkPermission('administrador');

$pageTitle = 'Gestión de Clientes - Administrador';

$db = Database::getInstance();

// Procesar acciones
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    
    if ($action === 'toggle_status') {
        $clienteId = (int)$_POST['cliente_id'];
        $nuevoEstado = (int)$_POST['nuevo_estado'];
        
        $query = "UPDATE clientes SET activo = ? WHERE id = ?";
        if ($db->execute($query, [$nuevoEstado, $clienteId])) {
            $estado = $nuevoEstado ? 'activado' : 'desactivado';
            logActivity($_SESSION['user_id'], 'toggle_client_status', "Cliente $estado: ID $clienteId");
            setFlashMessage("Cliente $estado correctamente.", 'success');
        } else {
            setFlashMessage('Error al cambiar el estado del cliente.', 'error');
        }
        redirect('clientes.php');
    }
    
    if ($action === 'delete_client') {
        $clienteId = (int)$_POST['cliente_id'];
        
        // Verificar si tiene ventas o pagos asociados (para evitar borrado accidental)
        $hasVentas = $db->select("SELECT COUNT(*) as count FROM ventas WHERE cliente_id = ?", [$clienteId])[0]['count'];
        $hasPagos = $db->select("SELECT COUNT(*) as count FROM pagos WHERE cliente_id = ?", [$clienteId])[0]['count'];
        
        if ($hasVentas > 0 || $hasPagos > 0) {
            setFlashMessage('No se puede eliminar el cliente porque tiene ventas o pagos asociados. Desactívelo en su lugar.', 'error');
        } else {
            if ($db->execute("DELETE FROM clientes WHERE id = ?", [$clienteId])) {
                logActivity($_SESSION['user_id'], 'delete_client', "Cliente eliminado permanentemente: ID $clienteId");
                setFlashMessage('Cliente eliminado permanentemente.', 'success');
            } else {
                setFlashMessage('Error al eliminar el cliente.', 'error');
            }
        }
        redirect('clientes.php');
    }
}

// Filtros
$filtroVendedor = $_GET['vendedor'] ?? '';
$filtroEstado = $_GET['estado'] ?? '';
$busqueda = $_GET['busqueda'] ?? '';

// Construir consulta con filtros
$whereConditions = [];
$params = [];

if (!empty($filtroVendedor)) {
    $whereConditions[] = "c.vendedor_id = ?";
    $params[] = $filtroVendedor;
}

if ($filtroEstado !== '') {
    $whereConditions[] = "c.activo = ?";
    $params[] = $filtroEstado;
}

if (!empty($busqueda)) {
    $whereConditions[] = "(c.nombre LIKE ? OR c.direccion LIKE ?)";
    $params[] = "%$busqueda%";
    $params[] = "%$busqueda%";
}

$whereClause = !empty($whereConditions) ? 'WHERE ' . implode(' AND ', $whereConditions) : '';

// Obtener clientes con información del vendedor y resumen
$query = "
    SELECT 
        c.*,
        u.nombre as vendedor_nombre,
        z.nombre as zona_nombre,
        (SELECT COUNT(*) FROM ventas v WHERE v.cliente_id = c.id) as total_entregas,
        (SELECT SUM(v.cantidad_bidones) FROM ventas v WHERE v.cliente_id = c.id) as total_bidones,
        (SELECT SUM(v.total) FROM ventas v WHERE v.cliente_id = c.id) as total_ventas,
        (SELECT SUM(p.monto) FROM pagos p WHERE p.cliente_id = c.id) as total_pagos,
        (SELECT COUNT(*) FROM ventas v WHERE v.cliente_id = c.id AND MONTH(v.fecha) = MONTH(CURDATE()) AND YEAR(v.fecha) = YEAR(CURDATE())) as entregas_mes,
        (SELECT SUM(v.cantidad_bidones) FROM ventas v WHERE v.cliente_id = c.id AND MONTH(v.fecha) = MONTH(CURDATE()) AND YEAR(v.fecha) = YEAR(CURDATE())) as bidones_mes
    FROM clientes c
    INNER JOIN usuarios u ON c.vendedor_id = u.id
    LEFT JOIN zonas z ON u.zona_id = z.id
    $whereClause
    ORDER BY c.nombre ASC
";

$clientes = $db->select($query, $params);

// Obtener vendedores para el filtro
$vendedores = $db->select("SELECT id, nombre FROM usuarios WHERE rol = 'vendedor' AND activo = 1 ORDER BY nombre");

include '../../includes/header.php';
?>

<div class="row">
    <div class="col-12">
        <div class="d-flex justify-content-between align-items-center mb-4">
            <h1 class="h3">
                <i class="fas fa-users me-2"></i>Gestión de Clientes
            </h1>
            <a href="cliente-nuevo.php" class="btn btn-primary">
                <i class="fas fa-plus me-2"></i>Nuevo Cliente
            </a>
        </div>
    </div>
</div>

<!-- Filtros -->
<div class="card mb-4">
    <div class="card-body">
        <form method="GET" class="row g-3">
            <div class="col-md-3">
                <label for="vendedor" class="form-label">Vendedor</label>
                <select name="vendedor" id="vendedor" class="form-select">
                    <option value="">Todos los vendedores</option>
                    <?php foreach ($vendedores as $vendedor): ?>
                        <option value="<?= $vendedor['id'] ?>" <?= $filtroVendedor == $vendedor['id'] ? 'selected' : '' ?>>
                            <?= htmlspecialchars($vendedor['nombre']) ?>
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>
            
            <div class="col-md-2">
                <label for="estado" class="form-label">Estado</label>
                <select name="estado" id="estado" class="form-select">
                    <option value="">Todos</option>
                    <option value="1" <?= $filtroEstado === '1' ? 'selected' : '' ?>>Activos</option>
                    <option value="0" <?= $filtroEstado === '0' ? 'selected' : '' ?>>Inactivos</option>
                </select>
            </div>
            
            <div class="col-md-4">
                <label for="busqueda" class="form-label">Buscar</label>
                <input type="text" name="busqueda" id="busqueda" class="form-control" 
                       placeholder="Nombre o dirección del cliente" value="<?= htmlspecialchars($busqueda) ?>">
            </div>
            
            <div class="col-md-3 d-flex align-items-end">
                <button type="submit" class="btn btn-outline-primary me-2">
                    <i class="fas fa-search me-1"></i>Filtrar
                </button>
                <a href="clientes.php" class="btn btn-outline-secondary">
                    <i class="fas fa-times me-1"></i>Limpiar
                </a>
            </div>
        </form>
    </div>
</div>

<!-- Tabla de clientes -->
<div class="card">
    <div class="card-body">
        <div class="table-responsive">
            <table class="table table-hover" id="clientesTable">
                <thead class="table-light">
                    <tr>
                        <th>Cliente</th>
                        <th>Vendedor</th>
                        <th>Zona</th>
                        <th>Entregas Mes</th>
                        <th>Bidones Mes</th>
                        <th>Total Histórico</th>
                        <th>Saldo</th>
                        <th>Estado</th>
                        <th>Acciones</th>
                    </tr>
                </thead>
                <tbody>
                    <?php if (empty($clientes)): ?>
                        <tr>
                            <td colspan="9" class="text-center text-muted py-4">
                                No se encontraron clientes con los filtros aplicados.
                            </td>
                        </tr>
                    <?php else: ?>
                        <?php foreach ($clientes as $cliente): ?>
                            <?php
                            $saldo = ($cliente['total_ventas'] ?? 0) - ($cliente['total_pagos'] ?? 0);
                            $saldoClass = $saldo > 0 ? 'text-danger' : ($saldo < 0 ? 'text-success' : 'text-muted');
                            ?>
                            <tr>
                                <td>
                                    <div>
                                        <strong><?= htmlspecialchars($cliente['nombre']) ?></strong>
                                        <br>
                                        <small class="text-muted"><?= htmlspecialchars($cliente['direccion']) ?></small>
                                    </div>
                                </td>
                                <td><?= htmlspecialchars($cliente['vendedor_nombre']) ?></td>
                                <td><?= htmlspecialchars($cliente['zona_nombre'] ?? 'Sin zona') ?></td>
                                <td>
                                    <span class="badge bg-info">
                                        <?= number_format($cliente['entregas_mes'] ?? 0) ?>
                                    </span>
                                </td>
                                <td>
                                    <span class="badge bg-primary">
                                        <?= number_format($cliente['bidones_mes'] ?? 0) ?>
                                    </span>
                                </td>
                                <td>
                                    <div>
                                        <small class="text-muted">
                                            <?= number_format($cliente['total_entregas'] ?? 0) ?> entregas
                                        </small>
                                        <br>
                                        <strong><?= formatCurrency($cliente['total_ventas'] ?? 0) ?></strong>
                                    </div>
                                </td>
                                <td>
                                    <span class="<?= $saldoClass ?>">
                                        <strong><?= formatCurrency($saldo) ?></strong>
                                    </span>
                                </td>
                                <td>
                                    <?php if ($cliente['activo']): ?>
                                        <span class="badge bg-success">Activo</span>
                                    <?php else: ?>
                                        <span class="badge bg-secondary">Inactivo</span>
                                    <?php endif; ?>
                                </td>
                                <td>
                                    <div class="btn-group btn-group-sm">
                                        <a href="cliente-editar.php?id=<?= $cliente['id'] ?>" 
                                           class="btn btn-outline-primary" title="Editar">
                                            <i class="fas fa-edit"></i>
                                        </a>
                                        <a href="cliente-detalle.php?id=<?= $cliente['id'] ?>" 
                                           class="btn btn-outline-info" title="Ver detalle">
                                            <i class="fas fa-eye"></i>
                                        </a>
                                        <?php if ($cliente['latitud'] && $cliente['longitud']): ?>
                                            <a href="https://maps.google.com/?q=<?= $cliente['latitud'] ?>,<?= $cliente['longitud'] ?>" 
                                               target="_blank" class="btn btn-outline-success" title="Ver en mapa">
                                                <i class="fas fa-map-marker-alt"></i>
                                            </a>
                                        <?php endif; ?>
                                        <button type="button" class="btn btn-outline-<?= $cliente['activo'] ? 'warning' : 'success' ?>" 
                                                onclick="toggleClientStatus(<?= $cliente['id'] ?>, <?= $cliente['activo'] ? 0 : 1 ?>)"
                                                title="<?= $cliente['activo'] ? 'Desactivar' : 'Activar' ?>">
                                            <i class="fas fa-<?= $cliente['activo'] ? 'pause' : 'play' ?>"></i>
                                        </button>
                                        <button type="button" class="btn btn-outline-danger" 
                                                onclick="deleteClient(<?= $cliente['id'] ?>, '<?= htmlspecialchars($cliente['nombre']) ?>')"
                                                title="Eliminar permanentemente">
                                            <i class="fas fa-trash"></i>
                                        </button>
                                    </div>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </tbody>
            </table>
        </div>
    </div>
</div>

<!-- Modal de confirmación de estado -->
<div class="modal fade" id="confirmStatusModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Confirmar Acción</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <p id="confirmMessage"></p>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancelar</button>
                <form method="POST" style="display: inline;">
                    <input type="hidden" name="action" value="toggle_status">
                    <input type="hidden" name="cliente_id" id="confirmClienteId">
                    <input type="hidden" name="nuevo_estado" id="confirmNuevoEstado">
                    <button type="submit" class="btn btn-primary">Confirmar</button>
                </form>
            </div>
        </div>
    </div>
</div>

<!-- Modal de confirmación de eliminación -->
<div class="modal fade" id="confirmDeleteModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title text-danger">Eliminar Cliente</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <p class="text-danger fw-bold">ADVERTENCIA: Esta acción es irreversible.</p>
                <p>¿Está seguro de que desea eliminar permanentemente al cliente <strong id="deleteClientName"></strong>?</p>
                <p class="small text-muted">Solo se podrá eliminar si no tiene ventas o pagos asociados.</p>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancelar</button>
                <form method="POST" style="display: inline;">
                    <input type="hidden" name="action" value="delete_client">
                    <input type="hidden" name="cliente_id" id="deleteClienteId">
                    <button type="submit" class="btn btn-danger">Eliminar Permanentemente</button>
                </form>
            </div>
        </div>
    </div>
</div>
<div class="modal fade" id="confirmStatusModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Confirmar Acción</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <p id="confirmMessage"></p>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancelar</button>
                <form method="POST" style="display: inline;">
                    <input type="hidden" name="action" value="toggle_status">
                    <input type="hidden" name="cliente_id" id="confirmClienteId">
                    <input type="hidden" name="nuevo_estado" id="confirmNuevoEstado">
                    <button type="submit" class="btn btn-primary">Confirmar</button>
                </form>
            </div>
        </div>
    </div>
</div>

<?php
$extraJS = '
<script>
function toggleClientStatus(clienteId, nuevoEstado) {
    const action = nuevoEstado ? "activar" : "desactivar";
    document.getElementById("confirmMessage").textContent = 
        `¿Está seguro que desea ${action} este cliente?`;
    document.getElementById("confirmClienteId").value = clienteId;
    document.getElementById("confirmNuevoEstado").value = nuevoEstado;
    
    const modal = new bootstrap.Modal(document.getElementById("confirmStatusModal"));
    modal.show();
}

function deleteClient(clienteId, clienteNombre) {
    document.getElementById("deleteClienteId").value = clienteId;
    document.getElementById("deleteClientName").textContent = clienteNombre;
    
    const modal = new bootstrap.Modal(document.getElementById("confirmDeleteModal"));
    modal.show();
}

// DataTable para mejor navegación
$(document).ready(function() {
    $("#clientesTable").DataTable({
        "language": {
            "url": "//cdn.datatables.net/plug-ins/1.10.24/i18n/Spanish.json"
        },
        "pageLength": 25,
        "order": [[0, "asc"]],
        "columnDefs": [
            { "orderable": false, "targets": [8] }
        ]
    });
});
</script>
<link rel="stylesheet" type="text/css" href="https://cdn.datatables.net/1.10.24/css/dataTables.bootstrap5.min.css">
<script type="text/javascript" charset="utf8" src="https://cdn.datatables.net/1.10.24/js/jquery.dataTables.js"></script>
<script type="text/javascript" charset="utf8" src="https://cdn.datatables.net/1.10.24/js/dataTables.bootstrap5.min.js"></script>
';

include '../../includes/footer.php';
?>
