<?php
/**
 * Funciones Auxiliares
 * Sistema de Gestión de Agua Mineral
 */

/**
 * Formatear número de forma segura (maneja valores nulos)
 */
function safeNumber($value, $decimals = 0) {
    if ($value === null || $value === '') {
        return 0;
    }
    return number_format((float)$value, $decimals, ',', '.');
}

/**
 * Formatear moneda de forma segura (Guaraníes paraguayos)
 */
function safeCurrency($value) {
    if ($value === null || $value === '') {
        return '₲ 0';
    }
    return '₲ ' . number_format((float)$value, 0, ',', '.');
}

/**
 * Formatear porcentaje de forma segura
 */
function safePercentage($value) {
    if ($value === null || $value === '') {
        return '0%';
    }
    return number_format((float)$value, 1, ',', '.') . '%';
}

/**
 * Obtener valor seguro de array (evita errores de índice inexistente)
 */
function safeArrayValue($array, $key, $default = null) {
    return isset($array[$key]) ? $array[$key] : $default;
}

/**
 * Validar si un valor es numérico y mayor que cero
 */
function isPositiveNumber($value) {
    return is_numeric($value) && (float)$value > 0;
}

/**
 * Calcular cambio porcentual de forma segura
 */
function calcularCambioSeguro($actual, $anterior) {
    $actual = (float)($actual ?? 0);
    $anterior = (float)($anterior ?? 0);
    
    if ($anterior == 0) {
        return $actual > 0 ? 100 : 0;
    }
    
    return round((($actual - $anterior) / $anterior) * 100, 1);
}

/**
 * Formatear fecha de forma segura
 */
function safeDate($date, $format = 'd/m/Y') {
    if (empty($date) || $date === '0000-00-00' || $date === '0000-00-00 00:00:00') {
        return '-';
    }
    
    try {
        return date($format, strtotime($date));
    } catch (Exception $e) {
        return '-';
    }
}

/**
 * Formatear texto de forma segura (evita XSS)
 */
function safeText($text, $maxLength = null) {
    $text = htmlspecialchars($text ?? '', ENT_QUOTES, 'UTF-8');
    
    if ($maxLength && strlen($text) > $maxLength) {
        return substr($text, 0, $maxLength) . '...';
    }
    
    return $text;
}

/**
 * Generar clase CSS para indicadores de cambio
 */
function getChangeClass($value) {
    $value = (float)($value ?? 0);
    
    if ($value > 0) {
        return 'text-success';
    } elseif ($value < 0) {
        return 'text-danger';
    } else {
        return 'text-muted';
    }
}

/**
 * Generar icono para indicadores de cambio
 */
function getChangeIcon($value) {
    $value = (float)($value ?? 0);
    
    if ($value > 0) {
        return 'fa-arrow-up';
    } elseif ($value < 0) {
        return 'fa-arrow-down';
    } else {
        return 'fa-minus';
    }
}

/**
 * Validar coordenadas GPS para Paraguay
 */
function validarCoordenadas($lat, $lng) {
    $lat = (float)$lat;
    $lng = (float)$lng;
    
    // Rangos aproximados para Paraguay
    return $lat >= -27.6 && $lat <= -19.3 && $lng >= -62.6 && $lng <= -54.3;
}

/**
 * Formatear número de bidones
 */
function formatBidones($cantidad) {
    $cantidad = (int)($cantidad ?? 0);
    return $cantidad . ($cantidad === 1 ? ' bidón' : ' bidones');
}

/**
 * Obtener estado del cliente basado en su actividad
 */
function getClienteEstado($ultimaVenta, $saldo) {
    $ultimaVenta = $ultimaVenta ? strtotime($ultimaVenta) : 0;
    $diasSinCompra = $ultimaVenta ? floor((time() - $ultimaVenta) / (60 * 60 * 24)) : 999;
    $saldo = (float)($saldo ?? 0);
    
    if ($saldo > 0) {
        return ['estado' => 'deudor', 'clase' => 'danger', 'texto' => 'Con deuda'];
    } elseif ($diasSinCompra > 30) {
        return ['estado' => 'inactivo', 'clase' => 'secondary', 'texto' => 'Inactivo'];
    } elseif ($diasSinCompra <= 7) {
        return ['estado' => 'activo', 'clase' => 'success', 'texto' => 'Activo'];
    } else {
        return ['estado' => 'regular', 'clase' => 'warning', 'texto' => 'Regular'];
    }
}

/**
 * Generar colores para gráficos
 */
function getChartColors($count) {
    $colors = [
        '#007bff', '#28a745', '#dc3545', '#ffc107', '#17a2b8',
        '#6f42c1', '#e83e8c', '#fd7e14', '#20c997', '#6c757d'
    ];
    
    $result = [];
    for ($i = 0; $i < $count; $i++) {
        $result[] = $colors[$i % count($colors)];
    }
    
    return $result;
}
?>
