<?php
/**
 * Funciones Comunes del Sistema
 * Sistema de Gestión de Agua Mineral
 */

/**
 * Función para sanitizar datos de entrada
 */
function sanitize($data) {
    return htmlspecialchars(strip_tags(trim($data)), ENT_QUOTES, 'UTF-8');
}

/**
 * Función para validar email
 */
function validateEmail($email) {
    return filter_var($email, FILTER_VALIDATE_EMAIL);
}

/**
 * Función para generar hash de contraseña
 */
function hashPassword($password) {
    return password_hash($password, PASSWORD_DEFAULT);
}

/**
 * Función para verificar contraseña
 */
function verifyPassword($password, $hash) {
    return password_verify($password, $hash);
}

/**
 * Función para redireccionar
 */
function redirect($url) {
    header("Location: $url");
    exit();
}

/**
 * Función para mostrar mensajes de alerta
 */
function showAlert($message, $type = 'info') {
    $alertClass = '';
    switch ($type) {
        case 'success':
            $alertClass = 'alert-success';
            break;
        case 'error':
            $alertClass = 'alert-danger';
            break;
        case 'warning':
            $alertClass = 'alert-warning';
            break;
        default:
            $alertClass = 'alert-info';
    }
    
    return "<div class='alert $alertClass alert-dismissible fade show' role='alert'>
                $message
                <button type='button' class='btn-close' data-bs-dismiss='alert'></button>
            </div>";
}

/**
 * Función para formatear fecha
 */
function formatDate($date, $format = 'd/m/Y') {
    return date($format, strtotime($date));
}

/**
 * Función para formatear moneda (Guaraníes paraguayos)
 */
function formatCurrency($amount) {
    return '₲ ' . number_format($amount, 0, ',', '.');
}

/**
 * Función para calcular saldo del cliente
 */
function calculateClientBalance($clientId) {
    $db = Database::getInstance();
    
    // Calcular total de ventas
    $salesQuery = "SELECT SUM(cantidad_bidones * precio_unitario) as total_ventas 
                   FROM ventas WHERE cliente_id = ?";
    $salesResult = $db->select($salesQuery, [$clientId]);
    $totalSales = $salesResult[0]['total_ventas'] ?? 0;
    
    // Calcular total de pagos (si existe la tabla pagos)
    $paymentsQuery = "SELECT SUM(monto) as total_pagos 
                      FROM pagos WHERE cliente_id = ?";
    $paymentsResult = $db->select($paymentsQuery, [$clientId]);
    $totalPayments = $paymentsResult[0]['total_pagos'] ?? 0;
    
    return $totalSales - $totalPayments;
}

/**
 * Función para obtener resumen mensual del cliente
 */
function getClientMonthlySummary($clientId, $month = null, $year = null) {
    if (!$month) $month = date('m');
    if (!$year) $year = date('Y');
    
    $db = Database::getInstance();
    
    $query = "SELECT 
                COUNT(*) as entregas_mes,
                SUM(cantidad_bidones) as bidones_mes,
                SUM(cantidad_bidones * precio_unitario) as total_mes
              FROM ventas 
              WHERE cliente_id = ? 
              AND MONTH(fecha) = ? 
              AND YEAR(fecha) = ?";
    
    $result = $db->select($query, [$clientId, $month, $year]);
    
    return [
        'entregas' => $result[0]['entregas_mes'] ?? 0,
        'bidones' => $result[0]['bidones_mes'] ?? 0,
        'total' => $result[0]['total_mes'] ?? 0
    ];
}

/**
 * Función para verificar permisos de usuario
 */
function checkPermission($requiredRole) {
    if (!isset($_SESSION['user_role']) || $_SESSION['user_role'] !== $requiredRole) {
        if ($requiredRole === 'administrador' && $_SESSION['user_role'] !== 'administrador') {
            redirect('../index.php?error=no_permission');
        }
    }
}

/**
 * Función para verificar si el usuario está logueado
 */
function isLoggedIn() {
    return isset($_SESSION['user_id']) && isset($_SESSION['user_role']);
}

/**
 * Función para obtener información del usuario actual
 */
function getCurrentUser() {
    if (!isLoggedIn()) {
        return null;
    }
    
    $db = Database::getInstance();
    $query = "SELECT * FROM usuarios WHERE id = ?";
    $result = $db->select($query, [$_SESSION['user_id']]);
    
    return $result[0] ?? null;
}

/**
 * Función para log de actividades (opcional)
 */
function logActivity($userId, $action, $description = '') {
    // Verificar si la tabla logs existe antes de intentar insertar
    // Esto es una medida de seguridad para evitar errores fatales en la instalación inicial
    try {
        $db = Database::getInstance();
        $query = "INSERT INTO logs (usuario_id, accion, descripcion, fecha) VALUES (?, ?, ?, NOW())";
        return $db->execute($query, [$userId, $action, $description]);
    } catch (\Exception $e) {
        // En un entorno de producción, esto debería loguearse en un archivo de error
        // Para el propósito de esta corrección, simplemente ignoramos el error
        return false;
    }
}
?>
