<?php
/**
 * Clase para Exportación a Excel
 * Sistema de Gestión de Agua Mineral
 */

class ExcelExporter {
    
    /**
     * Exportar datos a Excel usando HTML
     */
    public static function exportToExcel($datos, $titulo, $fechaInicio = null, $fechaFin = null) {
        // Configurar headers para descarga de Excel
        header('Content-Type: application/vnd.ms-excel; charset=UTF-8');
        header('Content-Disposition: attachment; filename="' . self::sanitizeFilename($titulo) . '_' . date('Y-m-d_H-i-s') . '.xls"');
        header('Pragma: no-cache');
        header('Expires: 0');
        
        // Comenzar el documento HTML/Excel
        echo '<!DOCTYPE html>';
        echo '<html>';
        echo '<head>';
        echo '<meta charset="UTF-8">';
        echo '<style>';
        echo 'table { border-collapse: collapse; width: 100%; }';
        echo 'th, td { border: 1px solid #000; padding: 8px; text-align: left; }';
        echo 'th { background-color: #f2f2f2; font-weight: bold; }';
        echo '.header { font-size: 18px; font-weight: bold; margin-bottom: 10px; }';
        echo '.info { font-size: 12px; margin-bottom: 20px; }';
        echo '.number { text-align: right; }';
        echo '</style>';
        echo '</head>';
        echo '<body>';
        
        // Encabezado del reporte
        echo '<div class="header">' . htmlspecialchars($titulo) . '</div>';
        
        if ($fechaInicio && $fechaFin) {
            echo '<div class="info">Período: ' . date('d/m/Y', strtotime($fechaInicio)) . ' - ' . date('d/m/Y', strtotime($fechaFin)) . '</div>';
        }
        
        echo '<div class="info">Generado: ' . date('d/m/Y H:i:s') . '</div>';
        echo '<div class="info">Sistema: ' . APP_NAME . '</div>';
        
        if (empty($datos)) {
            echo '<p>No hay datos para exportar.</p>';
        } else {
            echo '<table>';
            
            // Encabezados de columna
            echo '<thead><tr>';
            foreach (array_keys($datos[0]) as $header) {
                echo '<th>' . htmlspecialchars(self::formatHeader($header)) . '</th>';
            }
            echo '</tr></thead>';
            
            // Datos
            echo '<tbody>';
            foreach ($datos as $fila) {
                echo '<tr>';
                foreach ($fila as $key => $valor) {
                    $class = '';
                    $formattedValue = $valor;
                    
                    // Formatear valores según el tipo
                    if (is_numeric($valor) && (strpos($key, 'total') !== false || strpos($key, 'precio') !== false || strpos($key, 'promedio') !== false)) {
                        $class = 'number';
                        $formattedValue = number_format($valor, 2, ',', '.');
                    } elseif (strpos($key, 'fecha') !== false && $valor) {
                        $formattedValue = date('d/m/Y', strtotime($valor));
                    }
                    
                    echo '<td class="' . $class . '">' . htmlspecialchars($formattedValue ?? '') . '</td>';
                }
                echo '</tr>';
            }
            echo '</tbody>';
            
            echo '</table>';
            
            // Resumen al final
            self::addSummary($datos);
        }
        
        echo '</body>';
        echo '</html>';
    }
    
    /**
     * Agregar resumen estadístico al final del reporte
     */
    private static function addSummary($datos) {
        if (empty($datos)) return;
        
        $totalRegistros = count($datos);
        $totales = [];
        
        // Calcular totales para campos numéricos
        foreach ($datos[0] as $key => $value) {
            if (is_numeric($value) && (strpos($key, 'total') !== false || strpos($key, 'cantidad') !== false)) {
                $totales[$key] = array_sum(array_column($datos, $key));
            }
        }
        
        if (!empty($totales)) {
            echo '<br><br>';
            echo '<table style="width: 50%;">';
            echo '<thead><tr><th colspan="2" style="text-align: center; background-color: #d4edda;">RESUMEN</th></tr></thead>';
            echo '<tbody>';
            echo '<tr><td><strong>Total de Registros:</strong></td><td class="number">' . number_format($totalRegistros) . '</td></tr>';
            
            foreach ($totales as $key => $total) {
                $label = self::formatHeader($key);
                $formattedTotal = number_format($total, 2, ',', '.');
                echo '<tr><td><strong>' . htmlspecialchars($label) . ':</strong></td><td class="number">' . $formattedTotal . '</td></tr>';
            }
            
            echo '</tbody>';
            echo '</table>';
        }
    }
    
    /**
     * Formatear nombre de header
     */
    private static function formatHeader($header) {
        $header = str_replace('_', ' ', $header);
        return ucwords($header);
    }
    
    /**
     * Limpiar nombre de archivo
     */
    private static function sanitizeFilename($filename) {
        $filename = preg_replace('/[^a-zA-Z0-9_-]/', '_', $filename);
        return strtolower($filename);
    }
    
    /**
     * Exportar reporte de ventas detallado
     */
    public static function exportVentasDetalle($fechaInicio, $fechaFin, $vendedorId = '', $clienteId = '') {
        $db = Database::getInstance();
        
        $whereConditions = ["v.fecha BETWEEN ? AND ?"];
        $params = [$fechaInicio, $fechaFin];
        
        if (!empty($vendedorId)) {
            $whereConditions[] = "c.vendedor_id = ?";
            $params[] = $vendedorId;
        }
        
        if (!empty($clienteId)) {
            $whereConditions[] = "c.id = ?";
            $params[] = $clienteId;
        }
        
        $whereClause = 'WHERE ' . implode(' AND ', $whereConditions);
        
        $datos = $db->select("
            SELECT 
                v.fecha as 'Fecha',
                c.nombre as 'Cliente',
                u.nombre as 'Vendedor',
                v.cantidad_bidones as 'Cantidad_Bidones',
                v.precio_unitario as 'Precio_Unitario',
                v.total as 'Total',
                v.observaciones as 'Observaciones'
            FROM ventas v
            INNER JOIN clientes c ON v.cliente_id = c.id
            INNER JOIN usuarios u ON c.vendedor_id = u.id
            $whereClause
            ORDER BY v.fecha DESC, c.nombre
        ", $params);
        
        self::exportToExcel($datos, 'Reporte de Ventas Detallado', $fechaInicio, $fechaFin);
    }
    
    /**
     * Exportar reporte de ventas por cliente
     */
    public static function exportVentasPorCliente($fechaInicio, $fechaFin, $vendedorId = '') {
        $db = Database::getInstance();
        
        $whereConditions = ["v.fecha BETWEEN ? AND ?"];
        $params = [$fechaInicio, $fechaFin];
        
        if (!empty($vendedorId)) {
            $whereConditions[] = "c.vendedor_id = ?";
            $params[] = $vendedorId;
        }
        
        $whereClause = 'WHERE ' . implode(' AND ', $whereConditions);
        
        $datos = $db->select("
            SELECT 
                c.nombre as 'Cliente',
                u.nombre as 'Vendedor',
                COUNT(v.id) as 'Total_Entregas',
                SUM(v.cantidad_bidones) as 'Total_Bidones',
                SUM(v.total) as 'Total_Ventas',
                AVG(v.cantidad_bidones) as 'Promedio_Bidones_por_Entrega',
                MIN(v.fecha) as 'Primera_Venta',
                MAX(v.fecha) as 'Ultima_Venta'
            FROM ventas v
            INNER JOIN clientes c ON v.cliente_id = c.id
            INNER JOIN usuarios u ON c.vendedor_id = u.id
            $whereClause
            GROUP BY c.id, c.nombre, u.nombre
            ORDER BY SUM(v.total) DESC
        ", $params);
        
        self::exportToExcel($datos, 'Reporte de Ventas por Cliente', $fechaInicio, $fechaFin);
    }
    
    /**
     * Exportar reporte de saldos de clientes
     */
    public static function exportSaldosClientes($vendedorId = '') {
        $db = Database::getInstance();
        
        $whereConditions = ["c.activo = 1"];
        $params = [];
        
        if (!empty($vendedorId)) {
            $whereConditions[] = "c.vendedor_id = ?";
            $params[] = $vendedorId;
        }
        
        $whereClause = 'WHERE ' . implode(' AND ', $whereConditions);
        
        $datos = $db->select("
            SELECT 
                c.nombre as 'Cliente',
                c.direccion as 'Direccion',
                u.nombre as 'Vendedor',
                COALESCE(ventas.total_ventas, 0) as 'Total_Ventas',
                COALESCE(pagos.total_pagos, 0) as 'Total_Pagos',
                (COALESCE(ventas.total_ventas, 0) - COALESCE(pagos.total_pagos, 0)) as 'Saldo_Actual',
                COALESCE(ventas.total_bidones, 0) as 'Total_Bidones_Historico'
            FROM clientes c
            INNER JOIN usuarios u ON c.vendedor_id = u.id
            LEFT JOIN (
                SELECT 
                    cliente_id,
                    SUM(total) as total_ventas,
                    SUM(cantidad_bidones) as total_bidones
                FROM ventas 
                GROUP BY cliente_id
            ) ventas ON c.id = ventas.cliente_id
            LEFT JOIN (
                SELECT 
                    cliente_id,
                    SUM(monto) as total_pagos
                FROM pagos 
                GROUP BY cliente_id
            ) pagos ON c.id = pagos.cliente_id
            $whereClause
            ORDER BY (COALESCE(ventas.total_ventas, 0) - COALESCE(pagos.total_pagos, 0)) DESC
        ", $params);
        
        self::exportToExcel($datos, 'Reporte de Saldos de Clientes');
    }
}
?>
