/**
 * JavaScript Principal del Sistema
 * Sistema de Gestión de Agua Mineral
 */

// Configuración global
const App = {
    config: {
        sessionTimeout: 3600000, // 1 hora en milisegundos
        autoSaveInterval: 30000,  // 30 segundos
        animationDuration: 300
    },
    
    // Inicialización
    init: function() {
        this.initTooltips();
        this.initPopovers();
        this.initFormValidation();
        this.initAutoSave();
        this.initKeyboardShortcuts();
        this.initTheme();
        this.initNotifications();
        
        console.log('Sistema de Agua Mineral - Inicializado');
    },
    
    // Inicializar tooltips de Bootstrap
    initTooltips: function() {
        const tooltipTriggerList = [].slice.call(document.querySelectorAll('[data-bs-toggle="tooltip"]'));
        tooltipTriggerList.map(function (tooltipTriggerEl) {
            return new bootstrap.Tooltip(tooltipTriggerEl);
        });
    },
    
    // Inicializar popovers de Bootstrap
    initPopovers: function() {
        const popoverTriggerList = [].slice.call(document.querySelectorAll('[data-bs-toggle="popover"]'));
        popoverTriggerList.map(function (popoverTriggerEl) {
            return new bootstrap.Popover(popoverTriggerEl);
        });
    },
    
    // Validación de formularios
    initFormValidation: function() {
        // Validación HTML5 personalizada
        const forms = document.querySelectorAll('.needs-validation');
        
        Array.prototype.slice.call(forms).forEach(function(form) {
            form.addEventListener('submit', function(event) {
                if (!form.checkValidity()) {
                    event.preventDefault();
                    event.stopPropagation();
                }
                form.classList.add('was-validated');
            }, false);
        });
        
        // Validación en tiempo real
        const inputs = document.querySelectorAll('input, select, textarea');
        inputs.forEach(input => {
            input.addEventListener('blur', function() {
                this.classList.add('was-validated');
            });
        });
    },
    
    // Auto-guardado de formularios
    initAutoSave: function() {
        const forms = document.querySelectorAll('[data-autosave="true"]');
        
        forms.forEach(form => {
            const formId = form.id || 'form_' + Date.now();
            
            // Cargar datos guardados
            this.loadFormData(form, formId);
            
            // Guardar cambios automáticamente
            const inputs = form.querySelectorAll('input, select, textarea');
            inputs.forEach(input => {
                input.addEventListener('input', () => {
                    this.saveFormData(form, formId);
                });
            });
        });
    },
    
    // Atajos de teclado
    initKeyboardShortcuts: function() {
        document.addEventListener('keydown', function(e) {
            // Ctrl/Cmd + S para guardar
            if ((e.ctrlKey || e.metaKey) && e.key === 's') {
                e.preventDefault();
                const submitBtn = document.querySelector('button[type="submit"]');
                if (submitBtn) {
                    submitBtn.click();
                }
            }
            
            // Escape para cerrar modales
            if (e.key === 'Escape') {
                const modals = document.querySelectorAll('.modal.show');
                modals.forEach(modal => {
                    const modalInstance = bootstrap.Modal.getInstance(modal);
                    if (modalInstance) {
                        modalInstance.hide();
                    }
                });
            }
            
            // F1 para ayuda (prevenir comportamiento por defecto)
            if (e.key === 'F1') {
                e.preventDefault();
                App.showHelp();
            }
        });
    },
    
    // Tema y preferencias
    initTheme: function() {
        const savedTheme = localStorage.getItem('theme');
        if (savedTheme) {
            document.documentElement.setAttribute('data-theme', savedTheme);
        }
    },
    
    // Sistema de notificaciones
    initNotifications: function() {
        // Crear contenedor de notificaciones si no existe
        if (!document.getElementById('notifications-container')) {
            const container = document.createElement('div');
            container.id = 'notifications-container';
            container.className = 'position-fixed top-0 end-0 p-3';
            container.style.zIndex = '1055';
            document.body.appendChild(container);
        }
    },
    
    // Utilidades
    utils: {
        // Formatear moneda (Guaraníes paraguayos)
        formatCurrency: function(amount) {
            return new Intl.NumberFormat('es-PY', {
                style: 'currency',
                currency: 'PYG'
            }).format(amount);
        },
        
        // Formatear fecha
        formatDate: function(date, options = {}) {
            const defaultOptions = {
                year: 'numeric',
                month: '2-digit',
                day: '2-digit'
            };
            return new Intl.DateTimeFormat('es-AR', {...defaultOptions, ...options}).format(new Date(date));
        },
        
        // Debounce para optimizar eventos
        debounce: function(func, wait) {
            let timeout;
            return function executedFunction(...args) {
                const later = () => {
                    clearTimeout(timeout);
                    func(...args);
                };
                clearTimeout(timeout);
                timeout = setTimeout(later, wait);
            };
        },
        
        // Validar email
        validateEmail: function(email) {
            const re = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
            return re.test(email);
        },
        
        // Generar ID único
        generateId: function() {
            return Date.now().toString(36) + Math.random().toString(36).substr(2);
        }
    },
    
    // Guardar datos del formulario
    saveFormData: function(form, formId) {
        const formData = new FormData(form);
        const data = {};
        
        for (let [key, value] of formData.entries()) {
            data[key] = value;
        }
        
        localStorage.setItem('form_' + formId, JSON.stringify(data));
    },
    
    // Cargar datos del formulario
    loadFormData: function(form, formId) {
        const savedData = localStorage.getItem('form_' + formId);
        
        if (savedData) {
            try {
                const data = JSON.parse(savedData);
                
                Object.keys(data).forEach(key => {
                    const input = form.querySelector(`[name="${key}"]`);
                    if (input && input.type !== 'password') {
                        input.value = data[key];
                    }
                });
            } catch (e) {
                console.warn('Error al cargar datos del formulario:', e);
            }
        }
    },
    
    // Limpiar datos guardados del formulario
    clearFormData: function(formId) {
        localStorage.removeItem('form_' + formId);
    },
    
    // Mostrar notificación
    showNotification: function(message, type = 'info', duration = 5000) {
        const container = document.getElementById('notifications-container');
        const id = this.utils.generateId();
        
        const alertClass = {
            'success': 'alert-success',
            'error': 'alert-danger',
            'warning': 'alert-warning',
            'info': 'alert-info'
        }[type] || 'alert-info';
        
        const icon = {
            'success': 'fa-check-circle',
            'error': 'fa-exclamation-circle',
            'warning': 'fa-exclamation-triangle',
            'info': 'fa-info-circle'
        }[type] || 'fa-info-circle';
        
        const notification = document.createElement('div');
        notification.id = id;
        notification.className = `alert ${alertClass} alert-dismissible fade show`;
        notification.innerHTML = `
            <i class="fas ${icon} me-2"></i>
            ${message}
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        `;
        
        container.appendChild(notification);
        
        // Auto-ocultar después del tiempo especificado
        if (duration > 0) {
            setTimeout(() => {
                const alert = document.getElementById(id);
                if (alert) {
                    const bsAlert = new bootstrap.Alert(alert);
                    bsAlert.close();
                }
            }, duration);
        }
    },
    
    // Confirmar acción
    confirm: function(message, callback) {
        if (window.confirm(message)) {
            callback();
        }
    },
    
    // Mostrar loading
    showLoading: function(element) {
        if (element) {
            element.classList.add('loading');
            element.disabled = true;
        }
    },
    
    // Ocultar loading
    hideLoading: function(element) {
        if (element) {
            element.classList.remove('loading');
            element.disabled = false;
        }
    },
    
    // Hacer petición AJAX
    ajax: function(url, options = {}) {
        const defaults = {
            method: 'GET',
            headers: {
                'Content-Type': 'application/json',
                'X-Requested-With': 'XMLHttpRequest'
            }
        };
        
        const config = {...defaults, ...options};
        
        return fetch(url, config)
            .then(response => {
                if (!response.ok) {
                    throw new Error(`HTTP error! status: ${response.status}`);
                }
                return response.json();
            })
            .catch(error => {
                console.error('Error en petición AJAX:', error);
                this.showNotification('Error en la comunicación con el servidor', 'error');
                throw error;
            });
    },
    
    // Mostrar ayuda
    showHelp: function() {
        this.showNotification('Ayuda: Use Ctrl+S para guardar, Escape para cerrar modales', 'info');
    },
    
    // Cambiar tema
    toggleTheme: function() {
        const currentTheme = document.documentElement.getAttribute('data-theme');
        const newTheme = currentTheme === 'dark' ? 'light' : 'dark';
        
        document.documentElement.setAttribute('data-theme', newTheme);
        localStorage.setItem('theme', newTheme);
    }
};

// Extensiones específicas del sistema
App.AguaMineral = {
    // Validar coordenadas GPS para Paraguay
    validateCoordinates: function(lat, lng) {
        const latNum = parseFloat(lat);
        const lngNum = parseFloat(lng);
        
        // Rangos aproximados para Paraguay
        return latNum >= -27.6 && latNum <= -19.3 && lngNum >= -62.6 && lngNum <= -54.3;
    },
    
    // Calcular distancia entre dos puntos GPS
    calculateDistance: function(lat1, lng1, lat2, lng2) {
        const R = 6371; // Radio de la Tierra en km
        const dLat = this.toRad(lat2 - lat1);
        const dLng = this.toRad(lng2 - lng1);
        
        const a = Math.sin(dLat/2) * Math.sin(dLat/2) +
                  Math.cos(this.toRad(lat1)) * Math.cos(this.toRad(lat2)) *
                  Math.sin(dLng/2) * Math.sin(dLng/2);
        
        const c = 2 * Math.atan2(Math.sqrt(a), Math.sqrt(1-a));
        return R * c;
    },
    
    toRad: function(value) {
        return value * Math.PI / 180;
    },
    
    // Formatear número de bidones
    formatBidones: function(cantidad) {
        return cantidad + (cantidad === 1 ? ' bidón' : ' bidones');
    }
};

// Inicializar cuando el DOM esté listo
document.addEventListener('DOMContentLoaded', function() {
    App.init();
});

// Manejar errores globales
window.addEventListener('error', function(e) {
    console.error('Error global:', e.error);
    App.showNotification('Ha ocurrido un error inesperado', 'error');
});

// Manejar promesas rechazadas
window.addEventListener('unhandledrejection', function(e) {
    console.error('Promesa rechazada:', e.reason);
    App.showNotification('Error en operación asíncrona', 'error');
});

// Exportar para uso global
window.App = App;
