<?php
/**
 * Sistema Médico - Nueva Consulta
 */
require_once '../includes/config.php';
require_once '../includes/database.php';
require_once '../includes/auth.php';

$page_title = 'Nueva Consulta';

$database = new Database();
$db = $database->getConnection();

$paciente_id = $_GET['paciente_id'] ?? 0;

if (!$paciente_id) {
    header('Location: buscar_paciente.php');
    exit();
}

// Obtener datos del paciente
$query = "SELECT * FROM pacientes WHERE id = :id";
$stmt = $db->prepare($query);
$stmt->bindParam(':id', $paciente_id);
$stmt->execute();
$paciente = $stmt->fetch();

if (!$paciente) {
    header('Location: buscar_paciente.php');
    exit();
}

// Obtener alertas médicas
$query = "SELECT * FROM alertas_medicas WHERE paciente_id = :id AND activo = 1";
$stmt = $db->prepare($query);
$stmt->bindParam(':id', $paciente_id);
$stmt->execute();
$alertas = $stmt->fetchAll();

// Obtener última consulta
$query = "SELECT * FROM consultas WHERE paciente_id = :id ORDER BY fecha_consulta DESC LIMIT 1";
$stmt = $db->prepare($query);
$stmt->bindParam(':id', $paciente_id);
$stmt->execute();
$ultima_consulta = $stmt->fetch();

$mensaje = '';
$tipo_mensaje = '';

// Procesar formulario
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $motivo_visita = trim($_POST['motivo_visita'] ?? '');
    $peso = trim($_POST['peso'] ?? '');
    $presion_arterial = trim($_POST['presion_arterial'] ?? '');
    $masa_muscular = trim($_POST['masa_muscular'] ?? '');
    $diagnostico = trim($_POST['diagnostico'] ?? '');
    $observaciones = trim($_POST['observaciones'] ?? '');
    
    // Tratamientos (array)
    $tratamientos = $_POST['tratamiento'] ?? [];
    $duraciones_trat = $_POST['duracion_tratamiento'] ?? [];
    $indicaciones_trat = $_POST['indicaciones_tratamiento'] ?? [];
    
    // Medicamentos (array)
    $medicamentos = $_POST['medicamento'] ?? [];
    $dosis = $_POST['dosis'] ?? [];
    $frecuencias = $_POST['frecuencia'] ?? [];
    $duraciones_med = $_POST['duracion_medicamento'] ?? [];
    $indicaciones_med = $_POST['indicaciones_medicamento'] ?? [];

    try {
        $db->beginTransaction();

        $current_user = Auth::getCurrentUser();

        // Insertar consulta
        $query = "INSERT INTO consultas (paciente_id, medico_id, motivo_visita, peso, presion_arterial, masa_muscular, diagnostico, observaciones) 
                  VALUES (:paciente_id, :medico_id, :motivo_visita, :peso, :presion_arterial, :masa_muscular, :diagnostico, :observaciones)";
        
        $stmt = $db->prepare($query);
        $stmt->bindParam(':paciente_id', $paciente_id);
        $stmt->bindParam(':medico_id', $current_user['id']);
        $stmt->bindParam(':motivo_visita', $motivo_visita);
        $stmt->bindParam(':peso', $peso);
        $stmt->bindParam(':presion_arterial', $presion_arterial);
        $stmt->bindParam(':masa_muscular', $masa_muscular);
        $stmt->bindParam(':diagnostico', $diagnostico);
        $stmt->bindParam(':observaciones', $observaciones);
        $stmt->execute();

        $consulta_id = $db->lastInsertId();

        // Insertar tratamientos
        foreach ($tratamientos as $index => $tratamiento) {
            if (!empty(trim($tratamiento))) {
                $query = "INSERT INTO tratamientos (consulta_id, descripcion, duracion, indicaciones) 
                          VALUES (:consulta_id, :descripcion, :duracion, :indicaciones)";
                $stmt = $db->prepare($query);
                $stmt->bindParam(':consulta_id', $consulta_id);
                $stmt->bindParam(':descripcion', $tratamiento);
                $stmt->bindParam(':duracion', $duraciones_trat[$index] ?? '');
                $stmt->bindParam(':indicaciones', $indicaciones_trat[$index] ?? '');
                $stmt->execute();
            }
        }

        // Insertar medicamentos
        foreach ($medicamentos as $index => $medicamento) {
            if (!empty(trim($medicamento))) {
                $query = "INSERT INTO medicamentos_recetados (consulta_id, nombre_medicamento, dosis, frecuencia, duracion, indicaciones) 
                          VALUES (:consulta_id, :nombre_medicamento, :dosis, :frecuencia, :duracion, :indicaciones)";
                $stmt = $db->prepare($query);
                $stmt->bindParam(':consulta_id', $consulta_id);
                $stmt->bindParam(':nombre_medicamento', $medicamento);
                $stmt->bindParam(':dosis', $dosis[$index] ?? '');
                $stmt->bindParam(':frecuencia', $frecuencias[$index] ?? '');
                $stmt->bindParam(':duracion', $duraciones_med[$index] ?? '');
                $stmt->bindParam(':indicaciones', $indicaciones_med[$index] ?? '');
                $stmt->execute();
            }
        }

        $db->commit();

        $mensaje = 'Consulta registrada exitosamente';
        $tipo_mensaje = 'success';

        // Redirigir a la ficha del paciente después de 2 segundos
        header("refresh:2;url=ficha_paciente.php?id=$paciente_id");

    } catch (Exception $e) {
        $db->rollBack();
        $mensaje = 'Error al registrar consulta: ' . $e->getMessage();
        $tipo_mensaje = 'error';
    }
}

include '../includes/header.php';
?>

<!-- Alertas de emergencia -->
<?php if (count($alertas) > 0): ?>
    <div class="alert-emergency">
        <strong>⚠️ ALERTAS MÉDICAS IMPORTANTES ⚠️</strong>
        <ul style="margin: 10px 0 0 20px;">
            <?php foreach ($alertas as $alerta): ?>
                <li>
                    <strong><?php echo strtoupper(str_replace('_', ' ', $alerta['tipo'])); ?>:</strong>
                    <?php echo htmlspecialchars($alerta['descripcion']); ?>
                </li>
            <?php endforeach; ?>
        </ul>
    </div>
<?php endif; ?>

<!-- Información del paciente -->
<div class="patient-info">
    <h3>Paciente: <?php echo htmlspecialchars($paciente['nombre'] . ' ' . $paciente['apellido']); ?></h3>
    <div class="patient-details">
        <div class="detail-item">
            <span class="detail-label">Cédula</span>
            <span class="detail-value"><?php echo htmlspecialchars($paciente['cedula']); ?></span>
        </div>
        <div class="detail-item">
            <span class="detail-label">Edad</span>
            <span class="detail-value">
                <?php
                $fecha_nac = new DateTime($paciente['fecha_nacimiento']);
                $hoy = new DateTime();
                echo $hoy->diff($fecha_nac)->y . ' años';
                ?>
            </span>
        </div>
        <div class="detail-item">
            <span class="detail-label">Teléfono</span>
            <span class="detail-value"><?php echo htmlspecialchars($paciente['telefono']); ?></span>
        </div>
    </div>
</div>

<!-- Datos de la última visita -->
<?php if ($ultima_consulta): ?>
    <div class="last-visit-data">
        <h4>📊 Datos de la Última Visita (<?php echo date('d/m/Y', strtotime($ultima_consulta['fecha_consulta'])); ?>)</h4>
        <div class="vital-signs">
            <div class="vital-sign">
                <span class="vital-sign-label">Peso</span>
                <span class="vital-sign-value"><?php echo $ultima_consulta['peso'] ? $ultima_consulta['peso'] . ' kg' : 'N/D'; ?></span>
            </div>
            <div class="vital-sign">
                <span class="vital-sign-label">Presión Arterial</span>
                <span class="vital-sign-value"><?php echo $ultima_consulta['presion_arterial'] ?: 'N/D'; ?></span>
            </div>
            <div class="vital-sign">
                <span class="vital-sign-label">Masa Muscular</span>
                <span class="vital-sign-value"><?php echo $ultima_consulta['masa_muscular'] ? $ultima_consulta['masa_muscular'] . ' %' : 'N/D'; ?></span>
            </div>
        </div>
    </div>
<?php endif; ?>

<!-- Formulario de nueva consulta -->
<div class="card">
    <div class="card-header">
        <h2>Registrar Nueva Consulta</h2>
    </div>
    <div class="card-body">
        <?php if ($mensaje): ?>
            <div class="alert alert-<?php echo $tipo_mensaje; ?>">
                <?php echo htmlspecialchars($mensaje); ?>
            </div>
        <?php endif; ?>

        <form method="POST" action="" id="formNuevaConsulta">
            <h3 style="color: var(--medical-blue); margin-bottom: 20px; border-bottom: 2px solid var(--medical-light-blue); padding-bottom: 10px;">
                Datos de la Consulta
            </h3>

            <div class="form-group">
                <label for="motivo_visita">Motivo de la Visita *</label>
                <textarea id="motivo_visita" name="motivo_visita" rows="3" required></textarea>
            </div>

            <h3 style="color: var(--medical-blue); margin: 30px 0 20px; border-bottom: 2px solid var(--medical-light-blue); padding-bottom: 10px;">
                Signos Vitales
            </h3>

            <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(250px, 1fr)); gap: 20px;">
                <div class="form-group">
                    <label for="peso">Peso (kg)</label>
                    <input type="number" id="peso" name="peso" step="0.01" placeholder="Ej: 70.5">
                </div>

                <div class="form-group">
                    <label for="presion_arterial">Presión Arterial</label>
                    <input type="text" id="presion_arterial" name="presion_arterial" placeholder="Ej: 120/80">
                </div>

                <div class="form-group">
                    <label for="masa_muscular">Masa Muscular (%)</label>
                    <input type="number" id="masa_muscular" name="masa_muscular" step="0.01" placeholder="Ej: 35.5">
                </div>
            </div>

            <h3 style="color: var(--medical-blue); margin: 30px 0 20px; border-bottom: 2px solid var(--medical-light-blue); padding-bottom: 10px;">
                Diagnóstico y Observaciones
            </h3>

            <div class="form-group">
                <label for="diagnostico">Diagnóstico</label>
                <textarea id="diagnostico" name="diagnostico" rows="4"></textarea>
            </div>

            <div class="form-group">
                <label for="observaciones">Observaciones</label>
                <textarea id="observaciones" name="observaciones" rows="3"></textarea>
            </div>

            <h3 style="color: var(--medical-blue); margin: 30px 0 20px; border-bottom: 2px solid var(--medical-light-blue); padding-bottom: 10px;">
                Tratamiento Médico
            </h3>

            <div id="tratamientos-container">
                <div class="tratamiento-item" style="background: var(--light-color); padding: 15px; border-radius: 8px; margin-bottom: 15px;">
                    <div class="form-group">
                        <label>Descripción del Tratamiento</label>
                        <textarea name="tratamiento[]" rows="2"></textarea>
                    </div>
                    <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 15px;">
                        <div class="form-group">
                            <label>Duración</label>
                            <input type="text" name="duracion_tratamiento[]" placeholder="Ej: 2 semanas">
                        </div>
                        <div class="form-group">
                            <label>Indicaciones</label>
                            <input type="text" name="indicaciones_tratamiento[]" placeholder="Ej: Reposo absoluto">
                        </div>
                    </div>
                </div>
            </div>
            <button type="button" class="btn btn-secondary" onclick="agregarTratamiento()">+ Agregar Otro Tratamiento</button>

            <h3 style="color: var(--medical-blue); margin: 30px 0 20px; border-bottom: 2px solid var(--medical-light-blue); padding-bottom: 10px;">
                Medicamentos Recetados
            </h3>

            <div id="medicamentos-container">
                <div class="medicamento-item" style="background: var(--light-color); padding: 15px; border-radius: 8px; margin-bottom: 15px;">
                    <div class="form-group">
                        <label>Nombre del Medicamento</label>
                        <input type="text" name="medicamento[]" placeholder="Ej: Paracetamol">
                    </div>
                    <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 15px;">
                        <div class="form-group">
                            <label>Dosis</label>
                            <input type="text" name="dosis[]" placeholder="Ej: 500 mg">
                        </div>
                        <div class="form-group">
                            <label>Frecuencia</label>
                            <input type="text" name="frecuencia[]" placeholder="Ej: Cada 8 horas">
                        </div>
                        <div class="form-group">
                            <label>Duración</label>
                            <input type="text" name="duracion_medicamento[]" placeholder="Ej: 7 días">
                        </div>
                    </div>
                    <div class="form-group">
                        <label>Indicaciones</label>
                        <input type="text" name="indicaciones_medicamento[]" placeholder="Ej: Tomar con alimentos">
                    </div>
                </div>
            </div>
            <button type="button" class="btn btn-secondary" onclick="agregarMedicamento()">+ Agregar Otro Medicamento</button>

            <div style="display: flex; gap: 15px; margin-top: 30px;">
                <button type="submit" class="btn btn-success">💾 Registrar Consulta</button>
                <a href="ficha_paciente.php?id=<?php echo $paciente_id; ?>" class="btn btn-secondary">Cancelar</a>
            </div>
        </form>
    </div>
</div>

<script>
function agregarTratamiento() {
    const container = document.getElementById('tratamientos-container');
    const nuevoTratamiento = document.createElement('div');
    nuevoTratamiento.className = 'tratamiento-item';
    nuevoTratamiento.style.cssText = 'background: var(--light-color); padding: 15px; border-radius: 8px; margin-bottom: 15px;';
    nuevoTratamiento.innerHTML = `
        <div class="form-group">
            <label>Descripción del Tratamiento</label>
            <textarea name="tratamiento[]" rows="2"></textarea>
        </div>
        <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 15px;">
            <div class="form-group">
                <label>Duración</label>
                <input type="text" name="duracion_tratamiento[]" placeholder="Ej: 2 semanas">
            </div>
            <div class="form-group">
                <label>Indicaciones</label>
                <input type="text" name="indicaciones_tratamiento[]" placeholder="Ej: Reposo absoluto">
            </div>
        </div>
    `;
    container.appendChild(nuevoTratamiento);
}

function agregarMedicamento() {
    const container = document.getElementById('medicamentos-container');
    const nuevoMedicamento = document.createElement('div');
    nuevoMedicamento.className = 'medicamento-item';
    nuevoMedicamento.style.cssText = 'background: var(--light-color); padding: 15px; border-radius: 8px; margin-bottom: 15px;';
    nuevoMedicamento.innerHTML = `
        <div class="form-group">
            <label>Nombre del Medicamento</label>
            <input type="text" name="medicamento[]" placeholder="Ej: Paracetamol">
        </div>
        <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 15px;">
            <div class="form-group">
                <label>Dosis</label>
                <input type="text" name="dosis[]" placeholder="Ej: 500 mg">
            </div>
            <div class="form-group">
                <label>Frecuencia</label>
                <input type="text" name="frecuencia[]" placeholder="Ej: Cada 8 horas">
            </div>
            <div class="form-group">
                <label>Duración</label>
                <input type="text" name="duracion_medicamento[]" placeholder="Ej: 7 días">
            </div>
        </div>
        <div class="form-group">
            <label>Indicaciones</label>
            <input type="text" name="indicaciones_medicamento[]" placeholder="Ej: Tomar con alimentos">
        </div>
    `;
    container.appendChild(nuevoMedicamento);
}
</script>

<?php include '../includes/footer.php'; ?>

