<?php
/**
 * Sistema Médico - Ficha del Paciente
 */
require_once '../includes/config.php';
require_once '../includes/database.php';
require_once '../includes/auth.php';

$page_title = 'Ficha del Paciente';

$database = new Database();
$db = $database->getConnection();

$paciente_id = $_GET['id'] ?? 0;

if (!$paciente_id) {
    header('Location: buscar_paciente.php');
    exit();
}

// Obtener datos del paciente
$query = "SELECT * FROM pacientes WHERE id = :id";
$stmt = $db->prepare($query);
$stmt->bindParam(':id', $paciente_id);
$stmt->execute();
$paciente = $stmt->fetch();

if (!$paciente) {
    header('Location: buscar_paciente.php');
    exit();
}

// Calcular edad
$fecha_nac = new DateTime($paciente['fecha_nacimiento']);
$hoy = new DateTime();
$edad = $hoy->diff($fecha_nac)->y;

// Obtener alertas médicas
$query = "SELECT * FROM alertas_medicas WHERE paciente_id = :id AND activo = 1 ORDER BY severidad DESC";
$stmt = $db->prepare($query);
$stmt->bindParam(':id', $paciente_id);
$stmt->execute();
$alertas = $stmt->fetchAll();

// Obtener última consulta
$query = "SELECT * FROM consultas WHERE paciente_id = :id ORDER BY fecha_consulta DESC LIMIT 1";
$stmt = $db->prepare($query);
$stmt->bindParam(':id', $paciente_id);
$stmt->execute();
$ultima_consulta = $stmt->fetch();

// Obtener historial de consultas
$query = "SELECT c.*, u.nombre_completo as medico
          FROM consultas c
          INNER JOIN usuarios u ON c.medico_id = u.id
          WHERE c.paciente_id = :id
          ORDER BY c.fecha_consulta DESC";
$stmt = $db->prepare($query);
$stmt->bindParam(':id', $paciente_id);
$stmt->execute();
$historial_consultas = $stmt->fetchAll();

include '../includes/header.php';
?>

<!-- Alertas de emergencia -->
<?php if (count($alertas) > 0): ?>
    <div class="alert-emergency">
        <strong>⚠️ ALERTAS MÉDICAS IMPORTANTES ⚠️</strong>
        <ul style="margin: 10px 0 0 20px;">
            <?php foreach ($alertas as $alerta): ?>
                <li>
                    <strong><?php echo strtoupper(str_replace('_', ' ', $alerta['tipo'])); ?>:</strong>
                    <?php echo htmlspecialchars($alerta['descripcion']); ?>
                </li>
            <?php endforeach; ?>
        </ul>
    </div>
<?php endif; ?>

<!-- Información del paciente -->
<div class="patient-info">
    <h3>Información del Paciente</h3>
    <div class="patient-details">
        <div class="detail-item">
            <span class="detail-label">Nombre Completo</span>
            <span class="detail-value"><?php echo htmlspecialchars($paciente['nombre'] . ' ' . $paciente['apellido']); ?></span>
        </div>
        <div class="detail-item">
            <span class="detail-label">Cédula</span>
            <span class="detail-value"><?php echo htmlspecialchars($paciente['cedula']); ?></span>
        </div>
        <div class="detail-item">
            <span class="detail-label">Fecha de Nacimiento</span>
            <span class="detail-value"><?php echo date('d/m/Y', strtotime($paciente['fecha_nacimiento'])); ?> (<?php echo $edad; ?> años)</span>
        </div>
        <div class="detail-item">
            <span class="detail-label">Teléfono</span>
            <span class="detail-value"><?php echo htmlspecialchars($paciente['telefono']); ?></span>
        </div>
        <div class="detail-item">
            <span class="detail-label">Domicilio</span>
            <span class="detail-value"><?php echo htmlspecialchars($paciente['domicilio']); ?></span>
        </div>
        <div class="detail-item">
            <span class="detail-label">Fecha de Registro</span>
            <span class="detail-value"><?php echo date('d/m/Y', strtotime($paciente['fecha_registro'])); ?></span>
        </div>
    </div>
</div>

<!-- Datos de la última visita -->
<?php if ($ultima_consulta): ?>
    <div class="last-visit-data">
        <h4>📊 Datos de la Última Visita (<?php echo date('d/m/Y', strtotime($ultima_consulta['fecha_consulta'])); ?>)</h4>
        <div class="vital-signs">
            <div class="vital-sign">
                <span class="vital-sign-label">Peso</span>
                <span class="vital-sign-value"><?php echo $ultima_consulta['peso'] ? $ultima_consulta['peso'] . ' kg' : 'N/D'; ?></span>
            </div>
            <div class="vital-sign">
                <span class="vital-sign-label">Presión Arterial</span>
                <span class="vital-sign-value"><?php echo $ultima_consulta['presion_arterial'] ?: 'N/D'; ?></span>
            </div>
            <div class="vital-sign">
                <span class="vital-sign-label">Masa Muscular</span>
                <span class="vital-sign-value"><?php echo $ultima_consulta['masa_muscular'] ? $ultima_consulta['masa_muscular'] . ' %' : 'N/D'; ?></span>
            </div>
        </div>
    </div>
<?php endif; ?>

<!-- Acciones rápidas -->
<div class="card">
    <div class="card-body">
        <div style="display: flex; gap: 15px; flex-wrap: wrap;">
            <a href="nueva_consulta.php?paciente_id=<?php echo $paciente_id; ?>" class="btn btn-success">
                ➕ Nueva Consulta
            </a>
            <a href="agendar_cita.php?paciente_id=<?php echo $paciente_id; ?>" class="btn btn-primary">
                📅 Agendar Cita
            </a>
            <a href="editar_paciente.php?id=<?php echo $paciente_id; ?>" class="btn btn-secondary">
                ✏️ Editar Datos
            </a>
        </div>
    </div>
</div>

<!-- Historial de consultas -->
<div class="card">
    <div class="card-header">
        <h2>Historial de Consultas</h2>
    </div>
    <div class="card-body">
        <?php if (count($historial_consultas) > 0): ?>
            <?php foreach ($historial_consultas as $consulta): ?>
                <div class="history-item">
                    <div class="history-date">
                        📅 <?php echo date('d/m/Y H:i', strtotime($consulta['fecha_consulta'])); ?> - 
                        Dr(a). <?php echo htmlspecialchars($consulta['medico']); ?>
                    </div>
                    <div class="history-content">
                        <p><strong>Motivo de la visita:</strong> <?php echo htmlspecialchars($consulta['motivo_visita']); ?></p>
                        
                        <p>
                            <strong>Signos vitales:</strong> 
                            Peso: <?php echo $consulta['peso'] ? $consulta['peso'] . ' kg' : 'N/D'; ?> | 
                            Presión: <?php echo $consulta['presion_arterial'] ?: 'N/D'; ?> | 
                            Masa muscular: <?php echo $consulta['masa_muscular'] ? $consulta['masa_muscular'] . ' %' : 'N/D'; ?>
                        </p>

                        <?php if ($consulta['diagnostico']): ?>
                            <p><strong>Diagnóstico:</strong> <?php echo nl2br(htmlspecialchars($consulta['diagnostico'])); ?></p>
                        <?php endif; ?>

                        <?php if ($consulta['observaciones']): ?>
                            <p><strong>Observaciones:</strong> <?php echo nl2br(htmlspecialchars($consulta['observaciones'])); ?></p>
                        <?php endif; ?>

                        <?php
                        // Obtener tratamientos de esta consulta
                        $query_trat = "SELECT * FROM tratamientos WHERE consulta_id = :consulta_id";
                        $stmt_trat = $db->prepare($query_trat);
                        $stmt_trat->bindParam(':consulta_id', $consulta['id']);
                        $stmt_trat->execute();
                        $tratamientos = $stmt_trat->fetchAll();

                        if (count($tratamientos) > 0):
                        ?>
                            <p><strong>Tratamiento:</strong></p>
                            <ul style="margin-left: 20px;">
                                <?php foreach ($tratamientos as $trat): ?>
                                    <li><?php echo htmlspecialchars($trat['descripcion']); ?>
                                        <?php if ($trat['duracion']): ?>
                                            (Duración: <?php echo htmlspecialchars($trat['duracion']); ?>)
                                        <?php endif; ?>
                                    </li>
                                <?php endforeach; ?>
                            </ul>
                        <?php endif; ?>

                        <?php
                        // Obtener medicamentos de esta consulta
                        $query_med = "SELECT * FROM medicamentos_recetados WHERE consulta_id = :consulta_id";
                        $stmt_med = $db->prepare($query_med);
                        $stmt_med->bindParam(':consulta_id', $consulta['id']);
                        $stmt_med->execute();
                        $medicamentos = $stmt_med->fetchAll();

                        if (count($medicamentos) > 0):
                        ?>
                            <p><strong>Medicamentos recetados:</strong></p>
                            <ul style="margin-left: 20px;">
                                <?php foreach ($medicamentos as $med): ?>
                                    <li>
                                        <strong><?php echo htmlspecialchars($med['nombre_medicamento']); ?></strong> - 
                                        <?php echo htmlspecialchars($med['dosis']); ?>, 
                                        <?php echo htmlspecialchars($med['frecuencia']); ?>
                                        <?php if ($med['duracion']): ?>
                                            por <?php echo htmlspecialchars($med['duracion']); ?>
                                        <?php endif; ?>
                                    </li>
                                <?php endforeach; ?>
                            </ul>
                        <?php endif; ?>
                    </div>
                </div>
            <?php endforeach; ?>
        <?php else: ?>
            <p>No hay consultas registradas para este paciente.</p>
        <?php endif; ?>
    </div>
</div>

<?php include '../includes/footer.php'; ?>

