<?php
/**
 * Sistema Médico - Reportes y Recaudación (Solo Administradores)
 */
require_once '../../includes/config.php';
require_once '../../includes/database.php';
require_once '../../includes/auth.php';
require_once '../../includes/permissions.php';

Permissions::requirePermission('ViewReports');

$page_title = 'Reportes y Recaudación';

$database = new Database();
$db = $database->getConnection();

// Obtener parámetros de fecha
$fecha_desde = $_GET['fecha_desde'] ?? date('Y-m-01'); // Primer día del mes
$fecha_hasta = $_GET['fecha_hasta'] ?? date('Y-m-d'); // Hoy

// Reporte de recaudación
$query = "SELECT 
          COUNT(*) as total_facturas,
          SUM(CASE WHEN estado = 'pagada' THEN 1 ELSE 0 END) as facturas_pagadas,
          SUM(CASE WHEN estado = 'pendiente' THEN 1 ELSE 0 END) as facturas_pendientes,
          SUM(CASE WHEN estado = 'anulada' THEN 1 ELSE 0 END) as facturas_anuladas,
          SUM(CASE WHEN estado = 'pagada' THEN total ELSE 0 END) as total_recaudado,
          SUM(CASE WHEN estado = 'pendiente' THEN total ELSE 0 END) as total_pendiente
          FROM facturas
          WHERE DATE(fecha_emision) BETWEEN :fecha_desde AND :fecha_hasta";

$stmt = $db->prepare($query);
$stmt->bindParam(':fecha_desde', $fecha_desde);
$stmt->bindParam(':fecha_hasta', $fecha_hasta);
$stmt->execute();
$resumen = $stmt->fetch();

// Recaudación diaria
$query = "SELECT 
          DATE(fecha_emision) as fecha,
          COUNT(*) as total_facturas,
          SUM(CASE WHEN estado = 'pagada' THEN total ELSE 0 END) as total_recaudado
          FROM facturas
          WHERE DATE(fecha_emision) BETWEEN :fecha_desde AND :fecha_hasta
          GROUP BY DATE(fecha_emision)
          ORDER BY fecha DESC";

$stmt = $db->prepare($query);
$stmt->bindParam(':fecha_desde', $fecha_desde);
$stmt->bindParam(':fecha_hasta', $fecha_hasta);
$stmt->execute();
$recaudacion_diaria = $stmt->fetchAll();

// Productos más vendidos
$query = "SELECT 
          p.nombre,
          p.codigo,
          SUM(fp.cantidad) as cantidad_vendida,
          SUM(fp.subtotal) as total_ingresos
          FROM factura_productos fp
          INNER JOIN productos p ON fp.producto_id = p.id
          INNER JOIN facturas f ON fp.factura_id = f.id
          WHERE f.estado = 'pagada' AND DATE(f.fecha_emision) BETWEEN :fecha_desde AND :fecha_hasta
          GROUP BY fp.producto_id, p.nombre, p.codigo
          ORDER BY cantidad_vendida DESC
          LIMIT 10";

$stmt = $db->prepare($query);
$stmt->bindParam(':fecha_desde', $fecha_desde);
$stmt->bindParam(':fecha_hasta', $fecha_hasta);
$stmt->execute();
$productos_mas_vendidos = $stmt->fetchAll();

// Servicios más solicitados
$query = "SELECT 
          s.nombre,
          s.codigo,
          SUM(fs.cantidad) as cantidad_solicitada,
          SUM(fs.subtotal) as total_ingresos
          FROM factura_servicios fs
          INNER JOIN servicios s ON fs.servicio_id = s.id
          INNER JOIN facturas f ON fs.factura_id = f.id
          WHERE f.estado = 'pagada' AND DATE(f.fecha_emision) BETWEEN :fecha_desde AND :fecha_hasta
          GROUP BY fs.servicio_id, s.nombre, s.codigo
          ORDER BY cantidad_solicitada DESC
          LIMIT 10";

$stmt = $db->prepare($query);
$stmt->bindParam(':fecha_desde', $fecha_desde);
$stmt->bindParam(':fecha_hasta', $fecha_hasta);
$stmt->execute();
$servicios_mas_solicitados = $stmt->fetchAll();

// Recaudación por método de pago
$query = "SELECT 
          metodo_pago,
          COUNT(*) as cantidad,
          SUM(total) as total_recaudado
          FROM facturas
          WHERE estado = 'pagada' AND DATE(fecha_emision) BETWEEN :fecha_desde AND :fecha_hasta
          GROUP BY metodo_pago
          ORDER BY total_recaudado DESC";

$stmt = $db->prepare($query);
$stmt->bindParam(':fecha_desde', $fecha_desde);
$stmt->bindParam(':fecha_hasta', $fecha_hasta);
$stmt->execute();
$por_metodo_pago = $stmt->fetchAll();

// Productos con stock bajo
$query = "SELECT * FROM productos 
          WHERE activo = 1 AND stock <= stock_minimo
          ORDER BY stock ASC";
$stmt = $db->query($query);
$productos_stock_bajo = $stmt->fetchAll();

include '../../includes/header.php';
?>

<div class="card">
    <div class="card-header">
        <h2>📊 Reportes y Recaudación</h2>
    </div>
    <div class="card-body">
        <!-- Filtro de fechas -->
        <form method="GET" style="background: var(--light-color); padding: 20px; border-radius: 8px; margin-bottom: 30px;">
            <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 15px; align-items: end;">
                <div class="form-group" style="margin: 0;">
                    <label>Fecha Desde</label>
                    <input type="date" name="fecha_desde" value="<?php echo $fecha_desde; ?>" required>
                </div>
                <div class="form-group" style="margin: 0;">
                    <label>Fecha Hasta</label>
                    <input type="date" name="fecha_hasta" value="<?php echo $fecha_hasta; ?>" required>
                </div>
                <div>
                    <button type="submit" class="btn btn-primary">🔍 Filtrar</button>
                    <a href="exportar_reportes.php?fecha_desde=<?php echo $fecha_desde; ?>&fecha_hasta=<?php echo $fecha_hasta; ?>&formato=excel" 
                       class="btn btn-success">📊 Excel</a>
                    <a href="exportar_reportes.php?fecha_desde=<?php echo $fecha_desde; ?>&fecha_hasta=<?php echo $fecha_hasta; ?>&formato=pdf" 
                       class="btn btn-danger">📄 PDF</a>
                </div>
            </div>
        </form>

        <!-- Resumen general -->
        <h3 style="color: var(--medical-blue); margin-bottom: 20px;">Resumen del Período</h3>
        <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 20px; margin-bottom: 40px;">
            <div style="background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; padding: 25px; border-radius: 10px; text-align: center;">
                <div style="font-size: 14px; opacity: 0.9; margin-bottom: 10px;">Total Facturas</div>
                <div style="font-size: 36px; font-weight: bold;"><?php echo number_format($resumen['total_facturas'], 0, ',', '.'); ?></div>
            </div>
            
            <div style="background: linear-gradient(135deg, #f093fb 0%, #f5576c 100%); color: white; padding: 25px; border-radius: 10px; text-align: center;">
                <div style="font-size: 14px; opacity: 0.9; margin-bottom: 10px;">Facturas Pagadas</div>
                <div style="font-size: 36px; font-weight: bold;"><?php echo number_format($resumen['facturas_pagadas'], 0, ',', '.'); ?></div>
            </div>
            
            <div style="background: linear-gradient(135deg, #4facfe 0%, #00f2fe 100%); color: white; padding: 25px; border-radius: 10px; text-align: center;">
                <div style="font-size: 14px; opacity: 0.9; margin-bottom: 10px;">Facturas Pendientes</div>
                <div style="font-size: 36px; font-weight: bold;"><?php echo number_format($resumen['facturas_pendientes'], 0, ',', '.'); ?></div>
            </div>
            
            <div style="background: linear-gradient(135deg, #43e97b 0%, #38f9d7 100%); color: white; padding: 25px; border-radius: 10px; text-align: center;">
                <div style="font-size: 14px; opacity: 0.9; margin-bottom: 10px;">Total Recaudado</div>
                <div style="font-size: 28px; font-weight: bold;">₲ <?php echo number_format($resumen['total_recaudado'], 0, ',', '.'); ?></div>
            </div>
            
            <div style="background: linear-gradient(135deg, #fa709a 0%, #fee140 100%); color: white; padding: 25px; border-radius: 10px; text-align: center;">
                <div style="font-size: 14px; opacity: 0.9; margin-bottom: 10px;">Total Pendiente</div>
                <div style="font-size: 28px; font-weight: bold;">₲ <?php echo number_format($resumen['total_pendiente'], 0, ',', '.'); ?></div>
            </div>
            
            <div style="background: linear-gradient(135deg, #30cfd0 0%, #330867 100%); color: white; padding: 25px; border-radius: 10px; text-align: center;">
                <div style="font-size: 14px; opacity: 0.9; margin-bottom: 10px;">Facturas Anuladas</div>
                <div style="font-size: 36px; font-weight: bold;"><?php echo number_format($resumen['facturas_anuladas'], 0, ',', '.'); ?></div>
            </div>
        </div>

        <!-- Recaudación diaria -->
        <h3 style="color: var(--medical-blue); margin-top: 40px; margin-bottom: 20px;">Recaudación Diaria</h3>
        <div class="table-responsive">
            <table>
                <thead>
                    <tr>
                        <th>Fecha</th>
                        <th>Facturas</th>
                        <th>Total Recaudado</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($recaudacion_diaria as $dia): ?>
                        <tr>
                            <td><?php echo date('d/m/Y', strtotime($dia['fecha'])); ?></td>
                            <td><?php echo $dia['total_facturas']; ?></td>
                            <td><strong>₲ <?php echo number_format($dia['total_recaudado'], 0, ',', '.'); ?></strong></td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>

        <!-- Productos más vendidos -->
        <h3 style="color: var(--medical-blue); margin-top: 40px; margin-bottom: 20px;">Top 10 Productos Más Vendidos</h3>
        <div class="table-responsive">
            <table>
                <thead>
                    <tr>
                        <th>Código</th>
                        <th>Producto</th>
                        <th>Cantidad Vendida</th>
                        <th>Total Ingresos</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($productos_mas_vendidos as $producto): ?>
                        <tr>
                            <td><?php echo htmlspecialchars($producto['codigo']); ?></td>
                            <td><?php echo htmlspecialchars($producto['nombre']); ?></td>
                            <td><?php echo $producto['cantidad_vendida']; ?></td>
                            <td><strong>₲ <?php echo number_format($producto['total_ingresos'], 0, ',', '.'); ?></strong></td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>

        <!-- Servicios más solicitados -->
        <h3 style="color: var(--medical-blue); margin-top: 40px; margin-bottom: 20px;">Top 10 Servicios Más Solicitados</h3>
        <div class="table-responsive">
            <table>
                <thead>
                    <tr>
                        <th>Código</th>
                        <th>Servicio</th>
                        <th>Cantidad Solicitada</th>
                        <th>Total Ingresos</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($servicios_mas_solicitados as $servicio): ?>
                        <tr>
                            <td><?php echo htmlspecialchars($servicio['codigo']); ?></td>
                            <td><?php echo htmlspecialchars($servicio['nombre']); ?></td>
                            <td><?php echo $servicio['cantidad_solicitada']; ?></td>
                            <td><strong>₲ <?php echo number_format($servicio['total_ingresos'], 0, ',', '.'); ?></strong></td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>

        <!-- Recaudación por método de pago -->
        <h3 style="color: var(--medical-blue); margin-top: 40px; margin-bottom: 20px;">Recaudación por Método de Pago</h3>
        <div class="table-responsive">
            <table>
                <thead>
                    <tr>
                        <th>Método de Pago</th>
                        <th>Cantidad de Facturas</th>
                        <th>Total Recaudado</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($por_metodo_pago as $metodo): ?>
                        <tr>
                            <td><?php echo ucfirst($metodo['metodo_pago']); ?></td>
                            <td><?php echo $metodo['cantidad']; ?></td>
                            <td><strong>₲ <?php echo number_format($metodo['total_recaudado'], 0, ',', '.'); ?></strong></td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>

        <!-- Productos con stock bajo -->
        <?php if (count($productos_stock_bajo) > 0): ?>
            <h3 style="color: var(--warning-color); margin-top: 40px; margin-bottom: 20px;">⚠️ Productos con Stock Bajo</h3>
            <div class="table-responsive">
                <table>
                    <thead>
                        <tr>
                            <th>Código</th>
                            <th>Producto</th>
                            <th>Stock Actual</th>
                            <th>Stock Mínimo</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($productos_stock_bajo as $producto): ?>
                            <tr style="background: #fff3cd;">
                                <td><?php echo htmlspecialchars($producto['codigo']); ?></td>
                                <td><?php echo htmlspecialchars($producto['nombre']); ?></td>
                                <td style="color: red; font-weight: bold;"><?php echo $producto['stock']; ?></td>
                                <td><?php echo $producto['stock_minimo']; ?></td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        <?php endif; ?>
    </div>
</div>

<?php include '../../includes/footer.php'; ?>
