#!/bin/bash

# Script de instalación para Sistema de Rendición de Cuentas ONG
# Autor: Sistema de Rendición de Cuentas
# Versión: 1.0

echo "=========================================="
echo "  Sistema de Rendición de Cuentas ONG"
echo "  Script de Instalación"
echo "=========================================="
echo ""

# Colores para mensajes
GREEN='\033[0;32m'
RED='\033[0;31m'
YELLOW='\033[1;33m'
NC='\033[0m' # No Color

# Función para mostrar mensajes
print_success() {
    echo -e "${GREEN}✓ $1${NC}"
}

print_error() {
    echo -e "${RED}✗ $1${NC}"
}

print_warning() {
    echo -e "${YELLOW}⚠ $1${NC}"
}

# Verificar si se ejecuta como root
if [ "$EUID" -ne 0 ]; then 
    print_error "Por favor ejecute este script como root (sudo ./install.sh)"
    exit 1
fi

# Configuración
DB_USER="rendicionong"
DB_PASS="ongpgy_2026"
DB_NAME="rendicionpy"
WEB_DIR="/var/www/html/rendicion_ong"

echo "Configuración:"
echo "  - Usuario DB: $DB_USER"
echo "  - Base de datos: $DB_NAME"
echo "  - Directorio web: $WEB_DIR"
echo ""

read -p "¿Desea continuar con la instalación? (s/n): " -n 1 -r
echo ""
if [[ ! $REPLY =~ ^[Ss]$ ]]; then
    print_warning "Instalación cancelada"
    exit 0
fi

# 1. Actualizar sistema
echo ""
echo "[1/8] Actualizando sistema..."
apt-get update -qq
print_success "Sistema actualizado"

# 2. Instalar dependencias
echo ""
echo "[2/8] Instalando dependencias..."
apt-get install -y -qq apache2 mysql-server php php-mysql php-gd php-mbstring php-xml unzip > /dev/null 2>&1
print_success "Dependencias instaladas"

# 3. Configurar MySQL
echo ""
echo "[3/8] Configurando base de datos..."

# Verificar si MySQL está corriendo
systemctl start mysql

# Crear base de datos y usuario
mysql -e "CREATE DATABASE IF NOT EXISTS $DB_NAME CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;" 2>/dev/null
mysql -e "CREATE USER IF NOT EXISTS '$DB_USER'@'localhost' IDENTIFIED BY '$DB_PASS';" 2>/dev/null
mysql -e "GRANT ALL PRIVILEGES ON $DB_NAME.* TO '$DB_USER'@'localhost';" 2>/dev/null
mysql -e "FLUSH PRIVILEGES;" 2>/dev/null

# Importar estructura
if [ -f "database.sql" ]; then
    mysql -u $DB_USER -p$DB_PASS $DB_NAME < database.sql 2>/dev/null
    print_success "Base de datos configurada e importada"
else
    print_warning "Archivo database.sql no encontrado, deberá importarlo manualmente"
fi

# 4. Copiar archivos
echo ""
echo "[4/8] Copiando archivos al directorio web..."
mkdir -p $WEB_DIR
cp -r * $WEB_DIR/ 2>/dev/null
print_success "Archivos copiados"

# 5. Configurar permisos
echo ""
echo "[5/8] Configurando permisos..."
chown -R www-data:www-data $WEB_DIR
chmod -R 755 $WEB_DIR
chmod -R 777 $WEB_DIR/uploads
print_success "Permisos configurados"

# 6. Habilitar módulos de Apache
echo ""
echo "[6/8] Configurando Apache..."
a2enmod rewrite > /dev/null 2>&1
a2enmod headers > /dev/null 2>&1

# Crear configuración de VirtualHost
cat > /etc/apache2/sites-available/rendicion_ong.conf << EOF
<VirtualHost *:80>
    ServerAdmin admin@localhost
    DocumentRoot $WEB_DIR
    
    <Directory $WEB_DIR>
        Options -Indexes +FollowSymLinks
        AllowOverride All
        Require all granted
    </Directory>
    
    ErrorLog \${APACHE_LOG_DIR}/rendicion_error.log
    CustomLog \${APACHE_LOG_DIR}/rendicion_access.log combined
</VirtualHost>
EOF

a2ensite rendicion_ong.conf > /dev/null 2>&1
print_success "Apache configurado"

# 7. Reiniciar servicios
echo ""
echo "[7/8] Reiniciando servicios..."
systemctl restart apache2
systemctl restart mysql
print_success "Servicios reiniciados"

# 8. Verificar instalación
echo ""
echo "[8/8] Verificando instalación..."

# Verificar Apache
if systemctl is-active --quiet apache2; then
    print_success "Apache está corriendo"
else
    print_error "Apache no está corriendo"
fi

# Verificar MySQL
if systemctl is-active --quiet mysql; then
    print_success "MySQL está corriendo"
else
    print_error "MySQL no está corriendo"
fi

# Verificar archivos
if [ -f "$WEB_DIR/index.php" ]; then
    print_success "Archivos instalados correctamente"
else
    print_error "Error en la instalación de archivos"
fi

echo ""
echo "=========================================="
echo "  Instalación Completada"
echo "=========================================="
echo ""
echo "Información de acceso:"
echo ""
echo "  URL del sistema: http://localhost/rendicion_ong"
echo "  o http://$(hostname -I | awk '{print $1}')/rendicion_ong"
echo ""
echo "  Usuario: Rendicion"
echo "  Contraseña: cuentas2025"
echo ""
echo "Base de datos:"
echo "  Host: localhost"
echo "  Usuario: $DB_USER"
echo "  Contraseña: $DB_PASS"
echo "  Base de datos: $DB_NAME"
echo ""
print_warning "IMPORTANTE: Cambie las contraseñas por defecto después del primer acceso"
echo ""
echo "Para ver los logs de Apache:"
echo "  tail -f /var/log/apache2/rendicion_error.log"
echo ""
echo "=========================================="

