<?php
require_once 'config.php';

if (!isLoggedIn() || (!hasRole('Administrador') && !hasRole('Supervisor'))) {
    redirect('dashboard.php');
}

$db = getDB();
$tipo_exportacion = $_GET['tipo'] ?? 'excel';

// Obtener información del usuario actual
$stmt = $db->prepare("SELECT u.*, r.nombre as rol_nombre FROM usuarios u 
                      INNER JOIN roles r ON u.rol_id = r.id 
                      WHERE u.id = ?");
$stmt->execute([$_SESSION['user_id']]);
$usuario = $stmt->fetch();

// Filtros
$fecha_desde = $_GET['fecha_desde'] ?? date('Y-m-01');
$fecha_hasta = $_GET['fecha_hasta'] ?? date('Y-m-d');
$usuario_filtro = $_GET['usuario_filtro'] ?? '';
$tipo_filtro = $_GET['tipo_filtro'] ?? '';

// Construir query según el rol
$where_conditions = ["m.fecha_movimiento BETWEEN ? AND ?"];
$params = [$fecha_desde, $fecha_hasta];

if ($usuario['rol_nombre'] === 'Supervisor') {
    $where_conditions[] = "(u.supervisor_id = ? OR m.usuario_id = ?)";
    $params[] = $_SESSION['user_id'];
    $params[] = $_SESSION['user_id'];
}

if ($usuario_filtro) {
    $where_conditions[] = "m.usuario_id = ?";
    $params[] = $usuario_filtro;
}

if ($tipo_filtro) {
    $where_conditions[] = "m.tipo = ?";
    $params[] = $tipo_filtro;
}

$where_clause = implode(' AND ', $where_conditions);

// Obtener movimientos
$query = "SELECT m.*, u.nombre_completo, c.nombre as categoria, r.nombre as rol 
          FROM movimientos m 
          INNER JOIN usuarios u ON m.usuario_id = u.id 
          INNER JOIN categorias c ON m.categoria_id = c.id 
          INNER JOIN roles r ON u.rol_id = r.id
          WHERE $where_clause
          ORDER BY m.fecha_movimiento DESC, m.created_at DESC";

$stmt = $db->prepare($query);
$stmt->execute($params);
$movimientos = $stmt->fetchAll();

// Calcular totales
$total_ingresos = 0;
$total_egresos = 0;
foreach ($movimientos as $mov) {
    if ($mov['tipo'] === 'Ingreso') {
        $total_ingresos += $mov['monto'];
    } else {
        $total_egresos += $mov['monto'];
    }
}

if ($tipo_exportacion === 'excel') {
    // Exportar a CSV (compatible con Excel)
    header('Content-Type: text/csv; charset=utf-8');
    header('Content-Disposition: attachment; filename="reporte_movimientos_' . date('Y-m-d') . '.csv"');
    
    $output = fopen('php://output', 'w');
    
    // BOM para UTF-8
    fprintf($output, chr(0xEF).chr(0xBB).chr(0xBF));
    
    // Encabezados
    fputcsv($output, ['REPORTE DE MOVIMIENTOS - ' . SITE_NAME], ';');
    fputcsv($output, ['Período: ' . date('d/m/Y', strtotime($fecha_desde)) . ' - ' . date('d/m/Y', strtotime($fecha_hasta))], ';');
    fputcsv($output, ['Generado por: ' . $usuario['nombre_completo']], ';');
    fputcsv($output, ['Fecha de generación: ' . date('d/m/Y H:i:s')], ';');
    fputcsv($output, [], ';');
    
    // Columnas
    fputcsv($output, ['Fecha', 'Usuario', 'Rol', 'Categoría', 'Tipo', 'Monto (Gs.)', 'Descripción'], ';');
    
    // Datos
    foreach ($movimientos as $mov) {
        fputcsv($output, [
            date('d/m/Y', strtotime($mov['fecha_movimiento'])),
            $mov['nombre_completo'],
            $mov['rol'],
            $mov['categoria'],
            $mov['tipo'],
            number_format($mov['monto'], 0, ',', '.'),
            $mov['descripcion']
        ], ';');
    }
    
    // Totales
    fputcsv($output, [], ';');
    fputcsv($output, ['TOTALES'], ';');
    fputcsv($output, ['Total Ingresos', '', '', '', '', number_format($total_ingresos, 0, ',', '.')], ';');
    fputcsv($output, ['Total Egresos', '', '', '', '', number_format($total_egresos, 0, ',', '.')], ';');
    fputcsv($output, ['Balance', '', '', '', '', number_format($total_ingresos - $total_egresos, 0, ',', '.')], ';');
    
    fclose($output);
    exit;
    
} elseif ($tipo_exportacion === 'pdf') {
    // Exportar a PDF usando HTML
    $html = '
    <!DOCTYPE html>
    <html>
    <head>
        <meta charset="UTF-8">
        <style>
            body { font-family: Arial, sans-serif; font-size: 10px; }
            h1 { font-size: 16px; text-align: center; color: #2E7D32; }
            h2 { font-size: 12px; text-align: center; margin: 5px 0; }
            table { width: 100%; border-collapse: collapse; margin-top: 20px; }
            th { background-color: #2E7D32; color: white; padding: 8px; text-align: left; }
            td { padding: 6px; border-bottom: 1px solid #ddd; }
            .totales { margin-top: 20px; font-weight: bold; }
            .text-right { text-align: right; }
            .ingreso { color: green; }
            .egreso { color: red; }
        </style>
    </head>
    <body>
        <h1>' . SITE_NAME . '</h1>
        <h2>REPORTE DE MOVIMIENTOS</h2>
        <p style="text-align: center;">
            Período: ' . date('d/m/Y', strtotime($fecha_desde)) . ' - ' . date('d/m/Y', strtotime($fecha_hasta)) . '<br>
            Generado por: ' . htmlspecialchars($usuario['nombre_completo']) . '<br>
            Fecha: ' . date('d/m/Y H:i:s') . '
        </p>
        
        <table>
            <thead>
                <tr>
                    <th>Fecha</th>
                    <th>Usuario</th>
                    <th>Rol</th>
                    <th>Categoría</th>
                    <th>Tipo</th>
                    <th class="text-right">Monto (Gs.)</th>
                </tr>
            </thead>
            <tbody>';
    
    foreach ($movimientos as $mov) {
        $clase = $mov['tipo'] === 'Ingreso' ? 'ingreso' : 'egreso';
        $html .= '<tr>
            <td>' . date('d/m/Y', strtotime($mov['fecha_movimiento'])) . '</td>
            <td>' . htmlspecialchars($mov['nombre_completo']) . '</td>
            <td>' . htmlspecialchars($mov['rol']) . '</td>
            <td>' . htmlspecialchars($mov['categoria']) . '</td>
            <td>' . htmlspecialchars($mov['tipo']) . '</td>
            <td class="text-right ' . $clase . '">' . number_format($mov['monto'], 0, ',', '.') . '</td>
        </tr>';
    }
    
    $html .= '</tbody>
        </table>
        
        <div class="totales">
            <p>Total Ingresos: Gs. ' . number_format($total_ingresos, 0, ',', '.') . '</p>
            <p>Total Egresos: Gs. ' . number_format($total_egresos, 0, ',', '.') . '</p>
            <p>Balance: Gs. ' . number_format($total_ingresos - $total_egresos, 0, ',', '.') . '</p>
        </div>
    </body>
    </html>';
    
    // Guardar HTML temporal
    $temp_html = '/tmp/reporte_' . uniqid() . '.html';
    file_put_contents($temp_html, $html);
    
    // Convertir a PDF usando wkhtmltopdf si está disponible, sino usar navegador
    $pdf_file = '/tmp/reporte_' . uniqid() . '.pdf';
    
    // Intentar con wkhtmltopdf
    exec("which wkhtmltopdf", $output, $return_var);
    if ($return_var === 0) {
        exec("wkhtmltopdf $temp_html $pdf_file 2>&1", $output, $return_var);
    }
    
    // Si wkhtmltopdf no está disponible, generar HTML para descarga
    if (!file_exists($pdf_file) || filesize($pdf_file) === 0) {
        header('Content-Type: text/html; charset=utf-8');
        header('Content-Disposition: attachment; filename="reporte_movimientos_' . date('Y-m-d') . '.html"');
        echo $html;
        unlink($temp_html);
        exit;
    }
    
    // Enviar PDF
    header('Content-Type: application/pdf');
    header('Content-Disposition: attachment; filename="reporte_movimientos_' . date('Y-m-d') . '.pdf"');
    readfile($pdf_file);
    
    // Limpiar archivos temporales
    unlink($temp_html);
    unlink($pdf_file);
    exit;
}
?>

