<?php
require_once 'config.php';

if (!isLoggedIn()) {
    redirect('index.php');
}

$db = getDB();
$mensaje = '';
$error = '';

// Obtener ID del movimiento
$movimiento_id = isset($_GET['id']) ? (int)$_GET['id'] : 0;

if (!$movimiento_id) {
    redirect('dashboard.php');
}

// Obtener información del usuario actual
$stmt = $db->prepare("SELECT u.*, r.nombre as rol_nombre FROM usuarios u 
                      INNER JOIN roles r ON u.rol_id = r.id 
                      WHERE u.id = ?");
$stmt->execute([$_SESSION['user_id']]);
$usuario = $stmt->fetch();

// Obtener el movimiento
$stmt = $db->prepare("SELECT m.*, c.nombre as categoria_nombre, u.nombre_completo as usuario_nombre 
                      FROM movimientos m 
                      INNER JOIN categorias c ON m.categoria_id = c.id 
                      INNER JOIN usuarios u ON m.usuario_id = u.id
                      WHERE m.id = ?");
$stmt->execute([$movimiento_id]);
$movimiento = $stmt->fetch();

if (!$movimiento) {
    redirect('dashboard.php');
}

// Verificar permisos
$puede_editar = false;
if ($usuario['rol_nombre'] === 'Administrador') {
    $puede_editar = true;
} elseif ($usuario['rol_nombre'] === 'Supervisor') {
    // Supervisor puede editar movimientos de sus operativos
    $stmt = $db->prepare("SELECT id FROM usuarios WHERE id = ? AND supervisor_id = ?");
    $stmt->execute([$movimiento['usuario_id'], $usuario['id']]);
    if ($stmt->fetch()) {
        $puede_editar = true;
    }
} elseif ($usuario['rol_nombre'] === 'Operativo' && $movimiento['usuario_id'] == $usuario['id']) {
    // Operativo puede editar solo sus propios movimientos
    $puede_editar = true;
}

if (!$puede_editar) {
    redirect('dashboard.php');
}

// Procesar formulario
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $categoria_id = (int)$_POST['categoria_id'];
    $monto_nuevo = (float)str_replace(['.', ','], ['', '.'], $_POST['monto']);
    $descripcion = cleanInput($_POST['descripcion']);
    $fecha_movimiento = cleanInput($_POST['fecha_movimiento']);
    
    // Procesar comprobante si se sube uno nuevo
    $comprobante_nombre = $movimiento['comprobante'];
    if (isset($_FILES['comprobante']) && $_FILES['comprobante']['error'] === UPLOAD_ERR_OK) {
        $allowed = ['jpg', 'jpeg', 'png', 'gif'];
        $filename = $_FILES['comprobante']['name'];
        $ext = strtolower(pathinfo($filename, PATHINFO_EXTENSION));
        
        if (in_array($ext, $allowed)) {
            // Eliminar comprobante anterior si existe
            if ($comprobante_nombre && file_exists(__DIR__ . '/uploads/' . $comprobante_nombre)) {
                unlink(__DIR__ . '/uploads/' . $comprobante_nombre);
            }
            
            $comprobante_nombre = 'comprobante_' . time() . '_' . uniqid() . '.' . $ext;
            $upload_path = __DIR__ . '/uploads/' . $comprobante_nombre;
            
            if (!move_uploaded_file($_FILES['comprobante']['tmp_name'], $upload_path)) {
                $error = 'Error al subir el nuevo comprobante';
                $comprobante_nombre = $movimiento['comprobante'];
            }
        }
    }
    
    if (!$error) {
        if ($monto_nuevo <= 0) {
            $error = 'El monto debe ser mayor a cero';
        } else {
            try {
                $db->beginTransaction();
                
                $monto_anterior = $movimiento['monto'];
                $diferencia = $monto_nuevo - $monto_anterior;
                
                // Actualizar movimiento
                $stmt = $db->prepare("UPDATE movimientos SET 
                                      categoria_id = ?,
                                      monto = ?,
                                      descripcion = ?,
                                      fecha_movimiento = ?,
                                      comprobante = ?
                                      WHERE id = ?");
                $stmt->execute([$categoria_id, $monto_nuevo, $descripcion, $fecha_movimiento, $comprobante_nombre, $movimiento_id]);
                
                // Si es egreso, actualizar saldo del usuario
                if ($movimiento['tipo'] === 'Egreso') {
                    // Restar la diferencia del saldo (si aumentó el gasto, resta más; si disminuyó, suma)
                    $stmt = $db->prepare("UPDATE usuarios SET 
                                          saldo_disponible = saldo_disponible - ?
                                          WHERE id = ?");
                    $stmt->execute([$diferencia, $movimiento['usuario_id']]);
                }
                
                // Log
                $stmt = $db->prepare("INSERT INTO logs (usuario_id, accion, detalles, ip_address) VALUES (?, ?, ?, ?)");
                $stmt->execute([$_SESSION['user_id'], 'Editar movimiento', "ID: $movimiento_id, Monto anterior: $monto_anterior, Monto nuevo: $monto_nuevo", $_SERVER['REMOTE_ADDR']]);
                
                $db->commit();
                
                $mensaje = 'Movimiento actualizado exitosamente';
                
                // Recargar movimiento
                $stmt = $db->prepare("SELECT m.*, c.nombre as categoria_nombre, u.nombre_completo as usuario_nombre 
                                      FROM movimientos m 
                                      INNER JOIN categorias c ON m.categoria_id = c.id 
                                      INNER JOIN usuarios u ON m.usuario_id = u.id
                                      WHERE m.id = ?");
                $stmt->execute([$movimiento_id]);
                $movimiento = $stmt->fetch();
                
            } catch (Exception $e) {
                $db->rollBack();
                $error = 'Error al actualizar el movimiento: ' . $e->getMessage();
            }
        }
    }
}

// Obtener categorías según el tipo del movimiento
$stmt = $db->prepare("SELECT * FROM categorias WHERE tipo = ? AND activo = 1 ORDER BY nombre");
$stmt->execute([$movimiento['tipo']]);
$categorias = $stmt->fetchAll();

$page_title = 'Editar Movimiento';
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo $page_title; ?> - <?php echo SITE_NAME; ?></title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.10.0/font/bootstrap-icons.css">
    <link rel="stylesheet" href="assets/css/style.css">
</head>
<body>
    <?php include 'includes/navbar.php'; ?>
    
    <div class="container-fluid">
        <div class="row">
            <?php include 'includes/sidebar.php'; ?>
            
            <main class="col-md-9 ms-sm-auto col-lg-10 px-md-4">
                <div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pt-3 pb-2 mb-3 border-bottom">
                    <h1 class="h2">
                        <i class="bi bi-pencil"></i> Editar Movimiento
                    </h1>
                    <a href="javascript:history.back()" class="btn btn-secondary">
                        <i class="bi bi-arrow-left"></i> Volver
                    </a>
                </div>
                
                <?php if ($mensaje): ?>
                    <div class="alert alert-success alert-dismissible fade show" role="alert">
                        <i class="bi bi-check-circle"></i> <?php echo $mensaje; ?>
                        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                    </div>
                <?php endif; ?>
                
                <?php if ($error): ?>
                    <div class="alert alert-danger alert-dismissible fade show" role="alert">
                        <i class="bi bi-exclamation-triangle"></i> <?php echo $error; ?>
                        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                    </div>
                <?php endif; ?>
                
                <div class="row">
                    <div class="col-md-8">
                        <div class="card">
                            <div class="card-header">
                                <h5 class="mb-0">Información del Movimiento</h5>
                            </div>
                            <div class="card-body">
                                <div class="row mb-3">
                                    <div class="col-md-6">
                                        <strong>Usuario:</strong> <?php echo htmlspecialchars($movimiento['usuario_nombre']); ?>
                                    </div>
                                    <div class="col-md-6">
                                        <strong>Tipo:</strong> 
                                        <span class="badge bg-<?php echo $movimiento['tipo'] === 'Ingreso' ? 'success' : 'danger'; ?>">
                                            <?php echo $movimiento['tipo']; ?>
                                        </span>
                                    </div>
                                </div>
                                
                                <form method="POST" enctype="multipart/form-data">
                                    <div class="mb-3">
                                        <label class="form-label">Categoría *</label>
                                        <select class="form-select" name="categoria_id" required>
                                            <?php foreach ($categorias as $cat): ?>
                                                <option value="<?php echo $cat['id']; ?>" 
                                                        <?php echo $cat['id'] == $movimiento['categoria_id'] ? 'selected' : ''; ?>>
                                                    <?php echo htmlspecialchars($cat['nombre']); ?>
                                                </option>
                                            <?php endforeach; ?>
                                        </select>
                                    </div>
                                    
                                    <div class="mb-3">
                                        <label class="form-label">Monto (Gs.) *</label>
                                        <input type="text" class="form-control" name="monto" id="monto" 
                                               value="<?php echo number_format($movimiento['monto'], 0, ',', '.'); ?>" required>
                                    </div>
                                    
                                    <div class="mb-3">
                                        <label class="form-label">Fecha *</label>
                                        <input type="date" class="form-control" name="fecha_movimiento" 
                                               value="<?php echo $movimiento['fecha_movimiento']; ?>" required>
                                    </div>
                                    
                                    <div class="mb-3">
                                        <label class="form-label">Descripción</label>
                                        <textarea class="form-control" name="descripcion" rows="3"><?php echo htmlspecialchars($movimiento['descripcion']); ?></textarea>
                                    </div>
                                    
                                    <div class="mb-3">
                                        <label class="form-label">Comprobante Fotográfico</label>
                                        <?php if ($movimiento['comprobante']): ?>
                                            <div class="mb-2">
                                                <img src="uploads/<?php echo htmlspecialchars($movimiento['comprobante']); ?>" 
                                                     class="img-thumbnail" style="max-width: 200px;">
                                                <p class="text-muted small">Comprobante actual</p>
                                            </div>
                                        <?php endif; ?>
                                        <input type="file" class="form-control" name="comprobante" 
                                               accept="image/*" capture="environment">
                                        <small class="text-muted">
                                            Puede subir un nuevo comprobante para reemplazar el actual (JPG, PNG, GIF)
                                        </small>
                                    </div>
                                    
                                    <div class="d-grid gap-2">
                                        <button type="submit" class="btn btn-primary">
                                            <i class="bi bi-save"></i> Guardar Cambios
                                        </button>
                                        <a href="javascript:history.back()" class="btn btn-secondary">
                                            <i class="bi bi-x-circle"></i> Cancelar
                                        </a>
                                    </div>
                                </form>
                            </div>
                        </div>
                    </div>
                    
                    <div class="col-md-4">
                        <div class="card">
                            <div class="card-header bg-info text-white">
                                <h6 class="mb-0"><i class="bi bi-info-circle"></i> Información Adicional</h6>
                            </div>
                            <div class="card-body">
                                <p><strong>Creado:</strong><br><?php echo date('d/m/Y H:i', strtotime($movimiento['created_at'])); ?></p>
                                <p><strong>Saldo Anterior:</strong><br><?php echo formatCurrency($movimiento['saldo_anterior']); ?></p>
                                <p><strong>Saldo Posterior:</strong><br><?php echo formatCurrency($movimiento['saldo_posterior']); ?></p>
                                <?php if ($movimiento['created_by']): ?>
                                    <p><strong>Creado por:</strong><br>
                                    <?php 
                                    $stmt = $db->prepare("SELECT nombre_completo FROM usuarios WHERE id = ?");
                                    $stmt->execute([$movimiento['created_by']]);
                                    echo htmlspecialchars($stmt->fetchColumn());
                                    ?>
                                    </p>
                                <?php endif; ?>
                            </div>
                        </div>
                        
                        <div class="alert alert-warning mt-3">
                            <i class="bi bi-exclamation-triangle"></i>
                            <strong>Atención:</strong> Al modificar el monto, el saldo del usuario se ajustará automáticamente.
                        </div>
                    </div>
                </div>
            </main>
        </div>
    </div>
    
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Formatear monto con separadores de miles
        document.getElementById('monto').addEventListener('input', function(e) {
            let value = e.target.value.replace(/\D/g, '');
            e.target.value = value.replace(/\B(?=(\d{3})+(?!\d))/g, '.');
        });
    </script>
</body>
</html>

