<?php
require_once 'config.php';

if (!isLoggedIn() || !hasRole('Administrador')) {
    redirect('dashboard.php');
}

$db = getDB();
$mensaje = '';
$error = '';

// Procesar acciones
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $accion = $_POST['accion'] ?? '';
    
    if ($accion === 'crear') {
        $nombre = cleanInput($_POST['nombre']);
        $tipo = cleanInput($_POST['tipo']);
        $descripcion = cleanInput($_POST['descripcion']);
        
        $stmt = $db->prepare("INSERT INTO categorias (nombre, tipo, descripcion) VALUES (?, ?, ?)");
        if ($stmt->execute([$nombre, $tipo, $descripcion])) {
            $mensaje = 'Categoría creada exitosamente';
            
            // Log
            $stmt = $db->prepare("INSERT INTO logs (usuario_id, accion, detalles, ip_address) VALUES (?, ?, ?, ?)");
            $stmt->execute([$_SESSION['user_id'], 'Crear categoría', "Nombre: $nombre, Tipo: $tipo", $_SERVER['REMOTE_ADDR']]);
        } else {
            $error = 'Error al crear la categoría';
        }
    } elseif ($accion === 'editar') {
        $categoria_id = (int)$_POST['categoria_id'];
        $nombre = cleanInput($_POST['nombre']);
        $tipo = cleanInput($_POST['tipo']);
        $descripcion = cleanInput($_POST['descripcion']);
        
        $stmt = $db->prepare("UPDATE categorias SET nombre = ?, tipo = ?, descripcion = ? WHERE id = ?");
        if ($stmt->execute([$nombre, $tipo, $descripcion, $categoria_id])) {
            $mensaje = 'Categoría actualizada exitosamente';
            
            // Log
            $stmt = $db->prepare("INSERT INTO logs (usuario_id, accion, detalles, ip_address) VALUES (?, ?, ?, ?)");
            $stmt->execute([$_SESSION['user_id'], 'Editar categoría', "ID: $categoria_id, Nombre: $nombre", $_SERVER['REMOTE_ADDR']]);
        } else {
            $error = 'Error al actualizar la categoría';
        }
    } elseif ($accion === 'eliminar') {
        $categoria_id = (int)$_POST['categoria_id'];
        
        // Verificar si la categoría está en uso
        $stmt = $db->prepare("SELECT COUNT(*) FROM movimientos WHERE categoria_id = ?");
        $stmt->execute([$categoria_id]);
        $en_uso = $stmt->fetchColumn();
        
        if ($en_uso > 0) {
            $error = "No se puede eliminar la categoría porque tiene $en_uso movimientos asociados. Puede desactivarla en su lugar.";
        } else {
            $stmt = $db->prepare("DELETE FROM categorias WHERE id = ?");
            if ($stmt->execute([$categoria_id])) {
                $mensaje = 'Categoría eliminada exitosamente';
                
                // Log
                $stmt = $db->prepare("INSERT INTO logs (usuario_id, accion, detalles, ip_address) VALUES (?, ?, ?, ?)");
                $stmt->execute([$_SESSION['user_id'], 'Eliminar categoría', "ID: $categoria_id", $_SERVER['REMOTE_ADDR']]);
            } else {
                $error = 'Error al eliminar la categoría';
            }
        }
    } elseif ($accion === 'desactivar') {
        $categoria_id = (int)$_POST['categoria_id'];
        
        $stmt = $db->prepare("UPDATE categorias SET activo = 0 WHERE id = ?");
        if ($stmt->execute([$categoria_id])) {
            $mensaje = 'Categoría desactivada exitosamente';
            
            // Log
            $stmt = $db->prepare("INSERT INTO logs (usuario_id, accion, detalles, ip_address) VALUES (?, ?, ?, ?)");
            $stmt->execute([$_SESSION['user_id'], 'Desactivar categoría', "ID: $categoria_id", $_SERVER['REMOTE_ADDR']]);
        } else {
            $error = 'Error al desactivar la categoría';
        }
    } elseif ($accion === 'activar') {
        $categoria_id = (int)$_POST['categoria_id'];
        
        $stmt = $db->prepare("UPDATE categorias SET activo = 1 WHERE id = ?");
        if ($stmt->execute([$categoria_id])) {
            $mensaje = 'Categoría activada exitosamente';
            
            // Log
            $stmt = $db->prepare("INSERT INTO logs (usuario_id, accion, detalles, ip_address) VALUES (?, ?, ?, ?)");
            $stmt->execute([$_SESSION['user_id'], 'Activar categoría', "ID: $categoria_id", $_SERVER['REMOTE_ADDR']]);
        } else {
            $error = 'Error al activar la categoría';
        }
    }
}

// Obtener categorías (activas e inactivas)
$categorias = $db->query("SELECT * FROM categorias ORDER BY activo DESC, tipo, nombre")->fetchAll();

// Obtener información del usuario actual
$stmt = $db->prepare("SELECT u.*, r.nombre as rol_nombre FROM usuarios u 
                      INNER JOIN roles r ON u.rol_id = r.id 
                      WHERE u.id = ?");
$stmt->execute([$_SESSION['user_id']]);
$usuario = $stmt->fetch();

$page_title = 'Gestión de Categorías';
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo $page_title; ?> - <?php echo SITE_NAME; ?></title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.10.0/font/bootstrap-icons.css">
    <link rel="stylesheet" href="assets/css/style.css">
</head>
<body>
    <?php include 'includes/navbar.php'; ?>
    
    <div class="container-fluid">
        <div class="row">
            <?php include 'includes/sidebar.php'; ?>
            
            <main class="col-md-9 ms-sm-auto col-lg-10 px-md-4">
                <div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pt-3 pb-2 mb-3 border-bottom">
                    <h1 class="h2">
                        <i class="bi bi-tags"></i> Gestión de Categorías
                    </h1>
                    <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#nuevaCategoriaModal">
                        <i class="bi bi-plus-circle"></i> Nueva Categoría
                    </button>
                </div>
                
                <?php if ($mensaje): ?>
                    <div class="alert alert-success alert-dismissible fade show" role="alert">
                        <i class="bi bi-check-circle"></i> <?php echo $mensaje; ?>
                        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                    </div>
                <?php endif; ?>
                
                <?php if ($error): ?>
                    <div class="alert alert-danger alert-dismissible fade show" role="alert">
                        <i class="bi bi-exclamation-triangle"></i> <?php echo $error; ?>
                        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                    </div>
                <?php endif; ?>
                
                <div class="row">
                    <div class="col-md-6 mb-4">
                        <div class="card">
                            <div class="card-header bg-success text-white">
                                <h5 class="mb-0"><i class="bi bi-arrow-down-circle"></i> Categorías de Ingreso</h5>
                            </div>
                            <div class="card-body">
                                <div class="table-responsive">
                                    <table class="table table-hover">
                                        <thead>
                                            <tr>
                                                <th>Estado</th>
                                                <th>Nombre</th>
                                                <th>Descripción</th>
                                                <th>Acciones</th>
                                            </tr>
                                        </thead>
                                        <tbody>
                                            <?php 
                                            $ingresos = array_filter($categorias, fn($c) => $c['tipo'] === 'Ingreso');
                                            if (empty($ingresos)): 
                                            ?>
                                                <tr>
                                                    <td colspan="4" class="text-center text-muted">No hay categorías de ingreso</td>
                                                </tr>
                                            <?php else: ?>
                                                <?php foreach ($ingresos as $cat): ?>
                                                    <tr class="<?php echo $cat['activo'] ? '' : 'table-secondary'; ?>">
                                                        <td>
                                                            <?php if ($cat['activo']): ?>
                                                                <span class="badge bg-success">Activo</span>
                                                            <?php else: ?>
                                                                <span class="badge bg-secondary">Inactivo</span>
                                                            <?php endif; ?>
                                                        </td>
                                                        <td><?php echo htmlspecialchars($cat['nombre']); ?></td>
                                                        <td><?php echo htmlspecialchars(substr($cat['descripcion'], 0, 30)); ?></td>
                                                        <td>
                                                            <button class="btn btn-sm btn-primary" 
                                                                    onclick="editarCategoria(<?php echo htmlspecialchars(json_encode($cat)); ?>)"
                                                                    title="Editar">
                                                                <i class="bi bi-pencil"></i>
                                                            </button>
                                                            
                                                            <?php if ($cat['activo']): ?>
                                                                <form method="POST" style="display: inline;" 
                                                                      onsubmit="return confirm('¿Desactivar esta categoría?');">
                                                                    <input type="hidden" name="accion" value="desactivar">
                                                                    <input type="hidden" name="categoria_id" value="<?php echo $cat['id']; ?>">
                                                                    <button type="submit" class="btn btn-sm btn-warning" title="Desactivar">
                                                                        <i class="bi bi-pause-circle"></i>
                                                                    </button>
                                                                </form>
                                                            <?php else: ?>
                                                                <form method="POST" style="display: inline;">
                                                                    <input type="hidden" name="accion" value="activar">
                                                                    <input type="hidden" name="categoria_id" value="<?php echo $cat['id']; ?>">
                                                                    <button type="submit" class="btn btn-sm btn-success" title="Activar">
                                                                        <i class="bi bi-play-circle"></i>
                                                                    </button>
                                                                </form>
                                                            <?php endif; ?>
                                                            
                                                            <form method="POST" style="display: inline;" 
                                                                  onsubmit="return confirm('¿Eliminar permanentemente esta categoría?');">
                                                                <input type="hidden" name="accion" value="eliminar">
                                                                <input type="hidden" name="categoria_id" value="<?php echo $cat['id']; ?>">
                                                                <button type="submit" class="btn btn-sm btn-danger" title="Eliminar">
                                                                    <i class="bi bi-trash"></i>
                                                                </button>
                                                            </form>
                                                        </td>
                                                    </tr>
                                                <?php endforeach; ?>
                                            <?php endif; ?>
                                        </tbody>
                                    </table>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <div class="col-md-6 mb-4">
                        <div class="card">
                            <div class="card-header bg-danger text-white">
                                <h5 class="mb-0"><i class="bi bi-arrow-up-circle"></i> Categorías de Egreso</h5>
                            </div>
                            <div class="card-body">
                                <div class="table-responsive">
                                    <table class="table table-hover">
                                        <thead>
                                            <tr>
                                                <th>Estado</th>
                                                <th>Nombre</th>
                                                <th>Descripción</th>
                                                <th>Acciones</th>
                                            </tr>
                                        </thead>
                                        <tbody>
                                            <?php 
                                            $egresos = array_filter($categorias, fn($c) => $c['tipo'] === 'Egreso');
                                            if (empty($egresos)): 
                                            ?>
                                                <tr>
                                                    <td colspan="4" class="text-center text-muted">No hay categorías de egreso</td>
                                                </tr>
                                            <?php else: ?>
                                                <?php foreach ($egresos as $cat): ?>
                                                    <tr class="<?php echo $cat['activo'] ? '' : 'table-secondary'; ?>">
                                                        <td>
                                                            <?php if ($cat['activo']): ?>
                                                                <span class="badge bg-success">Activo</span>
                                                            <?php else: ?>
                                                                <span class="badge bg-secondary">Inactivo</span>
                                                            <?php endif; ?>
                                                        </td>
                                                        <td><?php echo htmlspecialchars($cat['nombre']); ?></td>
                                                        <td><?php echo htmlspecialchars(substr($cat['descripcion'], 0, 30)); ?></td>
                                                        <td>
                                                            <button class="btn btn-sm btn-primary" 
                                                                    onclick="editarCategoria(<?php echo htmlspecialchars(json_encode($cat)); ?>)"
                                                                    title="Editar">
                                                                <i class="bi bi-pencil"></i>
                                                            </button>
                                                            
                                                            <?php if ($cat['activo']): ?>
                                                                <form method="POST" style="display: inline;" 
                                                                      onsubmit="return confirm('¿Desactivar esta categoría?');">
                                                                    <input type="hidden" name="accion" value="desactivar">
                                                                    <input type="hidden" name="categoria_id" value="<?php echo $cat['id']; ?>">
                                                                    <button type="submit" class="btn btn-sm btn-warning" title="Desactivar">
                                                                        <i class="bi bi-pause-circle"></i>
                                                                    </button>
                                                                </form>
                                                            <?php else: ?>
                                                                <form method="POST" style="display: inline;">
                                                                    <input type="hidden" name="accion" value="activar">
                                                                    <input type="hidden" name="categoria_id" value="<?php echo $cat['id']; ?>">
                                                                    <button type="submit" class="btn btn-sm btn-success" title="Activar">
                                                                        <i class="bi bi-play-circle"></i>
                                                                    </button>
                                                                </form>
                                                            <?php endif; ?>
                                                            
                                                            <form method="POST" style="display: inline;" 
                                                                  onsubmit="return confirm('¿Eliminar permanentemente esta categoría?');">
                                                                <input type="hidden" name="accion" value="eliminar">
                                                                <input type="hidden" name="categoria_id" value="<?php echo $cat['id']; ?>">
                                                                <button type="submit" class="btn btn-sm btn-danger" title="Eliminar">
                                                                    <i class="bi bi-trash"></i>
                                                                </button>
                                                            </form>
                                                        </td>
                                                    </tr>
                                                <?php endforeach; ?>
                                            <?php endif; ?>
                                        </tbody>
                                    </table>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </main>
        </div>
    </div>
    
    <!-- Modal Nueva Categoría -->
    <div class="modal fade" id="nuevaCategoriaModal" tabindex="-1">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title"><i class="bi bi-plus-circle"></i> Nueva Categoría</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <form method="POST">
                    <div class="modal-body">
                        <input type="hidden" name="accion" value="crear">
                        
                        <div class="mb-3">
                            <label class="form-label">Nombre *</label>
                            <input type="text" class="form-control" name="nombre" required>
                        </div>
                        
                        <div class="mb-3">
                            <label class="form-label">Tipo *</label>
                            <select class="form-select" name="tipo" required>
                                <option value="">Seleccione un tipo</option>
                                <option value="Ingreso">Ingreso</option>
                                <option value="Egreso">Egreso</option>
                            </select>
                        </div>
                        
                        <div class="mb-3">
                            <label class="form-label">Descripción</label>
                            <textarea class="form-control" name="descripcion" rows="3"></textarea>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancelar</button>
                        <button type="submit" class="btn btn-primary">
                            <i class="bi bi-save"></i> Guardar Categoría
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>
    
    <!-- Modal Editar Categoría -->
    <div class="modal fade" id="editarCategoriaModal" tabindex="-1">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title"><i class="bi bi-pencil"></i> Editar Categoría</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <form method="POST">
                    <div class="modal-body">
                        <input type="hidden" name="accion" value="editar">
                        <input type="hidden" name="categoria_id" id="edit_categoria_id">
                        
                        <div class="mb-3">
                            <label class="form-label">Nombre *</label>
                            <input type="text" class="form-control" name="nombre" id="edit_nombre" required>
                        </div>
                        
                        <div class="mb-3">
                            <label class="form-label">Tipo *</label>
                            <select class="form-select" name="tipo" id="edit_tipo" required>
                                <option value="Ingreso">Ingreso</option>
                                <option value="Egreso">Egreso</option>
                            </select>
                        </div>
                        
                        <div class="mb-3">
                            <label class="form-label">Descripción</label>
                            <textarea class="form-control" name="descripcion" id="edit_descripcion" rows="3"></textarea>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancelar</button>
                        <button type="submit" class="btn btn-primary">
                            <i class="bi bi-save"></i> Actualizar Categoría
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>
    
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        function editarCategoria(categoria) {
            document.getElementById('edit_categoria_id').value = categoria.id;
            document.getElementById('edit_nombre').value = categoria.nombre;
            document.getElementById('edit_tipo').value = categoria.tipo;
            document.getElementById('edit_descripcion').value = categoria.descripcion || '';
            
            const modal = new bootstrap.Modal(document.getElementById('editarCategoriaModal'));
            modal.show();
        }
    </script>
</body>
</html>

