<?php
require_once 'config.php';

if (!isLoggedIn() || !hasRole('Supervisor')) {
    redirect('dashboard.php');
}

$db = getDB();
$mensaje = '';
$error = '';

// Obtener información del supervisor
$stmt = $db->prepare("SELECT u.*, r.nombre as rol_nombre FROM usuarios u 
                      INNER JOIN roles r ON u.rol_id = r.id 
                      WHERE u.id = ?");
$stmt->execute([$_SESSION['user_id']]);
$usuario = $stmt->fetch();

// Calcular saldo disponible del supervisor
$stmt = $db->prepare("SELECT COALESCE(SUM(fondo_asignado), 0) as total_asignado_operativos 
                      FROM usuarios WHERE supervisor_id = ? AND activo = 1");
$stmt->execute([$usuario['id']]);
$asignado_operativos = $stmt->fetchColumn();
$saldo_disponible_supervisor = $usuario['fondo_asignado'] - $asignado_operativos;

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $asignado_a = (int)$_POST['asignado_a'];
    $monto = (float)str_replace(['.', ','], ['', '.'], $_POST['monto']);
    $descripcion = cleanInput($_POST['descripcion']);
    $fecha_asignacion = cleanInput($_POST['fecha_asignacion']);
    
    // Verificar que el operativo pertenezca al supervisor
    $stmt = $db->prepare("SELECT id, fondo_asignado, saldo_disponible FROM usuarios 
                          WHERE id = ? AND supervisor_id = ? AND activo = 1");
    $stmt->execute([$asignado_a, $_SESSION['user_id']]);
    $operativo = $stmt->fetch();
    
    if (!$operativo) {
        $error = 'El operativo seleccionado no pertenece a su supervisión';
    } elseif ($monto <= 0) {
        $error = 'El monto debe ser mayor a cero';
    } elseif ($monto > $saldo_disponible_supervisor) {
        $error = 'El monto supera su saldo disponible (' . formatCurrency($saldo_disponible_supervisor) . ')';
    } else {
        try {
            $db->beginTransaction();
            
            // Registrar asignación
            $stmt = $db->prepare("INSERT INTO asignaciones (asignado_por, asignado_a, monto, descripcion, fecha_asignacion) 
                                  VALUES (?, ?, ?, ?, ?)");
            $stmt->execute([$_SESSION['user_id'], $asignado_a, $monto, $descripcion, $fecha_asignacion]);
            
            // Actualizar fondos del operativo
            $stmt = $db->prepare("UPDATE usuarios SET 
                                  fondo_asignado = fondo_asignado + ?,
                                  saldo_disponible = saldo_disponible + ?
                                  WHERE id = ?");
            $stmt->execute([$monto, $monto, $asignado_a]);
            
            // Actualizar saldo disponible del supervisor (restar el monto asignado)
            $stmt = $db->prepare("UPDATE usuarios SET 
                                  saldo_disponible = saldo_disponible - ?
                                  WHERE id = ?");
            $stmt->execute([$monto, $_SESSION['user_id']]);
            
            // Registrar como movimiento de ingreso para el operativo
            $saldo_nuevo = $operativo['saldo_disponible'] + $monto;
            
            $stmt = $db->prepare("INSERT INTO movimientos (usuario_id, categoria_id, tipo, monto, descripcion, 
                                  fecha_movimiento, saldo_anterior, saldo_posterior, created_by) 
                                  VALUES (?, 2, 'Ingreso', ?, ?, ?, ?, ?, ?)");
            $stmt->execute([$asignado_a, $monto, $descripcion, $fecha_asignacion, 
                           $operativo['saldo_disponible'], $saldo_nuevo, $_SESSION['user_id']]);
            
            // Log
            $stmt = $db->prepare("INSERT INTO logs (usuario_id, accion, detalles, ip_address) VALUES (?, ?, ?, ?)");
            $stmt->execute([$_SESSION['user_id'], 'Asignar fondos a operativo', "Monto: $monto a operativo ID: $asignado_a", $_SERVER['REMOTE_ADDR']]);
            
            $db->commit();
            
            $mensaje = 'Fondos asignados exitosamente al operativo';
            
            // Actualizar datos del usuario en memoria
            $stmt = $db->prepare("SELECT saldo_disponible FROM usuarios WHERE id = ?");
            $stmt->execute([$_SESSION['user_id']]);
            $usuario['saldo_disponible'] = $stmt->fetchColumn();
            
            // Recalcular saldo disponible
            $stmt = $db->prepare("SELECT COALESCE(SUM(fondo_asignado), 0) as total_asignado_operativos 
                                  FROM usuarios WHERE supervisor_id = ? AND activo = 1");
            $stmt->execute([$usuario['id']]);
            $asignado_operativos = $stmt->fetchColumn();
            $saldo_disponible_supervisor = $usuario['fondo_asignado'] - $asignado_operativos;
            
        } catch (Exception $e) {
            $db->rollBack();
            $error = 'Error al asignar fondos: ' . $e->getMessage();
        }
    }
}

// Obtener operativos del supervisor
$operativos = $db->prepare("SELECT id, nombre_completo, fondo_asignado, saldo_disponible 
                            FROM usuarios WHERE supervisor_id = ? AND activo = 1 
                            ORDER BY nombre_completo");
$operativos->execute([$_SESSION['user_id']]);
$operativos = $operativos->fetchAll();

// Obtener historial de asignaciones
$stmt = $db->prepare("SELECT a.*, u.nombre_completo as asignado_a_nombre 
                      FROM asignaciones a 
                      INNER JOIN usuarios u ON a.asignado_a = u.id 
                      WHERE a.asignado_por = ?
                      ORDER BY a.created_at DESC 
                      LIMIT 20");
$stmt->execute([$_SESSION['user_id']]);
$asignaciones = $stmt->fetchAll();

// Si hay operativo_id en la URL, preseleccionarlo
$operativo_preseleccionado = isset($_GET['operativo_id']) ? (int)$_GET['operativo_id'] : 0;

$page_title = 'Asignar Fondos a Operativos';
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo $page_title; ?> - <?php echo SITE_NAME; ?></title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.10.0/font/bootstrap-icons.css">
    <link rel="stylesheet" href="assets/css/style.css">
</head>
<body>
    <?php include 'includes/navbar.php'; ?>
    
    <div class="container-fluid">
        <div class="row">
            <?php include 'includes/sidebar.php'; ?>
            
            <main class="col-md-9 ms-sm-auto col-lg-10 px-md-4">
                <div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pt-3 pb-2 mb-3 border-bottom">
                    <h1 class="h2">
                        <i class="bi bi-cash-coin"></i> Asignar Fondos a Operativos
                    </h1>
                    <a href="operativos.php" class="btn btn-secondary">
                        <i class="bi bi-arrow-left"></i> Volver a Operativos
                    </a>
                </div>
                
                <?php if ($mensaje): ?>
                    <div class="alert alert-success alert-dismissible fade show" role="alert">
                        <i class="bi bi-check-circle"></i> <?php echo $mensaje; ?>
                        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                    </div>
                <?php endif; ?>
                
                <?php if ($error): ?>
                    <div class="alert alert-danger alert-dismissible fade show" role="alert">
                        <i class="bi bi-exclamation-triangle"></i> <?php echo $error; ?>
                        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                    </div>
                <?php endif; ?>
                
                <!-- Saldo disponible del supervisor -->
                <div class="row mb-4">
                    <div class="col-md-12">
                        <div class="card stat-card stat-success">
                            <div class="card-body">
                                <div class="d-flex justify-content-between">
                                    <div>
                                        <h6 class="text-muted mb-2">Su Saldo Disponible para Asignar</h6>
                                        <h2 class="mb-0"><?php echo formatCurrency($saldo_disponible_supervisor); ?></h2>
                                        <small class="text-muted">
                                            Fondo asignado: <?php echo formatCurrency($usuario['fondo_asignado']); ?> | 
                                            Ya asignado a operativos: <?php echo formatCurrency($asignado_operativos); ?>
                                        </small>
                                    </div>
                                    <div class="stat-icon">
                                        <i class="bi bi-wallet2"></i>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                
                <div class="row">
                    <div class="col-md-6 mb-4">
                        <div class="card">
                            <div class="card-header">
                                <h5 class="mb-0"><i class="bi bi-plus-circle"></i> Nueva Asignación</h5>
                            </div>
                            <div class="card-body">
                                <?php if (empty($operativos)): ?>
                                    <div class="alert alert-warning">
                                        <i class="bi bi-exclamation-triangle"></i> 
                                        No tiene operativos creados. 
                                        <a href="operativos.php" class="alert-link">Crear operativo</a>
                                    </div>
                                <?php elseif ($saldo_disponible_supervisor <= 0): ?>
                                    <div class="alert alert-danger">
                                        <i class="bi bi-exclamation-circle"></i> 
                                        No tiene saldo disponible para asignar. Ya asignó todo su presupuesto.
                                    </div>
                                <?php else: ?>
                                    <form method="POST">
                                        <div class="mb-3">
                                            <label class="form-label">Asignar a Operativo *</label>
                                            <select class="form-select" name="asignado_a" required>
                                                <option value="">Seleccione un operativo</option>
                                                <?php foreach ($operativos as $op): ?>
                                                    <option value="<?php echo $op['id']; ?>"
                                                            <?php echo $operativo_preseleccionado == $op['id'] ? 'selected' : ''; ?>>
                                                        <?php echo htmlspecialchars($op['nombre_completo']); ?> 
                                                        (Saldo: <?php echo formatCurrency($op['saldo_disponible']); ?>)
                                                    </option>
                                                <?php endforeach; ?>
                                            </select>
                                        </div>
                                        
                                        <div class="mb-3">
                                            <label class="form-label">Monto (Gs.) *</label>
                                            <input type="text" class="form-control" name="monto" id="monto" 
                                                   placeholder="0" required>
                                            <small class="text-muted">
                                                Disponible: <?php echo formatCurrency($saldo_disponible_supervisor); ?>
                                            </small>
                                        </div>
                                        
                                        <div class="mb-3">
                                            <label class="form-label">Fecha *</label>
                                            <input type="date" class="form-control" name="fecha_asignacion" 
                                                   value="<?php echo date('Y-m-d'); ?>" required>
                                        </div>
                                        
                                        <div class="mb-3">
                                            <label class="form-label">Descripción</label>
                                            <textarea class="form-control" name="descripcion" rows="3" 
                                                      placeholder="Descripción de la asignación"></textarea>
                                        </div>
                                        
                                        <div class="d-grid">
                                            <button type="submit" class="btn btn-primary">
                                                <i class="bi bi-save"></i> Asignar Fondos
                                            </button>
                                        </div>
                                    </form>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                    
                    <div class="col-md-6 mb-4">
                        <div class="card">
                            <div class="card-header">
                                <h5 class="mb-0"><i class="bi bi-clock-history"></i> Últimas Asignaciones</h5>
                            </div>
                            <div class="card-body">
                                <div class="table-responsive">
                                    <table class="table table-sm">
                                        <thead>
                                            <tr>
                                                <th>Fecha</th>
                                                <th>Operativo</th>
                                                <th>Monto</th>
                                            </tr>
                                        </thead>
                                        <tbody>
                                            <?php if (!empty($asignaciones)): ?>
                                                <?php foreach ($asignaciones as $asig): ?>
                                                    <tr>
                                                        <td><?php echo date('d/m/Y', strtotime($asig['fecha_asignacion'])); ?></td>
                                                        <td><?php echo htmlspecialchars($asig['asignado_a_nombre']); ?></td>
                                                        <td><?php echo formatCurrency($asig['monto']); ?></td>
                                                    </tr>
                                                <?php endforeach; ?>
                                            <?php else: ?>
                                                <tr>
                                                    <td colspan="3" class="text-center text-muted">No hay asignaciones</td>
                                                </tr>
                                            <?php endif; ?>
                                        </tbody>
                                    </table>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </main>
        </div>
    </div>
    
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Formatear monto con separadores de miles
        document.getElementById('monto').addEventListener('input', function(e) {
            let value = e.target.value.replace(/\D/g, '');
            e.target.value = value.replace(/\B(?=(\d{3})+(?!\d))/g, '.');
        });
    </script>
</body>
</html>

