<?php
/**
 * Sistema Médico - Ver Facturas
 */
require_once '../includes/config.php';
require_once '../includes/database.php';
require_once '../includes/auth.php';
require_once '../includes/permissions.php';

Permissions::requirePermission('CreateInvoices');

$page_title = 'Facturas';

$database = new Database();
$db = $database->getConnection();

$mensaje = '';
$tipo_mensaje = '';

// Procesar acciones
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['marcar_pagada'])) {
    if (Permissions::isAdmin() || Permissions::isColaborador()) {
        $factura_id = $_POST['factura_id'] ?? 0;
        
        try {
            $query = "UPDATE facturas SET estado = 'pagada', fecha_pago = NOW() WHERE id = :id";
            $stmt = $db->prepare($query);
            $stmt->bindParam(':id', $factura_id);
            $stmt->execute();
            
            $mensaje = 'Factura marcada como pagada';
            $tipo_mensaje = 'success';
        } catch (Exception $e) {
            $mensaje = 'Error: ' . $e->getMessage();
            $tipo_mensaje = 'error';
        }
    }
}

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['anular_factura'])) {
    if (Permissions::isAdmin()) {
        $factura_id = $_POST['factura_id'] ?? 0;
        
        try {
            $db->beginTransaction();
            
            // Devolver stock de productos
            $query = "SELECT fp.producto_id, fp.cantidad 
                      FROM factura_productos fp
                      WHERE fp.factura_id = :factura_id";
            $stmt = $db->prepare($query);
            $stmt->bindParam(':factura_id', $factura_id);
            $stmt->execute();
            $productos_factura = $stmt->fetchAll();
            
            foreach ($productos_factura as $prod) {
                $query = "UPDATE productos SET stock = stock + :cantidad WHERE id = :id";
                $stmt = $db->prepare($query);
                $stmt->bindParam(':cantidad', $prod['cantidad']);
                $stmt->bindParam(':id', $prod['producto_id']);
                $stmt->execute();
            }
            
            // Anular factura
            $query = "UPDATE facturas SET estado = 'anulada' WHERE id = :id";
            $stmt = $db->prepare($query);
            $stmt->bindParam(':id', $factura_id);
            $stmt->execute();
            
            $db->commit();
            
            $mensaje = 'Factura anulada y stock devuelto';
            $tipo_mensaje = 'success';
        } catch (Exception $e) {
            $db->rollBack();
            $mensaje = 'Error: ' . $e->getMessage();
            $tipo_mensaje = 'error';
        }
    }
}

// Filtros
$filtro_estado = $_GET['estado'] ?? 'todas';
$filtro_fecha = $_GET['fecha'] ?? 'todas';

$where_conditions = [];
$params = [];

if ($filtro_estado != 'todas') {
    $where_conditions[] = "f.estado = :estado";
    $params[':estado'] = $filtro_estado;
}

if ($filtro_fecha == 'hoy') {
    $where_conditions[] = "DATE(f.fecha_emision) = CURDATE()";
} elseif ($filtro_fecha == 'semana') {
    $where_conditions[] = "f.fecha_emision >= DATE_SUB(NOW(), INTERVAL 7 DAY)";
} elseif ($filtro_fecha == 'mes') {
    $where_conditions[] = "MONTH(f.fecha_emision) = MONTH(NOW()) AND YEAR(f.fecha_emision) = YEAR(NOW())";
}

$where_clause = !empty($where_conditions) ? 'WHERE ' . implode(' AND ', $where_conditions) : '';

// Obtener facturas
$query = "SELECT f.*, 
          CONCAT(p.nombre, ' ', p.apellido) as paciente_nombre,
          u.nombre_completo as usuario_nombre
          FROM facturas f
          INNER JOIN pacientes p ON f.paciente_id = p.id
          INNER JOIN usuarios u ON f.usuario_id = u.id
          $where_clause
          ORDER BY f.fecha_emision DESC
          LIMIT 100";

$stmt = $db->prepare($query);
foreach ($params as $key => $value) {
    $stmt->bindValue($key, $value);
}
$stmt->execute();
$facturas = $stmt->fetchAll();

include '../includes/header.php';
?>

<div class="card">
    <div class="card-header">
        <h2>Facturas</h2>
    </div>
    <div class="card-body">
        <?php if ($mensaje): ?>
            <div class="alert alert-<?php echo $tipo_mensaje; ?>">
                <?php echo htmlspecialchars($mensaje); ?>
            </div>
        <?php endif; ?>

        <!-- Filtros -->
        <div style="display: flex; gap: 10px; margin-bottom: 20px; flex-wrap: wrap;">
            <a href="facturacion.php" class="btn btn-success">➕ Nueva Factura</a>
            
            <select onchange="window.location.href='?estado=' + this.value + '&fecha=<?php echo $filtro_fecha; ?>'" 
                    style="padding: 10px; border-radius: 5px; border: 1px solid var(--border-color);">
                <option value="todas" <?php echo $filtro_estado == 'todas' ? 'selected' : ''; ?>>Todos los estados</option>
                <option value="pendiente" <?php echo $filtro_estado == 'pendiente' ? 'selected' : ''; ?>>Pendientes</option>
                <option value="pagada" <?php echo $filtro_estado == 'pagada' ? 'selected' : ''; ?>>Pagadas</option>
                <option value="anulada" <?php echo $filtro_estado == 'anulada' ? 'selected' : ''; ?>>Anuladas</option>
            </select>
            
            <select onchange="window.location.href='?estado=<?php echo $filtro_estado; ?>&fecha=' + this.value" 
                    style="padding: 10px; border-radius: 5px; border: 1px solid var(--border-color);">
                <option value="todas" <?php echo $filtro_fecha == 'todas' ? 'selected' : ''; ?>>Todas las fechas</option>
                <option value="hoy" <?php echo $filtro_fecha == 'hoy' ? 'selected' : ''; ?>>Hoy</option>
                <option value="semana" <?php echo $filtro_fecha == 'semana' ? 'selected' : ''; ?>>Última semana</option>
                <option value="mes" <?php echo $filtro_fecha == 'mes' ? 'selected' : ''; ?>>Este mes</option>
            </select>
        </div>

        <!-- Tabla de facturas -->
        <div class="table-responsive">
            <table>
                <thead>
                    <tr>
                        <th>Número</th>
                        <th>Fecha</th>
                        <th>Paciente</th>
                        <th>Total</th>
                        <th>Estado</th>
                        <th>Método Pago</th>
                        <th>Acciones</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($facturas as $factura): ?>
                        <tr>
                            <td><strong><?php echo htmlspecialchars($factura['numero_factura']); ?></strong></td>
                            <td><?php echo date('d/m/Y H:i', strtotime($factura['fecha_emision'])); ?></td>
                            <td><?php echo htmlspecialchars($factura['paciente_nombre']); ?></td>
                            <td><strong>₲ <?php echo number_format($factura['total'], 0, ',', '.'); ?></strong></td>
                            <td>
                                <?php
                                $color = 'gray';
                                if ($factura['estado'] == 'pagada') $color = 'green';
                                elseif ($factura['estado'] == 'pendiente') $color = 'orange';
                                elseif ($factura['estado'] == 'anulada') $color = 'red';
                                ?>
                                <span style="color: <?php echo $color; ?>; font-weight: bold;">
                                    <?php echo strtoupper($factura['estado']); ?>
                                </span>
                            </td>
                            <td><?php echo ucfirst($factura['metodo_pago']); ?></td>
                            <td>
                                <a href="ver_factura.php?id=<?php echo $factura['id']; ?>" class="btn btn-primary" 
                                   style="padding: 6px 12px; font-size: 12px;">
                                    Ver
                                </a>
                                <?php if ($factura['estado'] == 'pendiente'): ?>
                                    <form method="POST" style="display: inline;">
                                        <input type="hidden" name="factura_id" value="<?php echo $factura['id']; ?>">
                                        <button type="submit" name="marcar_pagada" class="btn btn-success" 
                                                style="padding: 6px 12px; font-size: 12px;">
                                            Marcar Pagada
                                        </button>
                                    </form>
                                <?php endif; ?>
                                <?php if (Permissions::isAdmin() && $factura['estado'] != 'anulada'): ?>
                                    <form method="POST" style="display: inline;">
                                        <input type="hidden" name="factura_id" value="<?php echo $factura['id']; ?>">
                                        <button type="submit" name="anular_factura" class="btn btn-danger" 
                                                style="padding: 6px 12px; font-size: 12px;"
                                                onclick="return confirm('¿Anular esta factura? Se devolverá el stock.')">
                                            Anular
                                        </button>
                                    </form>
                                <?php endif; ?>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>

        <?php if (count($facturas) == 0): ?>
            <p style="text-align: center; color: var(--secondary-color); margin: 30px 0;">
                No se encontraron facturas con los filtros seleccionados.
            </p>
        <?php endif; ?>
    </div>
</div>

<?php include '../includes/footer.php'; ?>
