<?php
/**
 * Sistema Médico - Nueva Facturación
 */
require_once '../includes/config.php';
require_once '../includes/database.php';
require_once '../includes/auth.php';
require_once '../includes/permissions.php';

// Verificar permisos
Permissions::requirePermission('CreateInvoices');

$page_title = 'Nueva Facturación';

$database = new Database();
$db = $database->getConnection();

$mensaje = '';
$tipo_mensaje = '';
$paciente = null;

// Buscar paciente
if (isset($_GET['paciente_id'])) {
    $paciente_id = $_GET['paciente_id'];
    
    $query = "SELECT p.*, 
              (SELECT COUNT(*) FROM alertas_medicas WHERE paciente_id = p.id AND activo = 1) as tiene_alertas
              FROM pacientes p 
              WHERE p.id = :id";
    $stmt = $db->prepare($query);
    $stmt->bindParam(':id', $paciente_id);
    $stmt->execute();
    $paciente = $stmt->fetch();
    
    // Obtener alertas del paciente
    if ($paciente && $paciente['tiene_alertas'] > 0) {
        $query = "SELECT * FROM alertas_medicas WHERE paciente_id = :paciente_id AND activo = 1";
        $stmt = $db->prepare($query);
        $stmt->bindParam(':paciente_id', $paciente_id);
        $stmt->execute();
        $alertas = $stmt->fetchAll();
    }
}

// Procesar creación de factura
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['crear_factura'])) {
    $paciente_id = $_POST['paciente_id'] ?? 0;
    $productos_ids = $_POST['producto_id'] ?? [];
    $productos_cantidades = $_POST['producto_cantidad'] ?? [];
    $servicios_ids = $_POST['servicio_id'] ?? [];
    $servicios_cantidades = $_POST['servicio_cantidad'] ?? [];
    $metodo_pago = $_POST['metodo_pago'] ?? 'efectivo';
    $observaciones = trim($_POST['observaciones'] ?? '');
    
    try {
        $db->beginTransaction();
        
        // Generar número de factura
        $query = "SELECT MAX(CAST(SUBSTRING(numero_factura, 5) AS UNSIGNED)) as ultimo FROM facturas WHERE numero_factura LIKE 'FAC-%'";
        $stmt = $db->query($query);
        $result = $stmt->fetch();
        $siguiente_numero = ($result['ultimo'] ?? 0) + 1;
        $numero_factura = 'FAC-' . str_pad($siguiente_numero, 8, '0', STR_PAD_LEFT);
        
        // Calcular totales
        $subtotal = 0;
        $items_factura = [];
        
        // Procesar productos
        foreach ($productos_ids as $index => $producto_id) {
            if (empty($producto_id)) continue;
            
            $cantidad = $productos_cantidades[$index] ?? 1;
            
            $query = "SELECT precio, stock FROM productos WHERE id = :id AND activo = 1";
            $stmt = $db->prepare($query);
            $stmt->bindParam(':id', $producto_id);
            $stmt->execute();
            $producto = $stmt->fetch();
            
            if (!$producto) {
                throw new Exception("Producto no encontrado o inactivo");
            }
            
            if ($producto['stock'] < $cantidad) {
                throw new Exception("Stock insuficiente para el producto");
            }
            
            $precio_unitario = $producto['precio'];
            $subtotal_item = $precio_unitario * $cantidad;
            $subtotal += $subtotal_item;
            
            $items_factura[] = [
                'tipo' => 'producto',
                'id' => $producto_id,
                'cantidad' => $cantidad,
                'precio_unitario' => $precio_unitario,
                'subtotal' => $subtotal_item
            ];
        }
        
        // Procesar servicios
        foreach ($servicios_ids as $index => $servicio_id) {
            if (empty($servicio_id)) continue;
            
            $cantidad = $servicios_cantidades[$index] ?? 1;
            
            $query = "SELECT precio FROM servicios WHERE id = :id AND activo = 1";
            $stmt = $db->prepare($query);
            $stmt->bindParam(':id', $servicio_id);
            $stmt->execute();
            $servicio = $stmt->fetch();
            
            if (!$servicio) {
                throw new Exception("Servicio no encontrado o inactivo");
            }
            
            $precio_unitario = $servicio['precio'];
            $subtotal_item = $precio_unitario * $cantidad;
            $subtotal += $subtotal_item;
            
            $items_factura[] = [
                'tipo' => 'servicio',
                'id' => $servicio_id,
                'cantidad' => $cantidad,
                'precio_unitario' => $precio_unitario,
                'subtotal' => $subtotal_item
            ];
        }
        
        if (empty($items_factura)) {
            throw new Exception("Debe agregar al menos un producto o servicio");
        }
        
        $total = $subtotal;
        $current_user = Auth::getCurrentUser();
        
        // Crear factura
        $query = "INSERT INTO facturas (numero_factura, paciente_id, usuario_id, subtotal, total, metodo_pago, observaciones, estado) 
                  VALUES (:numero_factura, :paciente_id, :usuario_id, :subtotal, :total, :metodo_pago, :observaciones, 'pendiente')";
        
        $stmt = $db->prepare($query);
        $stmt->bindParam(':numero_factura', $numero_factura);
        $stmt->bindParam(':paciente_id', $paciente_id);
        $stmt->bindParam(':usuario_id', $current_user['id']);
        $stmt->bindParam(':subtotal', $subtotal);
        $stmt->bindParam(':total', $total);
        $stmt->bindParam(':metodo_pago', $metodo_pago);
        $stmt->bindParam(':observaciones', $observaciones);
        $stmt->execute();
        
        $factura_id = $db->lastInsertId();
        
        // Insertar items de la factura
        foreach ($items_factura as $item) {
            if ($item['tipo'] == 'producto') {
                $query = "INSERT INTO factura_productos (factura_id, producto_id, cantidad, precio_unitario, subtotal) 
                          VALUES (:factura_id, :producto_id, :cantidad, :precio_unitario, :subtotal)";
                $stmt = $db->prepare($query);
                $stmt->bindParam(':factura_id', $factura_id);
                $stmt->bindParam(':producto_id', $item['id']);
                $stmt->bindParam(':cantidad', $item['cantidad']);
                $stmt->bindParam(':precio_unitario', $item['precio_unitario']);
                $stmt->bindParam(':subtotal', $item['subtotal']);
                $stmt->execute();
                
                // Actualizar stock
                $query = "UPDATE productos SET stock = stock - :cantidad WHERE id = :id";
                $stmt = $db->prepare($query);
                $stmt->bindParam(':cantidad', $item['cantidad']);
                $stmt->bindParam(':id', $item['id']);
                $stmt->execute();
            } else {
                $query = "INSERT INTO factura_servicios (factura_id, servicio_id, cantidad, precio_unitario, subtotal) 
                          VALUES (:factura_id, :servicio_id, :cantidad, :precio_unitario, :subtotal)";
                $stmt = $db->prepare($query);
                $stmt->bindParam(':factura_id', $factura_id);
                $stmt->bindParam(':servicio_id', $item['id']);
                $stmt->bindParam(':cantidad', $item['cantidad']);
                $stmt->bindParam(':precio_unitario', $item['precio_unitario']);
                $stmt->bindParam(':subtotal', $item['subtotal']);
                $stmt->execute();
            }
        }
        
        $db->commit();
        
        header("Location: ver_factura.php?id=$factura_id&success=1");
        exit();
        
    } catch (Exception $e) {
        $db->rollBack();
        $mensaje = 'Error al crear factura: ' . $e->getMessage();
        $tipo_mensaje = 'error';
    }
}

// Obtener productos activos
$query = "SELECT * FROM productos WHERE activo = 1 ORDER BY nombre";
$stmt = $db->query($query);
$productos = $stmt->fetchAll();

// Obtener servicios activos
$query = "SELECT * FROM servicios WHERE activo = 1 ORDER BY nombre";
$stmt = $db->query($query);
$servicios = $stmt->fetchAll();

include '../includes/header.php';
?>

<div class="card">
    <div class="card-header">
        <h2>Nueva Facturación</h2>
    </div>
    <div class="card-body">
        <?php if ($mensaje): ?>
            <div class="alert alert-<?php echo $tipo_mensaje; ?>">
                <?php echo htmlspecialchars($mensaje); ?>
            </div>
        <?php endif; ?>

        <?php if (!$paciente): ?>
            <!-- Búsqueda de paciente -->
            <div class="search-box">
                <form method="GET" style="display: flex; gap: 10px; flex-wrap: wrap;">
                    <input type="text" name="buscar" placeholder="Buscar paciente por cédula, nombre o apellido..." 
                           value="<?php echo htmlspecialchars($_GET['buscar'] ?? ''); ?>" style="flex: 1; min-width: 250px;">
                    <button type="submit" class="btn btn-primary">🔍 Buscar</button>
                </form>
            </div>

            <?php if (isset($_GET['buscar'])): ?>
                <?php
                $termino = '%' . $_GET['buscar'] . '%';
                $query = "SELECT * FROM pacientes 
                          WHERE nombre LIKE :termino OR apellido LIKE :termino OR cedula LIKE :termino
                          ORDER BY apellido, nombre LIMIT 20";
                $stmt = $db->prepare($query);
                $stmt->bindParam(':termino', $termino);
                $stmt->execute();
                $resultados = $stmt->fetchAll();
                ?>

                <?php if (count($resultados) > 0): ?>
                    <h3 style="margin-top: 30px; color: var(--medical-blue);">Resultados de búsqueda</h3>
                    <div class="table-responsive">
                        <table>
                            <thead>
                                <tr>
                                    <th>Cédula</th>
                                    <th>Nombre Completo</th>
                                    <th>Teléfono</th>
                                    <th>Acción</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($resultados as $resultado): ?>
                                    <tr>
                                        <td><?php echo htmlspecialchars($resultado['cedula']); ?></td>
                                        <td><?php echo htmlspecialchars($resultado['nombre'] . ' ' . $resultado['apellido']); ?></td>
                                        <td><?php echo htmlspecialchars($resultado['telefono'] ?? '-'); ?></td>
                                        <td>
                                            <a href="?paciente_id=<?php echo $resultado['id']; ?>" class="btn btn-primary">
                                                Seleccionar
                                            </a>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                <?php else: ?>
                    <p style="margin-top: 20px; color: var(--warning-color);">No se encontraron pacientes con ese criterio.</p>
                <?php endif; ?>
            <?php endif; ?>

        <?php else: ?>
            <!-- Formulario de facturación -->
            
            <!-- Información del paciente -->
            <div class="patient-info">
                <h3>Paciente Seleccionado</h3>
                <div class="patient-details">
                    <div class="detail-item">
                        <span class="detail-label">Nombre Completo</span>
                        <span class="detail-value"><?php echo htmlspecialchars($paciente['nombre'] . ' ' . $paciente['apellido']); ?></span>
                    </div>
                    <div class="detail-item">
                        <span class="detail-label">Cédula</span>
                        <span class="detail-value"><?php echo htmlspecialchars($paciente['cedula']); ?></span>
                    </div>
                    <div class="detail-item">
                        <span class="detail-label">Teléfono</span>
                        <span class="detail-value"><?php echo htmlspecialchars($paciente['telefono'] ?? '-'); ?></span>
                    </div>
                </div>
            </div>

            <!-- Alertas médicas -->
            <?php if ($paciente['tiene_alertas'] > 0): ?>
                <div class="alert-emergency">
                    <h4>⚠️ ALERTAS MÉDICAS</h4>
                    <?php foreach ($alertas as $alerta): ?>
                        <div style="margin: 10px 0;">
                            <strong><?php echo htmlspecialchars($alerta['tipo']); ?>:</strong>
                            <?php echo htmlspecialchars($alerta['descripcion']); ?>
                        </div>
                    <?php endforeach; ?>
                </div>
            <?php endif; ?>

            <!-- Formulario de facturación -->
            <form method="POST" id="formFactura">
                <input type="hidden" name="paciente_id" value="<?php echo $paciente['id']; ?>">

                <h3 style="margin-top: 30px; color: var(--medical-blue);">Productos</h3>
                <div id="productosContainer">
                    <div class="producto-item" style="display: grid; grid-template-columns: 2fr 1fr auto; gap: 15px; margin-bottom: 15px; align-items: end;">
                        <div class="form-group" style="margin: 0;">
                            <label>Producto</label>
                            <select name="producto_id[]" onchange="calcularTotal()">
                                <option value="">Seleccionar producto...</option>
                                <?php foreach ($productos as $prod): ?>
                                    <option value="<?php echo $prod['id']; ?>" data-precio="<?php echo $prod['precio']; ?>">
                                        <?php echo htmlspecialchars($prod['nombre']); ?> - ₲ <?php echo number_format($prod['precio'], 0, ',', '.'); ?> 
                                        (Stock: <?php echo $prod['stock']; ?>)
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <div class="form-group" style="margin: 0;">
                            <label>Cantidad</label>
                            <input type="number" name="producto_cantidad[]" value="1" min="1" onchange="calcularTotal()">
                        </div>
                        <button type="button" onclick="eliminarProducto(this)" class="btn btn-danger" style="padding: 10px 15px;">✗</button>
                    </div>
                </div>
                <button type="button" onclick="agregarProducto()" class="btn btn-secondary" style="margin-bottom: 20px;">+ Agregar Producto</button>

                <h3 style="margin-top: 30px; color: var(--medical-blue);">Servicios</h3>
                <div id="serviciosContainer">
                    <div class="servicio-item" style="display: grid; grid-template-columns: 2fr 1fr auto; gap: 15px; margin-bottom: 15px; align-items: end;">
                        <div class="form-group" style="margin: 0;">
                            <label>Servicio</label>
                            <select name="servicio_id[]" onchange="calcularTotal()">
                                <option value="">Seleccionar servicio...</option>
                                <?php foreach ($servicios as $serv): ?>
                                    <option value="<?php echo $serv['id']; ?>" data-precio="<?php echo $serv['precio']; ?>">
                                        <?php echo htmlspecialchars($serv['nombre']); ?> - ₲ <?php echo number_format($serv['precio'], 0, ',', '.'); ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <div class="form-group" style="margin: 0;">
                            <label>Cantidad</label>
                            <input type="number" name="servicio_cantidad[]" value="1" min="1" onchange="calcularTotal()">
                        </div>
                        <button type="button" onclick="eliminarServicio(this)" class="btn btn-danger" style="padding: 10px 15px;">✗</button>
                    </div>
                </div>
                <button type="button" onclick="agregarServicio()" class="btn btn-secondary" style="margin-bottom: 20px;">+ Agregar Servicio</button>

                <h3 style="margin-top: 30px; color: var(--medical-blue);">Detalles de Pago</h3>
                
                <div class="form-group">
                    <label>Método de Pago</label>
                    <select name="metodo_pago" required>
                        <option value="efectivo">Efectivo</option>
                        <option value="tarjeta">Tarjeta</option>
                        <option value="transferencia">Transferencia</option>
                        <option value="cheque">Cheque</option>
                        <option value="otro">Otro</option>
                    </select>
                </div>

                <div class="form-group">
                    <label>Observaciones</label>
                    <textarea name="observaciones" rows="3"></textarea>
                </div>

                <!-- Total -->
                <div style="background: var(--medical-light-blue); padding: 20px; border-radius: 8px; margin-top: 20px;">
                    <h2 style="color: var(--medical-blue); text-align: right; margin: 0;">
                        Total: ₲ <span id="totalFactura">0</span>
                    </h2>
                </div>

                <div style="display: flex; gap: 10px; margin-top: 20px;">
                    <button type="submit" name="crear_factura" class="btn btn-success">💾 Crear Factura</button>
                    <a href="facturacion.php" class="btn btn-secondary">Cancelar</a>
                </div>
            </form>
        <?php endif; ?>
    </div>
</div>

<script>
function agregarProducto() {
    const container = document.getElementById('productosContainer');
    const item = container.querySelector('.producto-item').cloneNode(true);
    item.querySelector('select').value = '';
    item.querySelector('input').value = '1';
    container.appendChild(item);
}

function eliminarProducto(btn) {
    const container = document.getElementById('productosContainer');
    if (container.querySelectorAll('.producto-item').length > 1) {
        btn.closest('.producto-item').remove();
        calcularTotal();
    }
}

function agregarServicio() {
    const container = document.getElementById('serviciosContainer');
    const item = container.querySelector('.servicio-item').cloneNode(true);
    item.querySelector('select').value = '';
    item.querySelector('input').value = '1';
    container.appendChild(item);
}

function eliminarServicio(btn) {
    const container = document.getElementById('serviciosContainer');
    if (container.querySelectorAll('.servicio-item').length > 1) {
        btn.closest('.servicio-item').remove();
        calcularTotal();
    }
}

function calcularTotal() {
    let total = 0;
    
    // Calcular productos
    document.querySelectorAll('.producto-item').forEach(item => {
        const select = item.querySelector('select[name="producto_id[]"]');
        const cantidad = parseInt(item.querySelector('input[name="producto_cantidad[]"]').value) || 0;
        const option = select.options[select.selectedIndex];
        const precio = parseFloat(option.getAttribute('data-precio')) || 0;
        total += precio * cantidad;
    });
    
    // Calcular servicios
    document.querySelectorAll('.servicio-item').forEach(item => {
        const select = item.querySelector('select[name="servicio_id[]"]');
        const cantidad = parseInt(item.querySelector('input[name="servicio_cantidad[]"]').value) || 0;
        const option = select.options[select.selectedIndex];
        const precio = parseFloat(option.getAttribute('data-precio')) || 0;
        total += precio * cantidad;
    });
    
    document.getElementById('totalFactura').textContent = total.toLocaleString('es-PY');
}

// Calcular total inicial
calcularTotal();
</script>

<?php include '../includes/footer.php'; ?>
