<?php
/**
 * Sistema Médico - Gestión de Productos (Solo Administradores)
 */
require_once '../../includes/config.php';
require_once '../../includes/database.php';
require_once '../../includes/auth.php';
require_once '../../includes/permissions.php';

// Verificar permisos de administrador
Permissions::requirePermission('ManageProductsServices');

$page_title = 'Gestión de Productos';

$database = new Database();
$db = $database->getConnection();

$mensaje = '';
$tipo_mensaje = '';

// Procesar acciones
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['crear_producto'])) {
        // Crear producto
        $categoria_id = $_POST['categoria_id'] ?? null;
        $codigo = trim($_POST['codigo'] ?? '');
        $nombre = trim($_POST['nombre'] ?? '');
        $descripcion = trim($_POST['descripcion'] ?? '');
        $precio = $_POST['precio'] ?? 0;
        $stock = $_POST['stock'] ?? 0;
        $stock_minimo = $_POST['stock_minimo'] ?? 0;

        try {
            $query = "INSERT INTO productos (categoria_id, codigo, nombre, descripcion, precio, stock, stock_minimo) 
                      VALUES (:categoria_id, :codigo, :nombre, :descripcion, :precio, :stock, :stock_minimo)";
            
            $stmt = $db->prepare($query);
            $stmt->bindParam(':categoria_id', $categoria_id);
            $stmt->bindParam(':codigo', $codigo);
            $stmt->bindParam(':nombre', $nombre);
            $stmt->bindParam(':descripcion', $descripcion);
            $stmt->bindParam(':precio', $precio);
            $stmt->bindParam(':stock', $stock);
            $stmt->bindParam(':stock_minimo', $stock_minimo);
            $stmt->execute();

            $mensaje = 'Producto creado exitosamente';
            $tipo_mensaje = 'success';
        } catch (Exception $e) {
            $mensaje = 'Error al crear producto: ' . $e->getMessage();
            $tipo_mensaje = 'error';
        }
    }
    elseif (isset($_POST['editar_producto'])) {
        // Editar producto
        $id = $_POST['producto_id'] ?? 0;
        $categoria_id = $_POST['categoria_id'] ?? null;
        $codigo = trim($_POST['codigo'] ?? '');
        $nombre = trim($_POST['nombre'] ?? '');
        $descripcion = trim($_POST['descripcion'] ?? '');
        $precio_nuevo = $_POST['precio'] ?? 0;
        $stock = $_POST['stock'] ?? 0;
        $stock_minimo = $_POST['stock_minimo'] ?? 0;

        try {
            // Obtener precio anterior para historial
            $query = "SELECT precio FROM productos WHERE id = :id";
            $stmt = $db->prepare($query);
            $stmt->bindParam(':id', $id);
            $stmt->execute();
            $producto_actual = $stmt->fetch();

            $db->beginTransaction();

            // Actualizar producto
            $query = "UPDATE productos 
                      SET categoria_id = :categoria_id, codigo = :codigo, nombre = :nombre, 
                          descripcion = :descripcion, precio = :precio, stock = :stock, stock_minimo = :stock_minimo
                      WHERE id = :id";
            
            $stmt = $db->prepare($query);
            $stmt->bindParam(':categoria_id', $categoria_id);
            $stmt->bindParam(':codigo', $codigo);
            $stmt->bindParam(':nombre', $nombre);
            $stmt->bindParam(':descripcion', $descripcion);
            $stmt->bindParam(':precio', $precio_nuevo);
            $stmt->bindParam(':stock', $stock);
            $stmt->bindParam(':stock_minimo', $stock_minimo);
            $stmt->bindParam(':id', $id);
            $stmt->execute();

            // Registrar cambio de precio si cambió
            if ($producto_actual && $producto_actual['precio'] != $precio_nuevo) {
                $current_user = Auth::getCurrentUser();
                $query = "INSERT INTO historial_precios (tipo, item_id, precio_anterior, precio_nuevo, usuario_id) 
                          VALUES ('producto', :item_id, :precio_anterior, :precio_nuevo, :usuario_id)";
                $stmt = $db->prepare($query);
                $stmt->bindParam(':item_id', $id);
                $stmt->bindParam(':precio_anterior', $producto_actual['precio']);
                $stmt->bindParam(':precio_nuevo', $precio_nuevo);
                $stmt->bindParam(':usuario_id', $current_user['id']);
                $stmt->execute();
            }

            $db->commit();

            $mensaje = 'Producto actualizado exitosamente';
            $tipo_mensaje = 'success';
        } catch (Exception $e) {
            $db->rollBack();
            $mensaje = 'Error al actualizar producto: ' . $e->getMessage();
            $tipo_mensaje = 'error';
        }
    }
    elseif (isset($_POST['eliminar_producto'])) {
        // Desactivar producto
        $id = $_POST['producto_id'] ?? 0;

        try {
            $query = "UPDATE productos SET activo = 0 WHERE id = :id";
            $stmt = $db->prepare($query);
            $stmt->bindParam(':id', $id);
            $stmt->execute();

            $mensaje = 'Producto desactivado exitosamente';
            $tipo_mensaje = 'success';
        } catch (Exception $e) {
            $mensaje = 'Error al desactivar producto: ' . $e->getMessage();
            $tipo_mensaje = 'error';
        }
    }
    elseif (isset($_POST['activar_producto'])) {
        // Activar producto
        $id = $_POST['producto_id'] ?? 0;

        try {
            $query = "UPDATE productos SET activo = 1 WHERE id = :id";
            $stmt = $db->prepare($query);
            $stmt->bindParam(':id', $id);
            $stmt->execute();

            $mensaje = 'Producto activado exitosamente';
            $tipo_mensaje = 'success';
        } catch (Exception $e) {
            $mensaje = 'Error al activar producto: ' . $e->getMessage();
            $tipo_mensaje = 'error';
        }
    }
}

// Obtener categorías de productos
$query = "SELECT * FROM categorias WHERE tipo = 'producto' AND activo = 1 ORDER BY nombre";
$stmt = $db->query($query);
$categorias = $stmt->fetchAll();

// Obtener productos
$filtro = $_GET['filtro'] ?? 'activos';
$where_clause = $filtro == 'todos' ? '' : 'WHERE p.activo = 1';

$query = "SELECT p.*, c.nombre as categoria_nombre
          FROM productos p
          LEFT JOIN categorias c ON p.categoria_id = c.id
          $where_clause
          ORDER BY p.nombre";

$stmt = $db->query($query);
$productos = $stmt->fetchAll();

include '../../includes/header.php';
?>

<div class="card">
    <div class="card-header">
        <h2>Gestión de Productos</h2>
    </div>
    <div class="card-body">
        <?php if ($mensaje): ?>
            <div class="alert alert-<?php echo $tipo_mensaje; ?>">
                <?php echo htmlspecialchars($mensaje); ?>
            </div>
        <?php endif; ?>

        <!-- Botones de acción -->
        <div style="display: flex; gap: 10px; margin-bottom: 20px; flex-wrap: wrap;">
            <button onclick="document.getElementById('modalCrear').style.display='block'" class="btn btn-success">
                ➕ Nuevo Producto
            </button>
            <a href="?filtro=activos" class="btn <?php echo $filtro == 'activos' ? 'btn-primary' : 'btn-secondary'; ?>">
                Activos
            </a>
            <a href="?filtro=todos" class="btn <?php echo $filtro == 'todos' ? 'btn-primary' : 'btn-secondary'; ?>">
                Todos
            </a>
        </div>

        <!-- Tabla de productos -->
        <div class="table-responsive">
            <table>
                <thead>
                    <tr>
                        <th>Código</th>
                        <th>Nombre</th>
                        <th>Categoría</th>
                        <th>Precio</th>
                        <th>Stock</th>
                        <th>Stock Mín.</th>
                        <th>Estado</th>
                        <th>Acciones</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($productos as $producto): ?>
                        <tr style="<?php echo !$producto['activo'] ? 'opacity: 0.5;' : ''; ?>">
                            <td><?php echo htmlspecialchars($producto['codigo']); ?></td>
                            <td><?php echo htmlspecialchars($producto['nombre']); ?></td>
                            <td><?php echo htmlspecialchars($producto['categoria_nombre'] ?? 'Sin categoría'); ?></td>
                            <td>₲ <?php echo number_format($producto['precio'], 0, ',', '.'); ?></td>
                            <td style="<?php echo $producto['stock'] <= $producto['stock_minimo'] ? 'color: red; font-weight: bold;' : ''; ?>">
                                <?php echo $producto['stock']; ?>
                            </td>
                            <td><?php echo $producto['stock_minimo']; ?></td>
                            <td>
                                <?php if ($producto['activo']): ?>
                                    <span style="color: green;">✓ Activo</span>
                                <?php else: ?>
                                    <span style="color: red;">✗ Inactivo</span>
                                <?php endif; ?>
                            </td>
                            <td>
                                <button onclick="editarProducto(<?php echo htmlspecialchars(json_encode($producto)); ?>)" 
                                        class="btn btn-primary" style="padding: 6px 12px; font-size: 12px;">
                                    Editar
                                </button>
                                <?php if ($producto['activo']): ?>
                                    <form method="POST" style="display: inline;">
                                        <input type="hidden" name="producto_id" value="<?php echo $producto['id']; ?>">
                                        <button type="submit" name="eliminar_producto" class="btn btn-danger" 
                                                style="padding: 6px 12px; font-size: 12px;"
                                                onclick="return confirm('¿Desactivar este producto?')">
                                            Desactivar
                                        </button>
                                    </form>
                                <?php else: ?>
                                    <form method="POST" style="display: inline;">
                                        <input type="hidden" name="producto_id" value="<?php echo $producto['id']; ?>">
                                        <button type="submit" name="activar_producto" class="btn btn-success" 
                                                style="padding: 6px 12px; font-size: 12px;">
                                            Activar
                                        </button>
                                    </form>
                                <?php endif; ?>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
    </div>
</div>

<!-- Modal Crear Producto -->
<div id="modalCrear" style="display: none; position: fixed; z-index: 1000; left: 0; top: 0; width: 100%; height: 100%; background-color: rgba(0,0,0,0.5); overflow: auto;">
    <div style="background-color: white; margin: 5% auto; padding: 30px; width: 90%; max-width: 600px; border-radius: 10px;">
        <h2 style="color: var(--medical-blue); margin-bottom: 20px;">Nuevo Producto</h2>
        <form method="POST">
            <div class="form-group">
                <label>Categoría</label>
                <select name="categoria_id">
                    <option value="">Sin categoría</option>
                    <?php foreach ($categorias as $cat): ?>
                        <option value="<?php echo $cat['id']; ?>"><?php echo htmlspecialchars($cat['nombre']); ?></option>
                    <?php endforeach; ?>
                </select>
            </div>
            <div class="form-group">
                <label>Código *</label>
                <input type="text" name="codigo" required>
            </div>
            <div class="form-group">
                <label>Nombre *</label>
                <input type="text" name="nombre" required>
            </div>
            <div class="form-group">
                <label>Descripción</label>
                <textarea name="descripcion" rows="3"></textarea>
            </div>
            <div class="form-group">
                <label>Precio (₲) *</label>
                <input type="number" name="precio" step="0.01" required>
            </div>
            <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 15px;">
                <div class="form-group">
                    <label>Stock Inicial</label>
                    <input type="number" name="stock" value="0">
                </div>
                <div class="form-group">
                    <label>Stock Mínimo</label>
                    <input type="number" name="stock_minimo" value="0">
                </div>
            </div>
            <div style="display: flex; gap: 10px; margin-top: 20px;">
                <button type="submit" name="crear_producto" class="btn btn-success">Crear Producto</button>
                <button type="button" onclick="document.getElementById('modalCrear').style.display='none'" class="btn btn-secondary">Cancelar</button>
            </div>
        </form>
    </div>
</div>

<!-- Modal Editar Producto -->
<div id="modalEditar" style="display: none; position: fixed; z-index: 1000; left: 0; top: 0; width: 100%; height: 100%; background-color: rgba(0,0,0,0.5); overflow: auto;">
    <div style="background-color: white; margin: 5% auto; padding: 30px; width: 90%; max-width: 600px; border-radius: 10px;">
        <h2 style="color: var(--medical-blue); margin-bottom: 20px;">Editar Producto</h2>
        <form method="POST" id="formEditar">
            <input type="hidden" name="producto_id" id="edit_id">
            <div class="form-group">
                <label>Categoría</label>
                <select name="categoria_id" id="edit_categoria_id">
                    <option value="">Sin categoría</option>
                    <?php foreach ($categorias as $cat): ?>
                        <option value="<?php echo $cat['id']; ?>"><?php echo htmlspecialchars($cat['nombre']); ?></option>
                    <?php endforeach; ?>
                </select>
            </div>
            <div class="form-group">
                <label>Código *</label>
                <input type="text" name="codigo" id="edit_codigo" required>
            </div>
            <div class="form-group">
                <label>Nombre *</label>
                <input type="text" name="nombre" id="edit_nombre" required>
            </div>
            <div class="form-group">
                <label>Descripción</label>
                <textarea name="descripcion" id="edit_descripcion" rows="3"></textarea>
            </div>
            <div class="form-group">
                <label>Precio (₲) *</label>
                <input type="number" name="precio" id="edit_precio" step="0.01" required>
            </div>
            <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 15px;">
                <div class="form-group">
                    <label>Stock</label>
                    <input type="number" name="stock" id="edit_stock">
                </div>
                <div class="form-group">
                    <label>Stock Mínimo</label>
                    <input type="number" name="stock_minimo" id="edit_stock_minimo">
                </div>
            </div>
            <div style="display: flex; gap: 10px; margin-top: 20px;">
                <button type="submit" name="editar_producto" class="btn btn-success">Guardar Cambios</button>
                <button type="button" onclick="document.getElementById('modalEditar').style.display='none'" class="btn btn-secondary">Cancelar</button>
            </div>
        </form>
    </div>
</div>

<script>
function editarProducto(producto) {
    document.getElementById('edit_id').value = producto.id;
    document.getElementById('edit_categoria_id').value = producto.categoria_id || '';
    document.getElementById('edit_codigo').value = producto.codigo;
    document.getElementById('edit_nombre').value = producto.nombre;
    document.getElementById('edit_descripcion').value = producto.descripcion || '';
    document.getElementById('edit_precio').value = producto.precio;
    document.getElementById('edit_stock').value = producto.stock;
    document.getElementById('edit_stock_minimo').value = producto.stock_minimo;
    document.getElementById('modalEditar').style.display = 'block';
}
</script>

<?php include '../../includes/footer.php'; ?>
