<?php
/**
 * Sistema Médico - Gestión de Citas
 */
require_once '../includes/config.php';
require_once '../includes/database.php';
require_once '../includes/auth.php';

$page_title = 'Gestión de Citas';

$database = new Database();
$db = $database->getConnection();

// Obtener todas las citas
$filtro = $_GET['filtro'] ?? 'todas';

$where_clause = '';
switch ($filtro) {
    case 'pendientes':
        $where_clause = "WHERE c.estado = 'pendiente' AND c.fecha_cita >= NOW()";
        break;
    case 'hoy':
        $where_clause = "WHERE DATE(c.fecha_cita) = CURDATE()";
        break;
    case 'proximas':
        $where_clause = "WHERE c.fecha_cita >= NOW() AND c.estado != 'cancelada'";
        break;
    case 'completadas':
        $where_clause = "WHERE c.estado = 'completada'";
        break;
    default:
        $where_clause = "WHERE c.fecha_cita >= DATE_SUB(NOW(), INTERVAL 30 DAY)";
}

$query = "SELECT c.*, p.nombre, p.apellido, p.cedula, u.nombre_completo as medico
          FROM citas c
          INNER JOIN pacientes p ON c.paciente_id = p.id
          INNER JOIN usuarios u ON c.medico_id = u.id
          $where_clause
          ORDER BY c.fecha_cita ASC";

$stmt = $db->query($query);
$citas = $stmt->fetchAll();

include '../includes/header.php';
?>

<div class="card">
    <div class="card-header">
        <h2>Gestión de Citas Médicas</h2>
    </div>
    <div class="card-body">
        <!-- Filtros -->
        <div style="display: flex; gap: 10px; margin-bottom: 25px; flex-wrap: wrap;">
            <a href="?filtro=todas" class="btn <?php echo $filtro == 'todas' ? 'btn-primary' : 'btn-secondary'; ?>">
                Todas
            </a>
            <a href="?filtro=hoy" class="btn <?php echo $filtro == 'hoy' ? 'btn-primary' : 'btn-secondary'; ?>">
                Hoy
            </a>
            <a href="?filtro=proximas" class="btn <?php echo $filtro == 'proximas' ? 'btn-primary' : 'btn-secondary'; ?>">
                Próximas
            </a>
            <a href="?filtro=pendientes" class="btn <?php echo $filtro == 'pendientes' ? 'btn-primary' : 'btn-secondary'; ?>">
                Pendientes
            </a>
            <a href="?filtro=completadas" class="btn <?php echo $filtro == 'completadas' ? 'btn-primary' : 'btn-secondary'; ?>">
                Completadas
            </a>
        </div>

        <!-- Tabla de citas -->
        <?php if (count($citas) > 0): ?>
            <div class="table-responsive">
                <table>
                    <thead>
                        <tr>
                            <th>Fecha y Hora</th>
                            <th>Paciente</th>
                            <th>Cédula</th>
                            <th>Médico</th>
                            <th>Tipo</th>
                            <th>Motivo</th>
                            <th>Estado</th>
                            <th>Acciones</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($citas as $cita): ?>
                            <tr>
                                <td><?php echo date('d/m/Y H:i', strtotime($cita['fecha_cita'])); ?></td>
                                <td><?php echo htmlspecialchars($cita['nombre'] . ' ' . $cita['apellido']); ?></td>
                                <td><?php echo htmlspecialchars($cita['cedula']); ?></td>
                                <td><?php echo htmlspecialchars($cita['medico']); ?></td>
                                <td>
                                    <?php
                                    $tipo_class = '';
                                    if ($cita['tipo'] == 'urgencia') {
                                        $tipo_class = 'background: var(--danger-color); color: white; padding: 5px 10px; border-radius: 5px;';
                                    } elseif ($cita['tipo'] == 'seguimiento') {
                                        $tipo_class = 'background: var(--info-color); color: white; padding: 5px 10px; border-radius: 5px;';
                                    } else {
                                        $tipo_class = 'background: var(--success-color); color: white; padding: 5px 10px; border-radius: 5px;';
                                    }
                                    echo '<span style="' . $tipo_class . '">' . ucfirst($cita['tipo']) . '</span>';
                                    ?>
                                </td>
                                <td><?php echo htmlspecialchars(substr($cita['motivo'], 0, 50)) . (strlen($cita['motivo']) > 50 ? '...' : ''); ?></td>
                                <td>
                                    <?php
                                    $estado_class = '';
                                    switch ($cita['estado']) {
                                        case 'pendiente':
                                            $estado_class = 'color: var(--warning-color);';
                                            break;
                                        case 'confirmada':
                                            $estado_class = 'color: var(--info-color);';
                                            break;
                                        case 'completada':
                                            $estado_class = 'color: var(--success-color);';
                                            break;
                                        case 'cancelada':
                                            $estado_class = 'color: var(--danger-color);';
                                            break;
                                    }
                                    echo '<span style="' . $estado_class . ' font-weight: bold;">' . ucfirst($cita['estado']) . '</span>';
                                    ?>
                                </td>
                                <td>
                                    <a href="ficha_paciente.php?id=<?php echo $cita['paciente_id']; ?>" class="btn btn-primary" style="padding: 6px 12px; font-size: 12px;">
                                        Ver Paciente
                                    </a>
                                    <?php if ($cita['estado'] == 'pendiente' || $cita['estado'] == 'confirmada'): ?>
                                        <a href="actualizar_cita.php?id=<?php echo $cita['id']; ?>" class="btn btn-secondary" style="padding: 6px 12px; font-size: 12px;">
                                            Actualizar
                                        </a>
                                    <?php endif; ?>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        <?php else: ?>
            <div class="alert alert-info">
                No hay citas registradas con el filtro seleccionado.
            </div>
        <?php endif; ?>

        <div style="margin-top: 30px;">
            <a href="buscar_paciente.php" class="btn btn-success">➕ Agendar Nueva Cita</a>
        </div>
    </div>
</div>

<?php include '../includes/footer.php'; ?>

