<?php
/**
 * Gestión de Ventas - Administrador
 * Sistema de Gestión de Agua Mineral
 */

require_once '../../includes/init.php';

// Verificar que sea administrador
checkPermission('administrador');

$pageTitle = 'Gestión de Ventas - Administrador';

$db = Database::getInstance();

// Filtros
$fechaInicio = $_GET['fecha_inicio'] ?? date('Y-m-01'); // Primer día del mes actual
$fechaFin = $_GET['fecha_fin'] ?? date('Y-m-t'); // Último día del mes actual
$vendedorId = (int)($_GET['vendedor_id'] ?? 0);
$clienteId = (int)($_GET['cliente_id'] ?? 0);

// Procesar acciones
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    
    if ($action === 'delete_venta') {
        $ventaId = (int)$_POST['venta_id'];
        
        // Obtener información de la venta antes de eliminar
        $venta = $db->select("
            SELECT v.*, c.nombre as cliente_nombre 
            FROM ventas v 
            INNER JOIN clientes c ON v.cliente_id = c.id 
            WHERE v.id = ?
        ", [$ventaId]);
        
        if (!empty($venta)) {
            if ($db->execute("DELETE FROM ventas WHERE id = ?", [$ventaId])) {
                logActivity($_SESSION['user_id'], 'delete_sale', 
                    "Venta eliminada: Cliente {$venta[0]['cliente_nombre']}, Fecha {$venta[0]['fecha']}");
                setFlashMessage('Venta eliminada exitosamente.', 'success');
            } else {
                setFlashMessage('Error al eliminar la venta.', 'error');
            }
        } else {
            setFlashMessage('Venta no encontrada.', 'error');
        }
        
        redirect('ventas.php');
    }
}

// Construir query con filtros
$whereConditions = ["1=1"];
$params = [];

if ($fechaInicio) {
    $whereConditions[] = "v.fecha >= ?";
    $params[] = $fechaInicio;
}

if ($fechaFin) {
    $whereConditions[] = "v.fecha <= ?";
    $params[] = $fechaFin;
}

if ($vendedorId) {
    $whereConditions[] = "c.vendedor_id = ?";
    $params[] = $vendedorId;
}

if ($clienteId) {
    $whereConditions[] = "v.cliente_id = ?";
    $params[] = $clienteId;
}

$whereClause = implode(" AND ", $whereConditions);

// Obtener ventas con información completa
$ventas = $db->select("
    SELECT 
        v.id,
        v.fecha,
        v.cantidad_bidones,
        v.precio_unitario,
        v.total,
        v.observaciones,
        c.nombre as cliente_nombre,
        c.direccion as cliente_direccion,
        c.telefono as cliente_telefono,
        u.nombre as vendedor_nombre,
        z.nombre as zona_nombre
    FROM ventas v
    INNER JOIN clientes c ON v.cliente_id = c.id
    INNER JOIN usuarios u ON c.vendedor_id = u.id
    LEFT JOIN zonas z ON u.zona_id = z.id
    WHERE $whereClause
    ORDER BY v.fecha DESC, v.id DESC
", $params);

// Estadísticas del período
$estadisticas = $db->select("
    SELECT 
        COUNT(*) as total_ventas,
        SUM(cantidad_bidones) as total_bidones,
        SUM(total) as total_ingresos,
        AVG(total) as promedio_venta,
        COUNT(DISTINCT cliente_id) as clientes_unicos,
        COUNT(DISTINCT c.vendedor_id) as vendedores_activos
    FROM ventas v
    INNER JOIN clientes c ON v.cliente_id = c.id
    WHERE $whereClause
", $params)[0];

// Obtener vendedores para filtro
$vendedores = $db->select("
    SELECT u.id, u.nombre 
    FROM usuarios u 
    WHERE u.rol = 'vendedor' AND u.activo = 1 
    ORDER BY u.nombre
");

// Obtener clientes para filtro
$clientes = $db->select("
    SELECT c.id, c.nombre, u.nombre as vendedor_nombre
    FROM clientes c 
    INNER JOIN usuarios u ON c.vendedor_id = u.id
    WHERE c.activo = 1 
    ORDER BY c.nombre
");

include '../../includes/header.php';
?>

<div class="row">
    <div class="col-12">
        <div class="d-flex justify-content-between align-items-center mb-4">
            <h1 class="h3">
                <i class="fas fa-shopping-cart me-2"></i>Gestión de Ventas
            </h1>
            <div>
                <button type="button" class="btn btn-success" onclick="exportarVentas()">
                    <i class="fas fa-file-excel me-2"></i>Exportar Excel
                </button>
                <button type="button" class="btn btn-danger" onclick="exportarPDF()">
                    <i class="fas fa-file-pdf me-2"></i>Exportar PDF
                </button>
            </div>
        </div>
    </div>
</div>

<!-- Filtros -->
<div class="card mb-4">
    <div class="card-header">
        <h5 class="mb-0">
            <i class="fas fa-filter me-2"></i>Filtros de Búsqueda
        </h5>
    </div>
    <div class="card-body">
        <form method="GET" class="row g-3">
            <div class="col-md-3">
                <label for="fecha_inicio" class="form-label">Fecha Inicio</label>
                <input type="date" name="fecha_inicio" id="fecha_inicio" class="form-control" 
                       value="<?= htmlspecialchars($fechaInicio) ?>">
            </div>
            
            <div class="col-md-3">
                <label for="fecha_fin" class="form-label">Fecha Fin</label>
                <input type="date" name="fecha_fin" id="fecha_fin" class="form-control" 
                       value="<?= htmlspecialchars($fechaFin) ?>">
            </div>
            
            <div class="col-md-3">
                <label for="vendedor_id" class="form-label">Vendedor</label>
                <select name="vendedor_id" id="vendedor_id" class="form-select">
                    <option value="">Todos los vendedores</option>
                    <?php foreach ($vendedores as $vendedor): ?>
                        <option value="<?= $vendedor['id'] ?>" 
                                <?= ($vendedorId == $vendedor['id']) ? 'selected' : '' ?>>
                            <?= htmlspecialchars($vendedor['nombre']) ?>
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>
            
            <div class="col-md-3">
                <label for="cliente_id" class="form-label">Cliente</label>
                <select name="cliente_id" id="cliente_id" class="form-select">
                    <option value="">Todos los clientes</option>
                    <?php foreach ($clientes as $cliente): ?>
                        <option value="<?= $cliente['id'] ?>" 
                                <?= ($clienteId == $cliente['id']) ? 'selected' : '' ?>>
                            <?= htmlspecialchars($cliente['nombre']) ?> (<?= htmlspecialchars($cliente['vendedor_nombre']) ?>)
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>
            
            <div class="col-12">
                <button type="submit" class="btn btn-primary">
                    <i class="fas fa-search me-2"></i>Filtrar
                </button>
                <a href="ventas.php" class="btn btn-secondary">
                    <i class="fas fa-times me-2"></i>Limpiar
                </a>
            </div>
        </form>
    </div>
</div>

<!-- Estadísticas del período -->
<div class="row mb-4">
    <div class="col-md-2">
        <div class="card bg-primary text-white">
            <div class="card-body text-center">
                <i class="fas fa-shopping-cart fa-2x mb-2"></i>
                <h4><?= safeNumber($estadisticas['total_ventas']) ?></h4>
                <small>Total Ventas</small>
            </div>
        </div>
    </div>
    
    <div class="col-md-2">
        <div class="card bg-success text-white">
            <div class="card-body text-center">
                <i class="fas fa-tint fa-2x mb-2"></i>
                <h4><?= safeNumber($estadisticas['total_bidones']) ?></h4>
                <small>Bidones Vendidos</small>
            </div>
        </div>
    </div>
    
    <div class="col-md-2">
        <div class="card bg-info text-white">
            <div class="card-body text-center">
                <i class="fas fa-dollar-sign fa-2x mb-2"></i>
                <h4><?= safeCurrency($estadisticas['total_ingresos']) ?></h4>
                <small>Total Ingresos</small>
            </div>
        </div>
    </div>
    
    <div class="col-md-2">
        <div class="card bg-warning text-white">
            <div class="card-body text-center">
                <i class="fas fa-chart-line fa-2x mb-2"></i>
                <h4><?= safeCurrency($estadisticas['promedio_venta']) ?></h4>
                <small>Promedio Venta</small>
            </div>
        </div>
    </div>
    
    <div class="col-md-2">
        <div class="card bg-secondary text-white">
            <div class="card-body text-center">
                <i class="fas fa-users fa-2x mb-2"></i>
                <h4><?= safeNumber($estadisticas['clientes_unicos']) ?></h4>
                <small>Clientes Únicos</small>
            </div>
        </div>
    </div>
    
    <div class="col-md-2">
        <div class="card bg-dark text-white">
            <div class="card-body text-center">
                <i class="fas fa-user-tie fa-2x mb-2"></i>
                <h4><?= safeNumber($estadisticas['vendedores_activos']) ?></h4>
                <small>Vendedores Activos</small>
            </div>
        </div>
    </div>
</div>

<!-- Lista de ventas -->
<div class="card">
    <div class="card-header">
        <h5 class="mb-0">
            Lista de Ventas 
            <span class="badge bg-primary"><?= count($ventas) ?> registros</span>
        </h5>
    </div>
    <div class="card-body">
        <div class="table-responsive">
            <table class="table table-striped table-hover" id="ventasTable">
                <thead class="table-dark">
                    <tr>
                        <th>Fecha</th>
                        <th>Cliente</th>
                        <th>Vendedor</th>
                        <th>Zona</th>
                        <th>Bidones</th>
                        <th>Precio Unit.</th>
                        <th>Total</th>
                        <th>Observaciones</th>
                        <th>Acciones</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($ventas as $venta): ?>
                        <tr>
                            <td>
                                <div class="fw-bold"><?= formatDate($venta['fecha']) ?></div>
                                <small class="text-muted">ID: <?= $venta['id'] ?></small>
                            </td>
                            <td>
                                <div class="fw-bold"><?= htmlspecialchars($venta['cliente_nombre']) ?></div>
                                <small class="text-muted">
                                    <?= htmlspecialchars($venta['cliente_direccion']) ?>
                                </small>
                                <?php if ($venta['cliente_telefono']): ?>
                                    <div><small><i class="fas fa-phone me-1"></i><?= htmlspecialchars($venta['cliente_telefono']) ?></small></div>
                                <?php endif; ?>
                            </td>
                            <td>
                                <span class="badge bg-info"><?= htmlspecialchars($venta['vendedor_nombre']) ?></span>
                            </td>
                            <td>
                                <?= htmlspecialchars($venta['zona_nombre'] ?? 'Sin zona') ?>
                            </td>
                            <td>
                                <span class="badge bg-success fs-6"><?= $venta['cantidad_bidones'] ?></span>
                            </td>
                            <td>
                                <?= safeCurrency($venta['precio_unitario']) ?>
                            </td>
                            <td>
                                <div class="fw-bold text-primary"><?= safeCurrency($venta['total']) ?></div>
                            </td>
                            <td>
                                <?php if ($venta['observaciones']): ?>
                                    <span class="text-muted" title="<?= htmlspecialchars($venta['observaciones']) ?>">
                                        <?= safeText($venta['observaciones'], 30) ?>
                                    </span>
                                <?php else: ?>
                                    <span class="text-muted">-</span>
                                <?php endif; ?>
                            </td>
                            <td>
                                <div class="btn-group btn-group-sm">
                                    <button type="button" class="btn btn-outline-info" 
                                            onclick="verDetalleVenta(<?= $venta['id'] ?>)" 
                                            title="Ver detalles">
                                        <i class="fas fa-eye"></i>
                                    </button>
                                    
                                    <button type="button" class="btn btn-outline-primary" 
                                            onclick="editarVenta(<?= $venta['id'] ?>)" 
                                            title="Editar">
                                        <i class="fas fa-edit"></i>
                                    </button>
                                    
                                    <button type="button" class="btn btn-outline-danger" 
                                            onclick="eliminarVenta(<?= $venta['id'] ?>, '<?= htmlspecialchars($venta['cliente_nombre']) ?>')" 
                                            title="Eliminar">
                                        <i class="fas fa-trash"></i>
                                    </button>
                                </div>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
    </div>
</div>

<!-- Modal para eliminar venta -->
<div class="modal fade" id="eliminarVentaModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Confirmar Eliminación</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <p>¿Está seguro de que desea eliminar esta venta?</p>
                <div id="ventaInfo"></div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancelar</button>
                <form method="POST" style="display: inline;">
                    <input type="hidden" name="action" value="delete_venta">
                    <input type="hidden" name="venta_id" id="ventaIdToDelete">
                    <button type="submit" class="btn btn-danger">Eliminar</button>
                </form>
            </div>
        </div>
    </div>
</div>

<?php
$extraJS = '
<script>
// DataTable
$(document).ready(function() {
    $("#ventasTable").DataTable({
        "language": {
            "url": "//cdn.datatables.net/plug-ins/1.10.24/i18n/Spanish.json"
        },
        "pageLength": 25,
        "order": [[0, "desc"]], // Ordenar por fecha descendente
        "columnDefs": [
            { "orderable": false, "targets": 8 } // Columna de acciones no ordenable
        ]
    });
});

// Funciones para acciones
function verDetalleVenta(ventaId) {
    // Implementar vista de detalles
    alert("Función de detalles en desarrollo. Venta ID: " + ventaId);
}

function editarVenta(ventaId) {
    // Implementar edición
    alert("Función de edición en desarrollo. Venta ID: " + ventaId);
}

function eliminarVenta(ventaId, clienteNombre) {
    document.getElementById("ventaIdToDelete").value = ventaId;
    document.getElementById("ventaInfo").innerHTML = 
        "<strong>Cliente:</strong> " + clienteNombre + "<br>" +
        "<strong>ID Venta:</strong> " + ventaId;
    
    var modal = new bootstrap.Modal(document.getElementById("eliminarVentaModal"));
    modal.show();
}

function exportarVentas() {
    const params = new URLSearchParams(window.location.search);
    params.set("export", "excel");
    window.open("export.php?" + params.toString(), "_blank");
}

function exportarPDF() {
    const params = new URLSearchParams(window.location.search);
    params.set("export", "pdf");
    window.open("export.php?" + params.toString(), "_blank");
}

// Mejorar el selector de clientes con búsqueda
$(document).ready(function() {
    if ($("#cliente_id").length) {
        $("#cliente_id").select2({
            placeholder: "Buscar cliente...",
            allowClear: true,
            width: "100%"
        });
    }
});
</script>

<link rel="stylesheet" type="text/css" href="https://cdn.datatables.net/1.10.24/css/dataTables.bootstrap5.min.css">
<script type="text/javascript" charset="utf8" src="https://cdn.datatables.net/1.10.24/js/jquery.dataTables.js"></script>
<script type="text/javascript" charset="utf8" src="https://cdn.datatables.net/1.10.24/js/dataTables.bootstrap5.min.js"></script>

<link href="https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/css/select2.min.css" rel="stylesheet" />
<link href="https://cdn.jsdelivr.net/npm/select2-bootstrap-5-theme@1.3.0/dist/select2-bootstrap-5-theme.min.css" rel="stylesheet" />
<script src="https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/js/select2.min.js"></script>
';

include '../../includes/footer.php';
?>
