<?php
/**
 * Editar Cliente - Administrador
 * Sistema de Gestión de Agua Mineral
 */

require_once '../../includes/init.php';

// Verificar que sea administrador
checkPermission('administrador');

$pageTitle = 'Editar Cliente - Administrador';

$db = Database::getInstance();

$clienteId = (int)($_GET['id'] ?? 0);

// 1. Obtener datos del cliente
$cliente = $db->select("SELECT * FROM clientes WHERE id = ?", [$clienteId]);

if (empty($cliente)) {
    setFlashMessage('Cliente no encontrado.', 'error');
    redirect('clientes.php');
}

$cliente = $cliente[0];

// Obtener vendedores activos
$vendedores = $db->select("SELECT id, nombre FROM usuarios WHERE rol = 'vendedor' AND activo = 1 ORDER BY nombre");

// Procesar formulario
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $nombre = sanitize($_POST['nombre']);
    $direccion = sanitize($_POST['direccion']);
    $telefono = sanitize($_POST['telefono']);
    $email = sanitize($_POST['email']);
    $vendedorId = (int)$_POST['vendedor_id'];
    $latitud = !empty($_POST['latitud']) ? (float)$_POST['latitud'] : null;
    $longitud = !empty($_POST['longitud']) ? (float)$_POST['longitud'] : null;
    $observaciones = sanitize($_POST['observaciones']);
    
    // Validaciones
    $errors = [];
    
    if (empty($nombre)) {
        $errors[] = 'El nombre es obligatorio.';
    }
    
    if (empty($direccion)) {
        $errors[] = 'La dirección es obligatoria.';
    }
    
    if (empty($vendedorId)) {
        $errors[] = 'Debe seleccionar un vendedor.';
    }
    
    if (!empty($email) && !validateEmail($email)) {
        $errors[] = 'El email no es válido.';
    }
    
    if ($latitud && $longitud && !validarCoordenadas($latitud, $longitud)) {
        $errors[] = 'Las coordenadas GPS no son válidas para Paraguay.';
    }
    
    // Verificar que el vendedor existe
    if ($vendedorId) {
        $vendedorExists = $db->select("SELECT id FROM usuarios WHERE id = ? AND rol = 'vendedor' AND activo = 1", [$vendedorId]);
        if (empty($vendedorExists)) {
            $errors[] = 'El vendedor seleccionado no es válido.';
        }
    }
    
    if (empty($errors)) {
        try {
            $query = "UPDATE clientes SET 
                        nombre = ?, 
                        direccion = ?, 
                        telefono = ?, 
                        email = ?, 
                        vendedor_id = ?, 
                        latitud = ?, 
                        longitud = ?, 
                        observaciones = ?,
                        updated_at = NOW()
                      WHERE id = ?";
            
            $params = [$nombre, $direccion, $telefono, $email, $vendedorId, $latitud, $longitud, $observaciones, $clienteId];
            
            if ($db->execute($query, $params)) {
                // Log de actividad
                logActivity($_SESSION['user_id'], 'update_client', "Cliente actualizado: $nombre (ID: $clienteId)");
                
                setFlashMessage('Cliente actualizado exitosamente.', 'success');
                redirect('clientes.php');
            } else {
                $errors[] = 'Error al actualizar el cliente. Intente nuevamente.';
            }
        } catch (Exception $e) {
            $errors[] = 'Error en la base de datos: ' . $e->getMessage();
        }
    }
    
    // Si hay errores, recargar los datos del POST para que no se pierdan
    $cliente = array_merge($cliente, $_POST);
}

include '../../includes/header.php';
?>

<div class="row">
    <div class="col-12">
        <div class="d-flex justify-content-between align-items-center mb-4">
            <h1 class="h3">
                <i class="fas fa-user-edit me-2"></i>Editar Cliente: <?= htmlspecialchars($cliente['nombre']) ?>
            </h1>
            <a href="clientes.php" class="btn btn-secondary">
                <i class="fas fa-arrow-left me-2"></i>Volver a Clientes
            </a>
        </div>
    </div>
</div>

<?php if (!empty($errors)): ?>
    <div class="alert alert-danger">
        <ul class="mb-0">
            <?php foreach ($errors as $error): ?>
                <li><?= htmlspecialchars($error) ?></li>
            <?php endforeach; ?>
        </ul>
    </div>
<?php endif; ?>

<div class="row">
    <div class="col-lg-8">
        <div class="card">
            <div class="card-header">
                <h5 class="mb-0">Información del Cliente</h5>
            </div>
            <div class="card-body">
                <form method="POST" class="needs-validation" novalidate>
                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="nombre" class="form-label">Nombre Completo *</label>
                                <input type="text" name="nombre" id="nombre" class="form-control" 
                                       value="<?= htmlspecialchars($cliente['nombre'] ?? '') ?>" required>
                                <div class="invalid-feedback">
                                    Por favor ingrese el nombre del cliente.
                                </div>
                            </div>
                        </div>
                        
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="telefono" class="form-label">Teléfono</label>
                                <input type="tel" name="telefono" id="telefono" class="form-control" 
                                       value="<?= htmlspecialchars($cliente['telefono'] ?? '') ?>" 
                                       placeholder="0981-123-456">
                            </div>
                        </div>
                    </div>
                    
                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="email" class="form-label">Email</label>
                                <input type="email" name="email" id="email" class="form-control" 
                                       value="<?= htmlspecialchars($cliente['email'] ?? '') ?>">
                                <div class="invalid-feedback">
                                    Por favor ingrese un email válido.
                                </div>
                            </div>
                        </div>
                        
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="vendedor_id" class="form-label">Vendedor Asignado *</label>
                                <select name="vendedor_id" id="vendedor_id" class="form-select" required>
                                    <option value="">Seleccionar vendedor</option>
                                    <?php foreach ($vendedores as $vendedor): ?>
                                        <option value="<?= $vendedor['id'] ?>" 
                                                <?= (($cliente['vendedor_id'] ?? '') == $vendedor['id']) ? 'selected' : '' ?>>
                                            <?= htmlspecialchars($vendedor['nombre']) ?>
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                                <div class="invalid-feedback">
                                    Por favor seleccione un vendedor.
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <div class="mb-3">
                        <label for="direccion" class="form-label">Dirección Completa *</label>
                        <textarea name="direccion" id="direccion" class="form-control" rows="2" required><?= htmlspecialchars($cliente['direccion'] ?? '') ?></textarea>
                        <div class="invalid-feedback">
                            Por favor ingrese la dirección del cliente.
                        </div>
                    </div>
                    
                    <div class="mb-3">
                        <label for="observaciones" class="form-label">Observaciones</label>
                        <textarea name="observaciones" id="observaciones" class="form-control" rows="3"><?= htmlspecialchars($cliente['observaciones'] ?? '') ?></textarea>
                        <div class="form-text">Información adicional sobre el cliente (opcional)</div>
                    </div>
                    
                    <hr>
                    
                    <h6 class="mb-3">
                        <i class="fas fa-map-marker-alt me-2"></i>Ubicación GPS (Opcional)
                    </h6>
                    
                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="latitud" class="form-label">Latitud</label>
                                <input type="number" name="latitud" id="latitud" class="form-control" 
                                       step="0.000001" min="-27.6" max="-19.3" 
                                       value="<?= htmlspecialchars($cliente['latitud'] ?? '') ?>" 
                                       placeholder="-25.263740">
                                <div class="form-text">Coordenada de latitud para Paraguay</div>
                            </div>
                        </div>
                        
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="longitud" class="form-label">Longitud</label>
                                <input type="number" name="longitud" id="longitud" class="form-control" 
                                       step="0.000001" min="-62.6" max="-54.3" 
                                       value="<?= htmlspecialchars($cliente['longitud'] ?? '') ?>" 
                                       placeholder="-57.575926">
                                <div class="form-text">Coordenada de longitud para Paraguay</div>
                            </div>
                        </div>
                    </div>
                    
                    <div class="mb-3">
                        <button type="button" class="btn btn-info btn-sm" onclick="buscarUbicacion()">
                            <i class="fas fa-search me-1"></i>Buscar Ubicación por Dirección
                        </button>
                        <button type="button" class="btn btn-success btn-sm" onclick="obtenerUbicacionActual()">
                            <i class="fas fa-crosshairs me-1"></i>Usar Mi Ubicación
                        </button>
                        <a href="cliente-ubicacion.php?id=<?= $clienteId ?>" class="btn btn-secondary btn-sm">
                            <i class="fas fa-map-marked-alt me-1"></i>Ver en Mapa
                        </a>
                    </div>
                    
                    <div class="d-grid gap-2 d-md-flex justify-content-md-end">
                        <a href="clientes.php" class="btn btn-secondary me-md-2">
                            <i class="fas fa-times me-1"></i>Cancelar
                        </a>
                        <button type="submit" class="btn btn-primary">
                            <i class="fas fa-save me-1"></i>Guardar Cambios
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>
    
    <div class="col-lg-4">
        <div class="card">
            <div class="card-header">
                <h6 class="mb-0">Información Adicional</h6>
            </div>
            <div class="card-body">
                <p><strong>ID Cliente:</strong> <?= $cliente['id'] ?></p>
                <p><strong>Registrado:</strong> <?= formatDate($cliente['fecha_registro']) ?></p>
                <p><strong>Última Actualización:</strong> <?= formatDate($cliente['updated_at'], 'd/m/Y H:i') ?></p>
                
                <hr>
                
                <h6>Resumen Financiero</h6>
                <?php
                // Obtener saldo del cliente
                $saldo = calculateClientBalance($clienteId);
                $estado = getClienteEstado($cliente['updated_at'], $saldo);
                ?>
                <p><strong>Saldo Actual:</strong> 
                    <span class="fw-bold text-<?= $saldo > 0 ? 'danger' : ($saldo < 0 ? 'success' : 'primary') ?>">
                        <?= safeCurrency(abs($saldo)) ?> <?= $saldo > 0 ? '(Debe)' : ($saldo < 0 ? '(A Favor)' : '(Al Día)') ?>
                    </span>
                </p>
                <p><strong>Estado:</strong> 
                    <span class="badge bg-<?= $estado['clase'] ?>"><?= $estado['texto'] ?></span>
                </p>
                
                <hr>
                
                <a href="cliente-ubicacion.php?id=<?= $clienteId ?>" class="btn btn-sm btn-outline-primary w-100 mb-2">
                    <i class="fas fa-map-marker-alt me-1"></i>Editar Ubicación GPS
                </a>
                <a href="ventas.php?cliente_id=<?= $clienteId ?>" class="btn btn-sm btn-outline-info w-100">
                    <i class="fas fa-history me-1"></i>Ver Historial de Ventas
                </a>
            </div>
        </div>
    </div>
</div>

<?php
$extraJS = '
<script>
// Validación del formulario
(function() {
    "use strict";
    window.addEventListener("load", function() {
        var forms = document.getElementsByClassName("needs-validation");
        var validation = Array.prototype.filter.call(forms, function(form) {
            form.addEventListener("submit", function(event) {
                if (form.checkValidity() === false) {
                    event.preventDefault();
                    event.stopPropagation();
                }
                form.classList.add("was-validated");
            }, false);
        });
    }, false);
})();

// Buscar ubicación por dirección
function buscarUbicacion() {
    const direccion = document.getElementById("direccion").value;
    
    if (!direccion.trim()) {
        alert("Por favor, ingrese una dirección para buscar.");
        return;
    }
    
    // Usar servicio de geocodificación (requiere API key de Google Maps)
    if (typeof google !== "undefined" && google.maps) {
        const geocoder = new google.maps.Geocoder();
        
        geocoder.geocode({ address: direccion + ", Paraguay" }, (results, status) => {
            if (status === "OK" && results[0]) {
                const location = results[0].geometry.location;
                
                document.getElementById("latitud").value = location.lat().toFixed(6);
                document.getElementById("longitud").value = location.lng().toFixed(6);
                
                alert("Ubicación encontrada y coordenadas actualizadas.");
            } else {
                alert("No se pudo encontrar la ubicación. Intente con una dirección más específica.");
            }
        });
    } else {
        alert("Servicio de geocodificación no disponible. Ingrese las coordenadas manualmente.");
    }
}

// Obtener ubicación actual
function obtenerUbicacionActual() {
    if (navigator.geolocation) {
        navigator.geolocation.getCurrentPosition(
            function(position) {
                const lat = position.coords.latitude;
                const lng = position.coords.longitude;
                
                // Validar que esté en Paraguay
                if (lat >= -27.6 && lat <= -19.3 && lng >= -62.6 && lng <= -54.3) {
                    document.getElementById("latitud").value = lat.toFixed(6);
                    document.getElementById("longitud").value = lng.toFixed(6);
                    alert("Ubicación actual obtenida correctamente.");
                } else {
                    alert("Su ubicación actual no está en Paraguay.");
                }
            },
            function(error) {
                alert("No se pudo obtener su ubicación: " + error.message);
            }
        );
    } else {
        alert("Su navegador no soporta geolocalización.");
    }
}

// Validación de coordenadas en tiempo real
document.getElementById("latitud").addEventListener("blur", function() {
    const lat = parseFloat(this.value);
    if (this.value && (lat < -27.6 || lat > -19.3)) {
        this.setCustomValidity("La latitud debe estar entre -27.6 y -19.3 para Paraguay");
    } else {
        this.setCustomValidity("");
    }
});

document.getElementById("longitud").addEventListener("blur", function() {
    const lng = parseFloat(this.value);
    if (this.value && (lng < -62.6 || lng > -54.3)) {
        this.setCustomValidity("La longitud debe estar entre -62.6 y -54.3 para Paraguay");
    } else {
        this.setCustomValidity("");
    }
});
</script>
';

include '../../includes/footer.php';
?>
